package com.finconsgroup.itserr.marketplace.metadata.integration.service;

import com.finconsgroup.itserr.marketplace.metadata.lib.ConstraintViolation;
import org.springframework.lang.NonNull;

import java.util.List;
import java.util.Map;
import java.util.UUID;

/**
 * <p>Service to be used to validate values to be set to a metadata.</p>
 * <p>This service integrates with metadata-dm microservice to retrieve metadata definition, maps the obtained metadata into metadata-lib library model and
 * then performs the validation.
 * </p>
 */
public interface MetadataValidationService {

    /**
     * <p>Validates values to be set to a metadata.</p>
     * <p>This method integrates with metadata-dm microservice, and does the following:
     * <ul>
     * <li>retrieves the metadata definition from metadata-dm microservice</li>
     * <li>maps the received metadata into metadata-lib domain objects</li>
     * <li>validates the argument values over the built model</li>
     * <li>returns the found violations</li>
     * </ul>
     * </p>
     *
     * @param metadataId Id of the metadata-dm microservice metadata. Cannot be null.
     * @param values Metadata fields values map. Key is the field name. Cannot be null.
     * @return Found violations, or empty list if none found. Never null.
     * @throws com.finconsgroup.itserr.marketplace.metadata.integration.exception.WP2MetadataNotFoundException If metadata not found.
     * @throws com.finconsgroup.itserr.marketplace.metadata.integration.exception.WP2MetadataRetrieveException If metadata could not be retrieved.
     */
    @NonNull
    List<ConstraintViolation> validate(
            @NonNull UUID metadataId,
            @NonNull Map<String, Object> values);

    /**
     * Same as {@link #validate(UUID, Map)}, but throws an exception if any violation is found.
     *
     * @param metadataId Id of the metadata-dm microservice metadata. Cannot be null.
     * @param values Metadata fields values map. Key is the field name. Cannot be null.
     * @throws com.finconsgroup.itserr.marketplace.metadata.integration.exception.WP2MetadataNotFoundException If metadata not found.
     * @throws com.finconsgroup.itserr.marketplace.metadata.integration.exception.WP2MetadataRetrieveException If metadata could not be retrieved.
     * @throws com.finconsgroup.itserr.marketplace.metadata.integration.exception.WP2InvalidMetadataException If any violation is found.
     * @see #validate(UUID, Map)
     */
    void validateOrThrow(
            @NonNull UUID metadataId,
            @NonNull Map<String, Object> values);

}
