package com.finconsgroup.itserr.marketplace.discussion.bs.component;

import com.finconsgroup.itserr.marketplace.core.web.security.jwt.JwtTokenHolder;
import com.finconsgroup.itserr.marketplace.discussion.bs.bean.DiscussionApplicationEvent;
import com.finconsgroup.itserr.marketplace.discussion.bs.bean.DiscussionThreadDetails;
import com.finconsgroup.itserr.marketplace.discussion.bs.mapper.EventToDiscussionMapper;
import com.finconsgroup.itserr.marketplace.discussion.bs.messaging.dto.DiscussionThreadMessagingAdditionalDataDto;
import com.finconsgroup.itserr.marketplace.discussion.bs.messaging.producer.DiscussionEventProducer;
import com.finconsgroup.itserr.messaging.dto.MessagingEventDto;
import com.finconsgroup.itserr.messaging.dto.MessagingEventUserDto;
import lombok.NonNull;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.jetbrains.annotations.NotNull;
import org.springframework.context.ApplicationListener;
import org.springframework.stereotype.Component;

/**
 * The implementation for {@link ApplicationListener} to listen to Discussion application events
 * and produce relevant notification messages.
 */
@Component
@Slf4j
@RequiredArgsConstructor
public class MessagingDiscussionApplicationEventListener implements ApplicationListener<DiscussionApplicationEvent> {

    private final DiscussionEventProducer discussionEventProducer;
    private final EventToDiscussionMapper eventToDiscussionMapper;

    @Override
    public void onApplicationEvent(@NonNull DiscussionApplicationEvent event) {
        MessagingEventDto<DiscussionThreadMessagingAdditionalDataDto> messagingEventDto =
                eventToDiscussionMapper.toMessagingEventDto(event.getDiscussionThreadDetails());

        // Ensure user information is populated
        MessagingEventUserDto user = MessagingEventUserDto.builder()
                .id(JwtTokenHolder.getUserId().orElse(null))
                .name(JwtTokenHolder.getName().orElse(null))
                .username(JwtTokenHolder.getPreferredUsername().orElse(null))
                .build();
        messagingEventDto.setUser(user);

        DiscussionThreadMessagingAdditionalDataDto additionalData = getDiscussionThreadMessagingAdditionalDataDto(event);
        messagingEventDto.setAdditionalData(additionalData);

        // Publish the event based on type
        publishEventForType(event, messagingEventDto);
    }

    @NotNull
    private static DiscussionThreadMessagingAdditionalDataDto getDiscussionThreadMessagingAdditionalDataDto(@NotNull DiscussionApplicationEvent event) {
        DiscussionThreadDetails discussionThreadDetails = event.getDiscussionThreadDetails();

        DiscussionThreadMessagingAdditionalDataDto additionalData = new DiscussionThreadMessagingAdditionalDataDto();
        additionalData.setDiscussionResourceId(discussionThreadDetails.getDiscussionResourceId());
        additionalData.setDiscussionResourceType(discussionThreadDetails.getDiscussionResourceType());
        additionalData.setNotifyUserIds(discussionThreadDetails.getUsersToNotify());
        additionalData.setDiscussionId(discussionThreadDetails.getDiscussionId());
        additionalData.setDiscussionTitle(discussionThreadDetails.getDiscussionTitle());
        return additionalData;
    }

    private void publishEventForType(DiscussionApplicationEvent applicationEvent,
                                     MessagingEventDto<DiscussionThreadMessagingAdditionalDataDto> messagingEventDto) {
        switch (applicationEvent.getEventType()) {
            case CREATED -> {
                discussionEventProducer.publishCreateEvent(messagingEventDto);
            }
        }
    }
}