package com.finconsgroup.itserr.marketplace.discussion.dm.repository;

import com.finconsgroup.itserr.marketplace.discussion.dm.dto.ThreadCountByReactionDto;
import com.finconsgroup.itserr.marketplace.discussion.dm.entity.Thread;
import com.finconsgroup.itserr.marketplace.discussion.dm.entity.ThreadReaction;
import com.finconsgroup.itserr.marketplace.discussion.dm.enums.ReactionType;
import org.springframework.data.jpa.repository.JpaRepository;
import org.springframework.data.jpa.repository.Query;
import org.springframework.data.repository.query.Param;
import org.springframework.stereotype.Repository;

import java.util.List;
import java.util.Optional;
import java.util.UUID;

/**
 * ThreadReactionRepository is a Spring Data JPA repository interface for performing CRUD operations
 * and interacting with the {@code ThreadReaction} entity in the database. It provides methods
 * for accessing, querying, and persisting thread reaction data.
 *
 * This interface extends {@code JpaRepository}, inheriting a standard set of data access
 * methods without requiring explicit implementation. The repository is annotated with
 * {@code @Repository} to indicate its role within the Spring framework’s persistence layer.
 *
 * Key Features:
 * - Supports standard JPA repository operations such as save, findById, findAll, and delete.
 * - Operates on the table associated with the {@code ThreadReaction} entity.
 * - Utilizes the {@code UUID} type as the primary key for uniquely identifying each thread reaction.
 */
@Repository
public interface ThreadReactionRepository extends JpaRepository<ThreadReaction, UUID> {

    Optional<ThreadReaction> findByThreadAndUserId(Thread thread, String userId);

    void deleteByThreadAndUserId(Thread thread, String userId);

    @Query("SELECT new com.finconsgroup.itserr.marketplace.discussion.dm.dto.ThreadCountByReactionDto(tr.reaction, COUNT(tr)) FROM ThreadReaction tr WHERE tr.thread = :thread GROUP BY tr.reaction")
    List<ThreadCountByReactionDto> countByThreadGroupByReactionType(@Param("thread") Thread thread);

}
