package com.finconsgroup.itserr.marketplace.event.bs.api;

import com.finconsgroup.itserr.marketplace.event.bs.enums.MessagingEventType;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.responses.ApiResponse;
import io.swagger.v3.oas.annotations.security.SecurityRequirement;
import io.swagger.v3.oas.annotations.tags.Tag;
import org.springframework.core.io.Resource;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.RequestParam;

/**
 * This interface defines the contract for REST API endpoints used for system testing and debugging.
 *
 * <p>
 * These endpoints are intended for internal use only and
 * provide endpoints used for debugging and runtime analysis.
 * </p>
 *
 * <p>Example usage:
 * <pre>
 * GET /api/v1/bs/diagnostics/download-logs - Download application log for analysis
 * </pre>
 * </p>
 */
@Tag(
        name = "Diagnostics",
        description = "Diagnostics API: provides endpoints used for " +
                "debugging and runtime analysis."
)
@SecurityRequirement(name = "BearerAuth")
public interface DiagnosticsApi {

    @Operation(
            summary = "Download log file",
            responses = {@ApiResponse(responseCode = "200", description = "OK"),}
    )
    @GetMapping(path = "/event/diagnostics/download-logs", produces = MediaType.APPLICATION_OCTET_STREAM_VALUE)
    ResponseEntity<Resource> downloadLogs();

    /**
     * Test API method for sending a message for an event .
     * Used to simulate or trigger the messaging logic for item creation, update, or deletion.
     *
     * @param messagingEventType the type of event (CREATED, UPDATED, DELETED, REGISTER or UNREGISTER)
     * @param name      the name of the event (defaults to "item name" if not provided)
     * @return ResponseEntity with operation result
     * @throws IllegalArgumentException if the event type is invalid
     */
    @Operation(
            summary = "Test method to send event event message",
            description = "Simulates publishing events for event changes (creation, update or deletion)",
            responses = {
                    @ApiResponse(responseCode = "200", description = "Message sent successfully"),
                    @ApiResponse(responseCode = "400", description = "Invalid request parameters"),
                    @ApiResponse(responseCode = "404", description = "Item not found (for update/delete operations)")
            }
    )
    @GetMapping(path = "/event/diagnostics/publish-message", produces = MediaType.APPLICATION_JSON_VALUE)
    ResponseEntity<String> publishMessage(
            @RequestParam(name = "eventType", required = false) MessagingEventType messagingEventType,
            @RequestParam(name = "name", required = false, defaultValue = "item name") String name
    );
}
