package com.finconsgroup.itserr.marketplace.event.dm.api;

import com.finconsgroup.itserr.marketplace.core.web.exception.ErrorResponseDto;
import com.finconsgroup.itserr.marketplace.event.dm.dto.InputProgramSubscribedParticipantDto;
import com.finconsgroup.itserr.marketplace.event.dm.dto.OutputEventDto;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.media.Content;
import io.swagger.v3.oas.annotations.media.Schema;
import io.swagger.v3.oas.annotations.responses.ApiResponse;
import io.swagger.v3.oas.annotations.security.SecurityRequirement;
import io.swagger.v3.oas.annotations.tags.Tag;
import jakarta.validation.Valid;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.web.bind.annotation.DeleteMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.ResponseStatus;

import java.util.UUID;

/**
 * This interface defines the contract for REST API endpoints related to Event Management.
 *
 * <p>
 * It provides endpoints to register and unregister to a program.
 * </p>
 *
 * <p>Example usage:
 * <pre>
 * POST /event/events/{eventId}/programs/{programId}/register - register to a program.
 * DELETE /event/events/{eventId}/programs/{programId}/register - un-register to a program.
 * </pre>
 * </p>
 */
@Tag(
        name = "ProgramApi",
        description = "The ProgramApi API: it provides endpoints to " +
                "register and unregister to a program."
)
@SecurityRequirement(name = "BearerAuth")
public interface ProgramApi {

    /**
     * Register a user to a program.
     */
    @Operation(
            summary = "register a user to a program",
            responses = {
                    @ApiResponse(responseCode = "200", description = "Success"),
                    @ApiResponse(responseCode = "400", description = "Bad Request", content = {
                            @Content(
                                    mediaType = MediaType.APPLICATION_JSON_VALUE,
                                    schema = @Schema(implementation = ErrorResponseDto.class)
                            )
                    }),
                    @ApiResponse(responseCode = "404", description = "Not Found", content = {
                            @Content(
                                    mediaType = MediaType.APPLICATION_JSON_VALUE,
                                    schema = @Schema(implementation = ErrorResponseDto.class)
                            )
                    }),
                    @ApiResponse(responseCode = "409", description = "Conflict", content = {
                            @Content(
                                    mediaType = MediaType.APPLICATION_JSON_VALUE,
                                    schema = @Schema(implementation = ErrorResponseDto.class)
                            )
                    })
            }
    )
    @PostMapping(
            value = "/event/events/{eventId}/programs/{programId}/register",
            produces = MediaType.APPLICATION_JSON_VALUE
    )
    @ResponseStatus(HttpStatus.OK)
    OutputEventDto registerProgram(@PathVariable("eventId") UUID eventId,
                                   @PathVariable("programId") UUID programId,
                                   @Valid @RequestBody InputProgramSubscribedParticipantDto inputProgramSubscribedParticipantDto);

    /**
     * Unregister a user to a program.
     */
    @Operation(
            summary = "unregister a user to a program",
            responses = {
                    @ApiResponse(responseCode = "200", description = "Success"),
                    @ApiResponse(responseCode = "404", description = "Not Found", content = {
                            @Content(
                                    mediaType = MediaType.APPLICATION_JSON_VALUE,
                                    schema = @Schema(implementation = ErrorResponseDto.class)
                            )
                    }),
            }
    )
    @DeleteMapping(
            value = "/event/events/{eventId}/programs/{programId}/register",
            produces = MediaType.APPLICATION_JSON_VALUE
    )
    @ResponseStatus(HttpStatus.OK)
    OutputEventDto unregisterProgram(@PathVariable("eventId") UUID eventId, @PathVariable("programId") UUID programId);
}
