package com.finconsgroup.itserr.marketplace.favouritesearch.dm.entity;

import com.finconsgroup.itserr.marketplace.core.entity.AbstractUUIDEntity;
import com.finconsgroup.itserr.marketplace.favouritesearch.dm.converter.SearchContextAttributeConverter;
import com.finconsgroup.itserr.marketplace.favouritesearch.dm.enums.SearchContext;
import jakarta.persistence.Column;
import jakarta.persistence.Convert;
import jakarta.persistence.Entity;
import jakarta.persistence.PrePersist;
import jakarta.persistence.Table;
import jakarta.persistence.Version;
import lombok.Data;
import lombok.EqualsAndHashCode;
import lombok.NoArgsConstructor;
import lombok.experimental.SuperBuilder;

import java.time.ZoneOffset;
import java.time.ZonedDateTime;
import java.time.temporal.ChronoUnit;
import java.util.UUID;

/**
 * This entity represents a soft deleted {@link FavouriteSearchEntity}.
 */
@Entity
@Table(name = "archived_favourite_search")
@Data
@SuperBuilder
@EqualsAndHashCode(callSuper = true)
@NoArgsConstructor
public class ArchivedFavouriteSearchEntity extends AbstractUUIDEntity {
    /**
     * The name of the favourite search.
     */
    @Column(name = "name", nullable = false, updatable = false)
    private String name;

    /**
     * ID of the associated user.
     */
    @Column(name = "user_id", nullable = false, updatable = false)
    private UUID userId;

    /**
     * The context of the favourite search.
     */
    @Column(name = "context", nullable = false, updatable = false)
    @Convert(converter = SearchContextAttributeConverter.class)
    private SearchContext context;

    /**
     * The text used to perform the search.
     */
    @Column(name = "search_text", nullable = false, updatable = false)
    private String searchText;

    /**
     * Raw filters of the search provided by the frontend
     */
    @Column(name = "search_filters", updatable = false)
    private String searchFilters;

    /**
     * The timestamp when the favouriteSearch was created
     * in the original (non-archived) table.
     */
    @Column(name = "creation_time", nullable = false, updatable = false)
    private ZonedDateTime creationTime;

    /**
     * The timestamp when the favouriteSearch was last updated
     * in the original (non-archived) table.
     */
    @Column(name = "update_time", nullable = false, updatable = false)
    private ZonedDateTime updateTime;

    /**
     * The timestamp when this record was inserted into the archive table,
     * representing the moment the favouriteSearch was archived.
     */
    @Column(name = "archival_time", nullable = false, updatable = false)
    private ZonedDateTime archivalTime;

    /**
     * The version field used for optimistic locking.
     * <p>
     * This value is automatically managed by JPA to detect concurrent updates.
     * Each time the entity is updated, the version is incremented.
     * If two transactions try to update the same entity simultaneously,
     * JPA will detect the conflict based on this version
     * and throw an {@link jakarta.persistence.OptimisticLockException}.
     */
    @Version
    private long version;

    /**
     * Sets {@code archivalTime} before the entity is persisted.
     * <p>
     * The timestamp is stored in UTC to ensure consistent and timezone-safe timestamps.
     * The result is truncated to microseconds to match PostgreSQL's default precision
     * for {@code TIMESTAMPTZ} columns (6 digits).
     * Avoids using Hibernate's {@code @CreationTimestamp} to ensure timestamps
     * are immediately available after {@code JpaRepository.save()},
     * without requiring an explicit {@code JpaRepository.flush()}.
     * </p>
     */
    @PrePersist
    public void onCreate() {
        archivalTime = ZonedDateTime.now(ZoneOffset.UTC).truncatedTo(ChronoUnit.MICROS);
    }

}
