package com.finconsgroup.itserr.marketplace.favourite.user.dm.entity;

import com.finconsgroup.itserr.marketplace.core.entity.AbstractUUIDEntity;
import com.finconsgroup.itserr.marketplace.favourite.user.dm.converter.ItemContextAttributeConverter;
import com.finconsgroup.itserr.marketplace.favourite.user.dm.enums.ItemContext;
import jakarta.persistence.Column;
import jakarta.persistence.Convert;
import jakarta.persistence.Entity;
import jakarta.persistence.PreUpdate;
import jakarta.persistence.Table;
import jakarta.persistence.Version;
import lombok.EqualsAndHashCode;
import lombok.Getter;
import lombok.NoArgsConstructor;
import lombok.Setter;
import lombok.experimental.SuperBuilder;

import java.time.Instant;
import java.util.UUID;

import static com.finconsgroup.itserr.marketplace.favourite.user.dm.util.DomainConstants.CONTEXT_LENGTH;
import static com.finconsgroup.itserr.marketplace.favourite.user.dm.util.DomainConstants.ITEM_ID_LENGTH;
import static com.finconsgroup.itserr.marketplace.favourite.user.dm.util.DomainConstants.SUB_CONTEXT_LENGTH;

/**
 * This entity represents a favourite user item.
 */
@Entity
@Table(name = "favourite_user_item")
@SuperBuilder
@Getter
@Setter
@EqualsAndHashCode(callSuper = true)
@NoArgsConstructor
public class FavouriteUserItemEntity extends AbstractUUIDEntity {

    /**
     * Item User id.
     */
    @Column(name = "user_id", nullable = false, updatable = false)
    private UUID userId;

    /**
     * Item context.
     */
    @Column(name = "context", nullable = false, updatable = false, length = CONTEXT_LENGTH)
    @Convert(converter = ItemContextAttributeConverter.class)
    private ItemContext context;

    /**
     * Item sub context.
     */
    @Column(name = "sub_context", length = SUB_CONTEXT_LENGTH)
    private String subContext;

    /**
     * Identifier of the referenced item.
     */
    @Column(name = "item_id", nullable = false, length = ITEM_ID_LENGTH)
    private String itemId;

    /**
     * Flag to indicate if the item has been followed by the user.
     */
    @Column(name = "followed", nullable = false)
    private boolean followed;

    /**
     * Creation timestamp
     */
    @Column(name = "creation_time", nullable = false)
    private Instant creationTime;

    /**
     * Update timestamp
     */
    @Column(name = "update_time", nullable = false)
    private Instant updateTime;

    /**
     * The version field used for optimistic locking.
     * <p>
     * This value is automatically managed by JPA to detect concurrent updates.
     * Each time the entity is updated, the version is incremented.
     * If two transactions try to update the same entity simultaneously,
     * JPA will detect the conflict based on this version
     * and throw an {@link jakarta.persistence.OptimisticLockException}.
     * </p>
     */
    @Version
    private long version;

    @Override
    public void prePersist() {
        super.prePersist();
        Instant currentTime = Instant.now();
        if (creationTime == null) {
            creationTime = currentTime;
        }
        if (updateTime == null) {
            updateTime = currentTime;
        }
    }

    /**
     * Sets the updateTime (always) on UPDATE to current timestamp.
     */
    @PreUpdate
    public void preUpdate() {
        updateTime = Instant.now();
    }
}
