package com.finconsgroup.itserr.marketplace.favourite.user.dm.repository;

import com.finconsgroup.itserr.marketplace.favourite.user.dm.entity.FavouriteUserItemEntity;
import com.finconsgroup.itserr.marketplace.favourite.user.dm.enums.ItemContext;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.data.jpa.repository.JpaRepository;
import org.springframework.data.jpa.repository.Query;
import org.springframework.data.repository.query.Param;
import org.springframework.lang.NonNull;

import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.UUID;

/**
 * Repository for operations related to {@link FavouriteUserItemEntity}
 */
public interface FavouriteUserItemRepository extends JpaRepository<FavouriteUserItemEntity, UUID> {
    /**
     * Counts the favourite user item having given name and context.
     *
     * @param userId  the user id
     * @param context the context
     * @param itemId  the item id
     * @return Number of favourite user item for the user matching the context and item id
     */
    int countByUserIdAndContextAndItemId(@NonNull UUID userId, @NonNull ItemContext context, @NonNull String itemId);

    /**
     * Finds the favourite user item for logged in user and provided id
     *
     * @param userId the logged in user id
     * @param uuid   the entity id
     * @return Optional representing the found FavouriteUserItemEntity, empty otherwise
     */
    Optional<FavouriteUserItemEntity> findByUserIdAndId(UUID userId, UUID uuid);

    /**
     * Finds the favourite user item for provided context.
     *
     * @param context  the context to search for
     * @param pageable the page request
     * @return the found page of favourite user item results
     */
    Page<FavouriteUserItemEntity> findByUserIdAndContext(@NonNull UUID userId, @NonNull ItemContext context, Pageable pageable);

    /**
     * Finds the favourite user item for provided context.
     *
     * @param context    the context to search for
     * @param subContext the sub context to search for
     * @param pageable   the page request
     * @return the found page of favourite user item results
     */
    Page<FavouriteUserItemEntity> findByUserIdAndContextAndSubContext(@NonNull UUID userId, @NonNull ItemContext context, @NonNull String subContext, Pageable pageable);

    /**
     * Finds the favourite user items for provided context and item ids.
     *
     * @param userId  the logged in user id
     * @param context the context to search for
     * @param itemIds the item ids to filter to
     * @return the found list of favourite user item results
     */
    List<FavouriteUserItemEntity> findByUserIdAndContextAndItemIdIn(@NonNull UUID userId, @NonNull ItemContext context,
                                                                    @NonNull Set<String> itemIds);

    /**
     * Finds the favourite user items for provided context and item ids.
     *
     * @param userId     the logged in user id
     * @param context    the context to search for
     * @param subContext the sub context to search for
     * @param itemIds    the item ids to filter to
     * @return the found list of favourite user item results
     */
    List<FavouriteUserItemEntity> findByUserIdAndContextAndSubContextAndItemIdIn(@NonNull UUID userId,
                                                                                 @NonNull ItemContext context,
                                                                                 @NonNull String subContext,
                                                                                 @NonNull Set<String> itemIds);

    /**
     * Finds the favourite items followed by user for provided context and item id.
     *
     * @param context    the context to search for
     * @param itemId     the item id to filter to
     * @param subContext the sub context to search for
     * @param pageable   the page request
     * @return the found list of favourite user item results
     */
    @Query("SELECT T FROM FavouriteUserItemEntity AS T " +
            "WHERE T.context = :context " +
            "AND T.itemId = :itemId " +
            "AND T.followed = true " +
            "AND (:subContext IS NULL OR T.subContext = :subContext)")
    Page<FavouriteUserItemEntity> findByContextAndItemIdAndFollowedIsTrueAndOptionalSubContext(
            @NonNull @Param("context") ItemContext context,
            @NonNull @Param("itemId") String itemId,
            @Param("subContext") String subContext,
            @NonNull Pageable pageable);

    /**
     * Finds the favourite items for all users for provided context and item id.
     *
     * @param context    the context to search for
     * @param itemId     the item id to filter to
     * @param subContext the sub context to search for
     * @return the found list of favourite user item results
     */
    @Query("SELECT T FROM FavouriteUserItemEntity AS T " +
            "WHERE T.context = :context " +
            "AND T.itemId = :itemId " +
            "AND (:subContext IS NULL OR T.subContext = :subContext)")
    List<FavouriteUserItemEntity> findAllByContextAndItemIdAndOptionalSubContext(
            @NonNull @Param("context") ItemContext context,
            @NonNull @Param("itemId") String itemId,
            @Param("subContext") String subContext);
}
