package com.finconsgroup.itserr.marketplace.institutional_page.bs.messaging.impl;

import com.finconsgroup.itserr.marketplace.institutional_page.bs.client.dm.dto.InstitutionalPageIPDmDto;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.client.dm.dto.ModerationIPDmDto;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.dto.InputInviteMembersDto;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.dto.InputModerationStatusDto;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.dto.InputParagraphDto;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.dto.InputPatchIPInvitationRequestDto;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.dto.InputPatchIPJoinRequestDto;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.dto.InputPatchMembershipDto;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.dto.InputRemoveMembershipDto;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.dto.InputSubmitJoinRequestDto;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.dto.OutputInstitutionalPageDto;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.dto.OutputPendingMemberRequestDto;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.dto.OutputPendingMemberRequestsDto;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.dto.OutputUserProfileDto;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.messaging.NotificationMessageFactory;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.messaging.data.InstitutionalPageCreatedNotificationData;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.messaging.data.InstitutionalPageDeletedNotificationData;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.messaging.data.InstitutionalPagePublicationRequestNotificationData;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.messaging.data.InstitutionalPageUpdatedNotificationData;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.messaging.data.InstitutionalPageStatusChangeNotificationData;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.messaging.data.MembershipRemovalNotificationData;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.messaging.data.ParagraphCreatedNotificationData;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.messaging.data.ParagraphUpdatedNotificationData;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.messaging.data.ParagraphDeletedNotificationData;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.messaging.data.MemberInvitationNotificationData;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.messaging.data.MemberInvitationStatusChangeNotificationData;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.messaging.data.JoinRequestNotificationData;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.messaging.data.JoinRequestStatusChangeNotificationData;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.messaging.data.MembershipChangeNotificationData;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.repository.UserProfileRepository;
import com.finconsgroup.itserr.marketplace.core.web.security.jwt.JwtTokenHolder;
import com.finconsgroup.itserr.messaging.dto.MessagingEventDto;
import com.finconsgroup.itserr.messaging.dto.MessagingEventUserDto;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Component;

import java.time.Instant;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.UUID;
import java.util.stream.Collectors;

@Component
@RequiredArgsConstructor
public class DefaultNotificationMessageFactory implements NotificationMessageFactory {
    private final UserProfileRepository userProfileRepository;

    private MessagingEventUserDto getCurrentUser() {
        UUID userId = JwtTokenHolder.getUserIdOrThrow();
        String username = JwtTokenHolder.getPreferredUsernameOrThrow();
        Optional<String> givenName = JwtTokenHolder.getGivenName();
        Optional<String> familyName = JwtTokenHolder.getFamilyName();
        String fullName = (givenName.orElse("") + " " + familyName.orElse("")).trim();
        
        return MessagingEventUserDto.builder()
                .id(userId)
                .username(username)
                .name(fullName.isEmpty() ? username : fullName)
                .build();
    }

    @Override
    public MessagingEventDto<InstitutionalPageCreatedNotificationData> createInstitutionalPageCreatedNotification(OutputInstitutionalPageDto institutionalPageDto) {
        UUID admin = userProfileRepository.getAdmin();
        
        InstitutionalPageCreatedNotificationData data = InstitutionalPageCreatedNotificationData.builder()
                .published(institutionalPageDto.getPublished())
                .notifyUserIds(List.of(admin))
                .build();
        
        return MessagingEventDto.<InstitutionalPageCreatedNotificationData>builder()
                .id(institutionalPageDto.getId().toString())
                .name(institutionalPageDto.getName())
                .user(getCurrentUser())
                .category(institutionalPageDto.getCategory())
                .timestamp(institutionalPageDto.getCreationTime())
                .additionalData(data)
                .build();
    }

    @Override
    public MessagingEventDto<InstitutionalPageUpdatedNotificationData> createInstitutionalPageUpdatedNotification(OutputInstitutionalPageDto institutionalPageDto) {
        UUID admin = userProfileRepository.getAdmin();
        
        InstitutionalPageUpdatedNotificationData data = InstitutionalPageUpdatedNotificationData.builder()
                .published(institutionalPageDto.getPublished())
                .notifyUserIds(List.of(admin))
                .build();
        
        return MessagingEventDto.<InstitutionalPageUpdatedNotificationData>builder()
                .id(institutionalPageDto.getId().toString())
                .name(institutionalPageDto.getName())
                .user(getCurrentUser())
                .category(institutionalPageDto.getCategory())
                .timestamp(institutionalPageDto.getUpdateTime())
                .additionalData(data)
                .build();
    }

    @Override
    public MessagingEventDto<InstitutionalPageDeletedNotificationData> createInstitutionalPageDeletedNotification(InstitutionalPageIPDmDto institutionalPageDto) {
        UUID admin = userProfileRepository.getAdmin();

        InstitutionalPageDeletedNotificationData data = InstitutionalPageDeletedNotificationData.builder()
                .published(institutionalPageDto.getPublished())
                .notifyUserIds(List.of(admin))
                .build();
        
        return MessagingEventDto.<InstitutionalPageDeletedNotificationData>builder()
                .id(institutionalPageDto.getId().toString())
                .name(institutionalPageDto.getName())
                .user(getCurrentUser())
                .category(institutionalPageDto.getCategory())
                .timestamp(Instant.now())
                .additionalData(data)
                .build();
    }

    @Override
    public MessagingEventDto<InstitutionalPageStatusChangeNotificationData> createInstitutionalPageStatusChangeNotification(InputModerationStatusDto inputModerationStatusDto, ModerationIPDmDto moderationIPDmDto) {
        List<UUID> wpLeaderIds = moderationIPDmDto.getInstitutionalPage().getWpLeads();
        
        InstitutionalPageStatusChangeNotificationData data = InstitutionalPageStatusChangeNotificationData.builder()
                .published(moderationIPDmDto.getInstitutionalPage().getPublished())
                .approved(inputModerationStatusDto.getApproved())
                .message(inputModerationStatusDto.getMessage())
                .operationType(moderationIPDmDto.getOperationType())
                .notifyUserIds(wpLeaderIds)
                .currentMembers(moderationIPDmDto.getInstitutionalPage().getMembers())
                .currentWpLeaders(moderationIPDmDto.getInstitutionalPage().getWpLeads())
                .maintainer(moderationIPDmDto.getInstitutionalPage().getMaintainer())
                .rootInstitutionalPageId(moderationIPDmDto.getInstitutionalPage().getRootInstitutionalPageId())
                .rootInstitutionalPageName(moderationIPDmDto.getInstitutionalPage().getRootInstitutionalPageName())
                .build();
        
        return MessagingEventDto.<InstitutionalPageStatusChangeNotificationData>builder()
                .id(moderationIPDmDto.getInstitutionalPage().getId().toString())
                .name(moderationIPDmDto.getInstitutionalPage().getName())
                .user(getCurrentUser())
                .category(moderationIPDmDto.getInstitutionalPage().getCategory())
                .timestamp(moderationIPDmDto.getInstitutionalPage().getUpdateTime())
                .additionalData(data)
                .build();
    }

    @Override
    public MessagingEventDto<InstitutionalPagePublicationRequestNotificationData> createInstitutionalPagePublicationRequestNotification(OutputInstitutionalPageDto institutionalPageDto) {
        InstitutionalPagePublicationRequestNotificationData data = InstitutionalPagePublicationRequestNotificationData.builder()
                .notifyUserIds(List.of(userProfileRepository.getAdmin()))
                .build();

        return MessagingEventDto.<InstitutionalPagePublicationRequestNotificationData>builder()
                .id(institutionalPageDto.getId().toString())
                .name(institutionalPageDto.getName())
                .user(getCurrentUser())
                .category(institutionalPageDto.getCategory())
                .timestamp(institutionalPageDto.getUpdateTime())
                .additionalData(data)
                .build();
    }

    @Override
    public MessagingEventDto<JoinRequestNotificationData> createJoinRequestNotification(UUID institutionalPageId, InputSubmitJoinRequestDto inputSubmitJoinRequestDto, OutputPendingMemberRequestsDto pendingMemberRequestsForUserDto) {
        // Find the institutional page from pending requests to get wp leaders
        var institutionalPage = pendingMemberRequestsForUserDto.getJoinRequests().stream()
                .map(OutputPendingMemberRequestDto::getInstitutionalPage)
                .filter(page -> page.getId().equals(institutionalPageId))
                .findFirst()
                .orElseThrow();
        
        List<UUID> wpLeaderIds = institutionalPage.getWpLeads().stream()
                .map(OutputUserProfileDto::getId)
                .toList();
        
        JoinRequestNotificationData data = JoinRequestNotificationData.builder()
                .notifyUserIds(wpLeaderIds)
                .build();
        
        return MessagingEventDto.<JoinRequestNotificationData>builder()
                .id(institutionalPage.getId().toString())
                .name(institutionalPage.getName())
                .user(getCurrentUser())
                .category(institutionalPage.getCategory())
                .timestamp(Instant.now())
                .additionalData(data)
                .build();
    }

    @Override
    public MessagingEventDto<JoinRequestNotificationData> createJoinRequestDeletedNotification(InstitutionalPageIPDmDto institutionalPageIPDmDto) {
        List<UUID> wpLeaderIds = institutionalPageIPDmDto.getWpLeads();
        
        JoinRequestNotificationData data = JoinRequestNotificationData.builder()
                .notifyUserIds(wpLeaderIds)
                .build();
        
        return MessagingEventDto.<JoinRequestNotificationData>builder()
                .id(institutionalPageIPDmDto.getId().toString())
                .name(institutionalPageIPDmDto.getName())
                .user(getCurrentUser())
                .category(institutionalPageIPDmDto.getCategory())
                .timestamp(Instant.now())
                .additionalData(data)
                .build();
    }

    @Override
    public MessagingEventDto<JoinRequestStatusChangeNotificationData> createJoinRequestStatusChangeNotification(InstitutionalPageIPDmDto institutionalPageIPDmDto, InputPatchIPJoinRequestDto inputPatchIPJoinRequestDto) {
        JoinRequestStatusChangeNotificationData data = JoinRequestStatusChangeNotificationData.builder()
                .approved(inputPatchIPJoinRequestDto.getApproved())
                .message(inputPatchIPJoinRequestDto.getMessage())
                .notifyUserIds(inputPatchIPJoinRequestDto.getUserIds())
                .rootInstitutionalPageId(institutionalPageIPDmDto.getRootInstitutionalPageId())
                .rootInstitutionalPageName(institutionalPageIPDmDto.getRootInstitutionalPageName())
                .requestingUsers(inputPatchIPJoinRequestDto.getUserIds())
                .build();
        
        return MessagingEventDto.<JoinRequestStatusChangeNotificationData>builder()
                .id(institutionalPageIPDmDto.getId().toString())
                .name(institutionalPageIPDmDto.getName())
                .user(getCurrentUser())
                .category(institutionalPageIPDmDto.getCategory())
                .timestamp(Instant.now())
                .additionalData(data)
                .build();
    }

    @Override
    public MessagingEventDto<MemberInvitationNotificationData> createMemberInvitationNotification(InstitutionalPageIPDmDto institutionalPageIPDmDto, InputInviteMembersDto inputInviteMembersDto) {
        MemberInvitationNotificationData data = MemberInvitationNotificationData.builder()
                .message(inputInviteMembersDto.getMessage())
                .wpLeader(inputInviteMembersDto.getWpLeader())
                .notifyUserIds(inputInviteMembersDto.getUserIds())
                .build();
        
        return MessagingEventDto.<MemberInvitationNotificationData>builder()
                .id(institutionalPageIPDmDto.getId().toString())
                .name(institutionalPageIPDmDto.getName())
                .user(getCurrentUser())
                .category(institutionalPageIPDmDto.getCategory())
                .timestamp(Instant.now())
                .additionalData(data)
                .build();
    }

    @Override
    public MessagingEventDto<MemberInvitationNotificationData> createMemberInvitationDeletedNotification(InstitutionalPageIPDmDto institutionalPageIPDmDto, UUID invitedUserId) {
        MemberInvitationNotificationData data = MemberInvitationNotificationData.builder()
                .notifyUserIds(List.of(invitedUserId))
                .build();
        
        return MessagingEventDto.<MemberInvitationNotificationData>builder()
                .id(institutionalPageIPDmDto.getId().toString())
                .name(institutionalPageIPDmDto.getName())
                .user(getCurrentUser())
                .category(institutionalPageIPDmDto.getCategory())
                .timestamp(Instant.now())
                .additionalData(data)
                .build();
    }

    @Override
    public MessagingEventDto<MemberInvitationStatusChangeNotificationData> createMemberInvitationStatusChangeNotification(
            InstitutionalPageIPDmDto institutionalPageIPDmDto,
            InputPatchIPInvitationRequestDto inputPatchIPInvitationRequestDto,
            Boolean wpLeader,
            List<OutputInstitutionalPageDto> hierarchyInstitutionalPages) {
        Map<UUID, String> hierarchyInstitutionalPagesMap = hierarchyInstitutionalPages.stream()
                .collect(Collectors.toMap(OutputInstitutionalPageDto::getId, OutputInstitutionalPageDto::getName));

        MemberInvitationStatusChangeNotificationData data = MemberInvitationStatusChangeNotificationData.builder()
                .approved(inputPatchIPInvitationRequestDto.getApproved())
                .message(inputPatchIPInvitationRequestDto.getMessage())
                .wpLeader(wpLeader)
                .notifyUserIds(institutionalPageIPDmDto.getWpLeads())
                .rootInstitutionalPageId(institutionalPageIPDmDto.getRootInstitutionalPageId())
                .rootInstitutionalPageName(institutionalPageIPDmDto.getRootInstitutionalPageName())
                .invitedUser(JwtTokenHolder.getUserIdOrThrow())
                .hierarchyInstitutionalPages(hierarchyInstitutionalPagesMap)
                .build();
        
        return MessagingEventDto.<MemberInvitationStatusChangeNotificationData>builder()
                .id(institutionalPageIPDmDto.getId().toString())
                .name(institutionalPageIPDmDto.getName())
                .user(getCurrentUser())
                .category(institutionalPageIPDmDto.getCategory())
                .timestamp(Instant.now())
                .additionalData(data)
                .build();
    }

    @Override
    public MessagingEventDto<MembershipChangeNotificationData> createMembershipChangeNotification(
            InputPatchMembershipDto inputPatchMembershipDto,
            OutputInstitutionalPageDto institutionalPageDto,
            List<OutputInstitutionalPageDto> hierarchyInstitutionalPages
    ) {
        MembershipChangeNotificationData data = MembershipChangeNotificationData.builder()
                .wpLeader(inputPatchMembershipDto.getWpLeader())
                .memberUserId(inputPatchMembershipDto.getUserId())
                .hierarchyInstitutionalPages(hierarchyInstitutionalPages.stream()
                        .collect(Collectors.toMap(OutputInstitutionalPageDto::getId, OutputInstitutionalPageDto::getName)))
                .notifyUserIds(List.of(inputPatchMembershipDto.getUserId()))
                .build();
        
        return MessagingEventDto.<MembershipChangeNotificationData>builder()
                .id(institutionalPageDto.getId().toString())
                .name(institutionalPageDto.getName())
                .user(getCurrentUser())
                .category(institutionalPageDto.getCategory())
                .timestamp(institutionalPageDto.getUpdateTime())
                .additionalData(data)
                .build();
    }

    @Override
    public MessagingEventDto<MembershipRemovalNotificationData> createMembershipRemovalNotification(
            InputRemoveMembershipDto inputRemoveMembershipDto,
            boolean wpLeader,
            OutputInstitutionalPageDto institutionalPageDto,
            List<OutputInstitutionalPageDto> hierarchyInstitutionalPages) {
        MembershipRemovalNotificationData data = MembershipRemovalNotificationData.builder()
                .wpLeader(wpLeader)
                .memberUserId(inputRemoveMembershipDto.getUserId())
                .hierarchyInstitutionalPages(hierarchyInstitutionalPages.stream().collect(Collectors.toMap(OutputInstitutionalPageDto::getId, OutputInstitutionalPageDto::getName)))
                .notifyUserIds(List.of(inputRemoveMembershipDto.getUserId()))
                .build();

        return MessagingEventDto.<MembershipRemovalNotificationData>builder()
                .id(institutionalPageDto.getId().toString())
                .name(institutionalPageDto.getName())
                .user(getCurrentUser())
                .category(institutionalPageDto.getCategory())
                .timestamp(institutionalPageDto.getUpdateTime())
                .additionalData(data)
                .build();
    }

    @Override
    public MessagingEventDto<ParagraphCreatedNotificationData> createParagraphCreatedNotification(InputParagraphDto inputParagraphDto, OutputInstitutionalPageDto institutionalPageDto) {
        UUID admin = userProfileRepository.getAdmin();
        
        ParagraphCreatedNotificationData data = ParagraphCreatedNotificationData.builder()
                .paragraphTitle(inputParagraphDto.getTitle())
                .notifyUserIds(List.of(admin))
                .build();

        return MessagingEventDto.<ParagraphCreatedNotificationData>builder()
                .id(institutionalPageDto.getId().toString())
                .name(institutionalPageDto.getName())
                .user(getCurrentUser())
                .category(institutionalPageDto.getCategory())
                .timestamp(Instant.now())
                .additionalData(data)
                .build();
    }

    @Override
    public MessagingEventDto<ParagraphUpdatedNotificationData> createParagraphUpdatedNotification(InputParagraphDto inputParagraphDto, OutputInstitutionalPageDto institutionalPageDto) {
        UUID admin = userProfileRepository.getAdmin();
        
        ParagraphUpdatedNotificationData data = ParagraphUpdatedNotificationData.builder()
                .paragraphTitle(inputParagraphDto.getTitle())
                .notifyUserIds(List.of(admin))
                .build();
        
        return MessagingEventDto.<ParagraphUpdatedNotificationData>builder()
                .id(institutionalPageDto.getId().toString())
                .name(institutionalPageDto.getName())
                .user(getCurrentUser())
                .category(institutionalPageDto.getCategory())
                .timestamp(institutionalPageDto.getUpdateTime())
                .additionalData(data)
                .build();
    }

    @Override
    public MessagingEventDto<ParagraphDeletedNotificationData> createParagraphDeletedNotification(InputParagraphDto inputParagraphDto, OutputInstitutionalPageDto institutionalPageDto) {
        UUID admin = userProfileRepository.getAdmin();
        
        ParagraphDeletedNotificationData data = ParagraphDeletedNotificationData.builder()
                .paragraphTitle(inputParagraphDto.getTitle())
                .notifyUserIds(List.of(admin))
                .build();
        
        return MessagingEventDto.<ParagraphDeletedNotificationData>builder()
                .id(institutionalPageDto.getId().toString())
                .name(institutionalPageDto.getName())
                .user(getCurrentUser())
                .category(institutionalPageDto.getCategory())
                .timestamp(Instant.now())
                .additionalData(data)
                .build();
    }

    @Override
    public MessagingEventDto<MemberInvitationNotificationData> createPendingInvitationRequestsNotification(OutputInstitutionalPageDto institutionalPageDto, OutputPendingMemberRequestsDto pendingRequestsDto) {
        // Extract user IDs from pending invitations
        List<UUID> invitedUserIds = pendingRequestsDto.getInvitations().stream()
                .map(OutputPendingMemberRequestDto::getUser)
                .map(OutputUserProfileDto::getId)
                .toList();

        MemberInvitationNotificationData data = MemberInvitationNotificationData.builder()
                .notifyUserIds(invitedUserIds)
                .build();
        
        return MessagingEventDto.<MemberInvitationNotificationData>builder()
                .id(institutionalPageDto.getId().toString())
                .name(institutionalPageDto.getName())
                .user(getCurrentUser())
                .category(institutionalPageDto.getCategory())
                .timestamp(Instant.now())
                .additionalData(data)
                .build();
    }
}
