package com.finconsgroup.itserr.marketplace.institutionalpage.dm.api;

import com.finconsgroup.itserr.marketplace.institutionalpage.dm.dto.InputPatchIPModerationRequestDto;
import com.finconsgroup.itserr.marketplace.institutionalpage.dm.dto.InputSearchPendingInstitutionalPageDto;
import com.finconsgroup.itserr.marketplace.institutionalpage.dm.dto.OutputInstitutionalPageDto;
import com.finconsgroup.itserr.marketplace.institutionalpage.dm.dto.OutputModerationDto;
import com.finconsgroup.itserr.marketplace.institutionalpage.dm.validation.ValidAssociationToLoad;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.responses.ApiResponse;
import io.swagger.v3.oas.annotations.security.SecurityRequirement;
import io.swagger.v3.oas.annotations.tags.Tag;
import jakarta.validation.Valid;
import jakarta.validation.constraints.Positive;
import jakarta.validation.constraints.PositiveOrZero;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Sort;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PatchMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseStatus;

import java.util.Set;
import java.util.UUID;

/**
 * This interface defines the contract for REST API endpoints related moderation workflow
 * for InstitutionalPages.
 *
 * <p>
 * It provides endpoints to approve or reject and retrieve latest version Pending InstitutionalPages.
 * </p>
 *
 * <p>Example usage:
 * <pre>
 * GET    /api/v1/dm/institutional-page/institutional-pages/pending                                - Retrieve all latest version Pending IPs.
 * POST   /api/v1/dm/institutional-page/institutional-pages/pending/search                         - Retrieves a paginated list of all latest version Pending InstitutionalPages, matching the search criteria..
 * GET    /api/v1/dm/institutional-page/institutional-pages/pending/{institutionalPageId}          - Retrieve a latest version Pending IP by id.
 * PATCH  /api/v1/dm/institutional-page/institutional-pages/pending/{institutionalPageId}          - Accept/Reject a Pending IP (admin).
 * </pre>
 * </p>
 */
@Tag(
        name = "Moderation",
        description = "The Moderation API: it provides endpoints to " +
                "approve or reject and retrieve latest version Pending InstitutionalPages."
)
@SecurityRequirement(name = "BearerAuth")
public interface ModerationApi {

    /**
     * Retrieves all latest version Pending InstitutionalPages.
     *
     * @param associationsToLoad comma separated list of the associations to be returned (default is "all").
     * @param pageNumber         the page number to retrieve (default is 0)
     * @param pageSize           the number of InstitutionalPages per page (default is 10)
     * @param sort               the field to sort by (default is "name")
     * @param direction          the direction of sorting (default is ascending)
     * @return a page of {@link OutputInstitutionalPageDto} and HTTP status 200 (OK)
     */
    @Operation(
            summary = "Retrieves all latest version Pending InstitutionalPages",
            responses = {@ApiResponse(responseCode = "200", description = "OK")}
    )
    @GetMapping(
            value = "/institutional-page/institutional-pages/pending",
            produces = MediaType.APPLICATION_JSON_VALUE
    )
    @ResponseStatus(HttpStatus.OK)
    Page<OutputInstitutionalPageDto> findAllLatestVersionPendingInstitutionalPages(
            @RequestParam(name = "associationsToLoad", defaultValue = "all", required = false) Set<@ValidAssociationToLoad String> associationsToLoad,
            @RequestParam(name = "pageNumber", defaultValue = "0", required = false) @PositiveOrZero int pageNumber,
            @RequestParam(name = "pageSize", defaultValue = "10", required = false) @Positive int pageSize,
            @RequestParam(name = "sort", defaultValue = "name", required = false) String sort,
            @RequestParam(name = "direction", defaultValue = "ASC", required = false) Sort.Direction direction
    );

    /**
     * Retrieves a paginated list of all latest version Pending InstitutionalPages the user is contributing to, matching the search criteria.
     *
     * @param inputSearchPendingInstitutionalPageDto the dto containing the filters to be applied
     * @param associationsToLoad                     comma separated list of the associations to be returned (default is "all").
     * @param pageNumber                             the page number to retrieve (default is 0)
     * @param pageSize                               the number of InstitutionalPages per page (default is 10)
     * @param sort                                   the field to sort by (default is "name")
     * @param direction                              the direction of sorting (default is ascending)
     * @return a page of {@link OutputInstitutionalPageDto} and HTTP status 200 (OK)
     */
    @Operation(
            summary = "Retrieves a paginated list of all latest version Pending InstitutionalPages the user is contributing to, matching the search criteria",
            responses = {@ApiResponse(responseCode = "200", description = "OK")}
    )
    @PostMapping(
            value = "/institutional-page/institutional-pages/pending/search",
            consumes = MediaType.APPLICATION_JSON_VALUE,
            produces = MediaType.APPLICATION_JSON_VALUE
    )
    @ResponseStatus(HttpStatus.OK)
    Page<OutputInstitutionalPageDto> searchLatestVersionPendingInstitutionalPages(
            @Valid @RequestBody InputSearchPendingInstitutionalPageDto inputSearchPendingInstitutionalPageDto,
            @RequestParam(name = "associationsToLoad", defaultValue = "all", required = false) Set<@ValidAssociationToLoad String> associationsToLoad,
            @RequestParam(name = "pageNumber", defaultValue = "0", required = false) @PositiveOrZero int pageNumber,
            @RequestParam(name = "pageSize", defaultValue = "10", required = false) @Positive int pageSize,
            @RequestParam(name = "sort", defaultValue = "name", required = false) String sort,
            @RequestParam(name = "direction", defaultValue = "ASC", required = false) Sort.Direction direction
    );

    /**
     * Retrieves a latest version Pending InstitutionalPage by id.
     *
     * @param institutionalPageId the id of the pending InstitutionalPage to retrieve
     * @return the {@link OutputInstitutionalPageDto} and HTTP status 200 (OK)
     */
    @Operation(
            summary = "Retrieves a latest version Pending InstitutionalPage by id (admin)",
            responses = {
                    @ApiResponse(responseCode = "200", description = "OK"),
                    @ApiResponse(responseCode = "404", description = "Not Found"),
            }
    )
    @GetMapping(
            value = "/institutional-page/institutional-pages/pending/{institutionalPageId}",
            produces = MediaType.APPLICATION_JSON_VALUE
    )
    @ResponseStatus(HttpStatus.OK)
    OutputInstitutionalPageDto findLatestVersionPendingInstitutionalPageById(
            @PathVariable("institutionalPageId") UUID institutionalPageId
    );

    /**
     * Accepts/Reject a pending InstitutionalPage (admin).
     *
     * @param institutionalPageId              the id of the pending InstitutionalPage
     * @param inputPatchIPModerationRequestDto the DTO to approve or reject a pending Institutional Page
     * @return the {@link OutputModerationDto} and HTTP status 200 (OK)
     */
    @Operation(
            summary = "Accepts/Reject a pending InstitutionalPage (admin)",
            responses = {
                    @ApiResponse(responseCode = "200", description = "OK"),
                    @ApiResponse(responseCode = "404", description = "Not Found"),
            }
    )
    @PatchMapping(
            value = "/institutional-page/institutional-pages/pending/{institutionalPageId}",
            produces = MediaType.APPLICATION_JSON_VALUE
    )
    @ResponseStatus(HttpStatus.OK)
    OutputModerationDto acceptOrRejectPendingInstitutionalPage(
            @PathVariable("institutionalPageId") UUID institutionalPageId,
            @Valid @RequestBody InputPatchIPModerationRequestDto inputPatchIPModerationRequestDto
    );

}
