package com.finconsgroup.itserr.marketplace.institutionalpage.dm.controller;

import com.finconsgroup.itserr.marketplace.core.web.security.jwt.JwtTokenHolder;
import com.finconsgroup.itserr.marketplace.institutionalpage.dm.api.MemberApi;
import com.finconsgroup.itserr.marketplace.institutionalpage.dm.dto.InputInviteMembersDto;
import com.finconsgroup.itserr.marketplace.institutionalpage.dm.dto.InputPatchIPInvitationRequestDto;
import com.finconsgroup.itserr.marketplace.institutionalpage.dm.dto.InputPatchIPJoinRequestDto;
import com.finconsgroup.itserr.marketplace.institutionalpage.dm.dto.InputPatchMembershipDto;
import com.finconsgroup.itserr.marketplace.institutionalpage.dm.dto.InputRemoveMembershipDto;
import com.finconsgroup.itserr.marketplace.institutionalpage.dm.dto.InputSubmitJoinRequestDto;
import com.finconsgroup.itserr.marketplace.institutionalpage.dm.dto.OutputInstitutionalPageDto;
import com.finconsgroup.itserr.marketplace.institutionalpage.dm.dto.OutputMembersInHierarchyDto;
import com.finconsgroup.itserr.marketplace.institutionalpage.dm.dto.OutputPendingMemberRequestsDto;
import com.finconsgroup.itserr.marketplace.institutionalpage.dm.service.MemberService;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.web.bind.annotation.RestController;

import java.util.UUID;

/**
 * REST controller class for handling Member related API requests.
 *
 * <p>Implements the {@link MemberApi} interface.</p>
 */
@Slf4j
@RequiredArgsConstructor
@RestController
public class MemberController implements MemberApi {

    private final MemberService memberService;

    @Override
    public Boolean hasIPModerateRole() {
        log.debug("GET /api/v1/dm/institutional-page/admin");
        return this.memberService.hasIPModerateRole();
    }

    @Override
    public OutputPendingMemberRequestsDto findPendingMemberRequestsForInstitutionalPage(
            final UUID institutionalPageId
    ) {
        log.debug("GET /api/v1/dm/institutional-page/institutional-pages/{}/members/requests", institutionalPageId);
        UUID userId = JwtTokenHolder.getUserIdOrThrow();
        return this.memberService.findPendingMemberRequestsForInstitutionalPage(userId, institutionalPageId);
    }

    @Override
    public OutputPendingMemberRequestsDto findPendingMemberRequestsForUser() {
        log.debug("GET /api/v1/dm/institutional-page/institutional-pages/members/requests");
        UUID userId = JwtTokenHolder.getUserIdOrThrow();
        return this.memberService.findPendingMemberRequestsForUser(userId);
    }

    @Override
    public OutputPendingMemberRequestsDto inviteMembers(
            final UUID institutionalPageId,
            final InputInviteMembersDto inputInviteMembersDto
    ) {
        log.debug("POST /api/v1/dm/institutional-page/institutional-pages/{}/members/requests/invitations, inputInviteMembersDto: {}", institutionalPageId, inputInviteMembersDto);
        UUID userId = JwtTokenHolder.getUserIdOrThrow();
        return this.memberService.inviteMembers(userId, institutionalPageId, inputInviteMembersDto);
    }

    @Override
    public OutputPendingMemberRequestsDto cancelInvitation(
            final UUID institutionalPageId,
            final UUID invitedUserId
    ) {
        log.debug("DELETE /api/v1/dm/institutional-page/institutional-pages/{}/members/requests/invitations/{}", institutionalPageId, invitedUserId);
        UUID userId = JwtTokenHolder.getUserIdOrThrow();
        return this.memberService.cancelInvitation(userId, institutionalPageId, invitedUserId);
    }

    @Override
    public OutputPendingMemberRequestsDto acceptOrRejectInvitation(
            final UUID institutionalPageId,
            final InputPatchIPInvitationRequestDto inputPatchIPInvitationRequestDto
    ) {
        log.debug("PATCH /api/v1/dm/institutional-page/institutional-pages/{}/members/requests/invitations, inputPatchIPInvitationRequestDto: {}", institutionalPageId, inputPatchIPInvitationRequestDto);
        UUID userId = JwtTokenHolder.getUserIdOrThrow();
        return this.memberService.acceptOrRejectInvitation(userId, institutionalPageId, inputPatchIPInvitationRequestDto);
    }

    @Override
    public OutputInstitutionalPageDto patchMembershipOfUsersForInstitutionalPage(
            final UUID institutionalPageId,
            final InputPatchMembershipDto inputPatchMembershipDto
    ) {
        log.debug("PATCH /api/v1/dm/institutional-page/institutional-pages/{}/members/membership, inputPatchMembershipDto: {}", institutionalPageId, inputPatchMembershipDto);
        UUID userId = JwtTokenHolder.getUserIdOrThrow();
        return this.memberService.patchMembershipOfUsersForInstitutionalPage(userId, institutionalPageId, inputPatchMembershipDto);
    }

    @Override
    public OutputInstitutionalPageDto removeMembershipOfUsersForInstitutionalPage(
            final UUID institutionalPageId,
            final InputRemoveMembershipDto inputRemoveMembershipDto
    ) {
        log.debug("DELETE /api/v1/dm/institutional-page/institutional-pages/{}/members/membership, inputRemoveMembershipDto: {}", institutionalPageId, inputRemoveMembershipDto);
        UUID userId = JwtTokenHolder.getUserIdOrThrow();
        return this.memberService.removeMembershipOfUsersForInstitutionalPage(userId, institutionalPageId, inputRemoveMembershipDto);
    }

    @Override
    public OutputPendingMemberRequestsDto submitJoinRequest(
            final UUID institutionalPageId,
            final InputSubmitJoinRequestDto inputSubmitJoinRequestDto
    ) {
        log.debug("POST /api/v1/dm/institutional-page/institutional-pages/{}/members/requests/join, inputSubmitJoinRequestDto: {}", institutionalPageId, inputSubmitJoinRequestDto);
        UUID userId = JwtTokenHolder.getUserIdOrThrow();
        return this.memberService.submitJoinRequest(userId, institutionalPageId, inputSubmitJoinRequestDto);
    }

    @Override
    public OutputPendingMemberRequestsDto cancelJoinRequest(final UUID institutionalPageId) {
        log.debug("DELETE /api/v1/dm/institutional-page/institutional-pages/{}/members/requests/join", institutionalPageId);
        UUID userId = JwtTokenHolder.getUserIdOrThrow();
        return this.memberService.cancelJoinRequest(userId, institutionalPageId);
    }

    @Override
    public OutputPendingMemberRequestsDto acceptOrRejectJoinRequests(
            final UUID institutionalPageId,
            final InputPatchIPJoinRequestDto inputPatchIPJoinRequestDto
    ) {
        log.debug("PATCH /api/v1/dm/institutional-page/institutional-pages/{}/members/requests/join, inputPatchIPJoinRequestDto: {}", institutionalPageId, inputPatchIPJoinRequestDto);
        UUID userId = JwtTokenHolder.getUserIdOrThrow();
        return this.memberService.acceptOrRejectJoinRequests(userId, institutionalPageId, inputPatchIPJoinRequestDto);
    }

    @Override
    public Page<OutputMembersInHierarchyDto> findAllUsersInHierarchy(
            final UUID rootInstitutionalPageId,
            final int pageNumber,
            final int pageSize
    ) {
        log.debug("GET /api/v1/dm/institutional-page/institutional-pages/{}/members/hierarchy", rootInstitutionalPageId);
        UUID userId = JwtTokenHolder.getUserIdOrThrow();
        Pageable pageable = PageRequest.of(pageNumber, pageSize);
        return memberService.findAllUsersInHierarchy(userId, pageable, rootInstitutionalPageId);
    }

}
