package com.finconsgroup.itserr.marketplace.institutionalpage.dm.component;

import com.finconsgroup.itserr.marketplace.institutionalpage.dm.dto.InputCreateInstitutionalPageDto;
import com.finconsgroup.itserr.marketplace.institutionalpage.dm.dto.InputUpdateInstitutionalPageDto;
import com.finconsgroup.itserr.marketplace.institutionalpage.dm.entity.ArchivedInstitutionalPageEntity;
import com.finconsgroup.itserr.marketplace.institutionalpage.dm.entity.InstitutionalPageEntity;
import com.finconsgroup.itserr.marketplace.institutionalpage.dm.entity.MemberEntity;
import com.finconsgroup.itserr.marketplace.institutionalpage.dm.entity.MemberRequestEntity;
import com.finconsgroup.itserr.marketplace.institutionalpage.dm.entity.ParagraphEntity;
import com.finconsgroup.itserr.marketplace.institutionalpage.dm.enums.MemberRequestType;
import com.finconsgroup.itserr.marketplace.institutionalpage.dm.mapper.ArchivedEntityMapper;
import com.finconsgroup.itserr.marketplace.institutionalpage.dm.mapper.InstitutionalPageMapper;
import com.finconsgroup.itserr.marketplace.institutionalpage.dm.mapper.MemberMapper;
import com.finconsgroup.itserr.marketplace.institutionalpage.dm.mapper.MemberRequestMapper;
import com.finconsgroup.itserr.marketplace.institutionalpage.dm.mapper.ParagraphMapper;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Component;

import java.util.ArrayList;
import java.util.LinkedList;
import java.util.List;
import java.util.Optional;
import java.util.UUID;

/**
 * Component responsible for building and updating entity instances based on input parameters,
 *
 * <p>
 * It provides factory methods for constructing new entities during creation workflows, cloning existing ones,
 * and updating entities with input data. It also ensures bidirectional associations are correctly maintained.
 * </p>
 */
@Component
@RequiredArgsConstructor
@Slf4j
public class EntityBuilder {

    private final InstitutionalPageMapper institutionalPageMapper;
    private final ParagraphMapper paragraphMapper;
    private final MemberMapper memberMapper;
    private final MemberRequestMapper memberRequestMapper;
    private final ArchivedEntityMapper archivedEntityMapper;

    // members

    public MemberEntity buildWPLeader(UUID userId, InstitutionalPageEntity institutionalPageEntity) {
        MemberEntity memberEntity = memberMapper.toMemberEntity(userId, true, institutionalPageEntity);
        return memberEntity;
    }

    public MemberEntity buildRegularMember(UUID userId, UUID institutionalPageId) {
        InstitutionalPageEntity institutionalPageEntity = InstitutionalPageEntity.builder()
                .id(institutionalPageId)
                .build();
        MemberEntity memberEntity = memberMapper.toMemberEntity(userId, false, institutionalPageEntity);
        return memberEntity;
    }

    // member requests

    public MemberRequestEntity buildInvitation(UUID institutionalPageId, UUID invitedUserId, boolean wpLeader, String message) {
        return memberRequestMapper.toMemberRequestEntity(
                invitedUserId,
                institutionalPageId,
                MemberRequestType.INVITATION,
                wpLeader,
                message
        );
    }

    public MemberRequestEntity buildJoinRequest(UUID institutionalPageId, UUID invitedUserId, String message) {
        return memberRequestMapper.toMemberRequestEntity(
                invitedUserId,
                institutionalPageId,
                MemberRequestType.JOIN_REQUEST,
                false,
                message
        );
    }

    // institutional pages

    public InstitutionalPageEntity buildInstitutionalPage(
            UUID userId,
            InstitutionalPageEntity parentInstitutionalPageEntity,
            InputCreateInstitutionalPageDto inputCreateInstitutionalPageDto
    ) {
        // map input fields to entity
        InstitutionalPageEntity institutionalPageEntity = institutionalPageMapper.toEntity(inputCreateInstitutionalPageDto);
        // set bidirectional association
        updateAssociationsForSave(institutionalPageEntity);
        // if provided from input, retrieves parent institutional page
        UUID parentInstitutionalPageId = inputCreateInstitutionalPageDto.getParentInstitutionalPageId();
        institutionalPageEntity.setParentInstitutionalPageId(parentInstitutionalPageId);
        // set members and ancestors
        List<MemberEntity> memberEntities = new LinkedList<>();
        List<UUID> ancestorIPs = new LinkedList<>();
        if (parentInstitutionalPageEntity == null) {
            memberEntities.add(buildWPLeader(userId, institutionalPageEntity));
        } else {
            ancestorIPs.addAll(parentInstitutionalPageEntity.getAncestorInstitutionalPageIds());
            ancestorIPs.addLast(parentInstitutionalPageEntity.getId());
        }
        institutionalPageEntity.setMembers(memberEntities);
        institutionalPageEntity.setAncestorInstitutionalPageIds(ancestorIPs);
        // set maintainer
        institutionalPageEntity.setMaintainer(userId);
        // set toDelete
        institutionalPageEntity.setToDelete(false);
        // set updateLockedBy
        institutionalPageEntity.setUpdateLockedBy(null);
        // trim name
        institutionalPageEntity.setName(institutionalPageEntity.getName().trim());
        return institutionalPageEntity;
    }

    public void updateAssociationsForSave(InstitutionalPageEntity institutionalPage) {
        Optional.ofNullable(institutionalPage.getMembers())
                .orElse(new LinkedList<>())
                .forEach(member -> member.setInstitutionalPage(institutionalPage));
        Optional.ofNullable(institutionalPage.getParagraphs())
                .orElse(new LinkedList<>())
                .forEach(paragraph -> paragraph.setInstitutionalPage(institutionalPage));
    }

    public InstitutionalPageEntity cloneInstitutionalPageEntity(InstitutionalPageEntity institutionalPageEntity) {
        InstitutionalPageEntity clonedInstitutionalPageEntity = institutionalPageMapper.clone(institutionalPageEntity);
        updateAssociationsForSave(clonedInstitutionalPageEntity);
        return clonedInstitutionalPageEntity;
    }

    public void updateEntity(InstitutionalPageEntity target, InstitutionalPageEntity source) {
        institutionalPageMapper.updateEntity(target, source);
        updateAssociationsForSave(target);
    }

    public void updateEntity(InstitutionalPageEntity institutionalPageEntity, InputUpdateInstitutionalPageDto inputUpdateInstitutionalPageDto) {
        institutionalPageMapper.updateEntity(institutionalPageEntity, inputUpdateInstitutionalPageDto);
        // map paragraphs
        if (institutionalPageEntity.getParagraphs() == null) {
            institutionalPageEntity.setParagraphs(new ArrayList<>());
        }
        institutionalPageEntity.getParagraphs().clear();
        if (inputUpdateInstitutionalPageDto.getParagraphs() != null) {
            List<ParagraphEntity> updatedParagraphs = paragraphMapper.inputParagraphListToParagraphEntityList(inputUpdateInstitutionalPageDto.getParagraphs());
            institutionalPageEntity.getParagraphs().addAll(updatedParagraphs);
        }
        //
        updateAssociationsForSave(institutionalPageEntity);
    }

    // archive

    public ArchivedInstitutionalPageEntity buildArchivedEntity(InstitutionalPageEntity institutionalPageEntity) {
        // map input fields to entity
        ArchivedInstitutionalPageEntity archivedInstitutionalPage = archivedEntityMapper
                .toArchiveEntity(institutionalPageEntity);
        // set bidirectional association
        updateAssociationsForSave(archivedInstitutionalPage);
        //
        return archivedInstitutionalPage;
    }

    public void updateAssociationsForSave(ArchivedInstitutionalPageEntity archivedInstitutionalPage) {
        Optional.ofNullable(archivedInstitutionalPage.getMembers())
                .orElse(new LinkedList<>())
                .forEach(member -> member.setInstitutionalPage(archivedInstitutionalPage));
        Optional.ofNullable(archivedInstitutionalPage.getParagraphs())
                .orElse(new LinkedList<>())
                .forEach(paragraph -> paragraph.setInstitutionalPage(archivedInstitutionalPage));
    }
}
