package com.finconsgroup.itserr.marketplace.institutionalpage.dm.component;

import com.finconsgroup.itserr.marketplace.core.web.exception.WP2BusinessException;
import com.finconsgroup.itserr.marketplace.institutionalpage.dm.entity.InstitutionalPageEntity;
import com.finconsgroup.itserr.marketplace.institutionalpage.dm.enums.ModerationStatus;
import com.finconsgroup.itserr.marketplace.institutionalpage.dm.exception.InstitutionalPageNotFoundException;
import com.finconsgroup.itserr.marketplace.institutionalpage.dm.model.SearchForMemberInstitutionalPageModel;
import com.finconsgroup.itserr.marketplace.institutionalpage.dm.model.SearchInstitutionalPageModel;
import com.finconsgroup.itserr.marketplace.institutionalpage.dm.repository.InstitutionalPageRepository;
import lombok.RequiredArgsConstructor;
import org.jetbrains.annotations.NotNull;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;

import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.UUID;
import java.util.function.Supplier;
import java.util.stream.Collectors;

/**
 * Helper component that encapsulates common logic for retrieving, saving, and deleting
 * {@link InstitutionalPageEntity} instances with various filtering and moderation criteria.
 *
 * <p>
 * Provides utility methods to navigate hierarchical relationships, handle moderation workflows,
 * resolve original and root versions, and ensure data consistency through validation and exception handling.
 * </p>
 */
@Component
@RequiredArgsConstructor
public class InstitutionalPageHelper {

    private final InstitutionalPageRepository institutionalPageRepository;
    private final MemberRequestHelper memberRequestHelper;

    //

    public boolean isRoot(InstitutionalPageEntity institutionalPageEntity) {
        return institutionalPageEntity.getParentInstitutionalPageId() == null;
    }

    // save and delete

    public InstitutionalPageEntity save(UUID userId, InstitutionalPageEntity institutionalPageEntity) {
        institutionalPageEntity.setLastModifiedBy(userId);
        return institutionalPageRepository.saveAndFlush(institutionalPageEntity);
    }

    public void saveAll(UUID userId, List<InstitutionalPageEntity> institutionalPageEntities) {
        institutionalPageEntities.forEach(institutionalPageEntity ->
                institutionalPageEntity.setLastModifiedBy(userId)
        );
        institutionalPageRepository.saveAllAndFlush(institutionalPageEntities);
    }

    public void delete(InstitutionalPageEntity institutionalPageEntity) {
        memberRequestHelper.deleteMemberRequestsForInstitutionalPage(institutionalPageEntity.getId());
        institutionalPageRepository.delete(institutionalPageEntity);
    }

    // retrieve by id

    public InstitutionalPageEntity retrieveInstitutionalPageOrThrow(
            UUID institutionalPageId,
            Supplier<? extends RuntimeException> exceptionSupplier
    ) {
        return institutionalPageRepository.findById(institutionalPageId).orElseThrow(exceptionSupplier);
    }

    public InstitutionalPageEntity retrieveInstitutionalPage(UUID institutionalPageId) {
        return retrieveInstitutionalPageOrThrow(
                institutionalPageId,
                () -> new InstitutionalPageNotFoundException(institutionalPageId)
        );
    }

    // latest by id

    public InstitutionalPageEntity retrieveLatestById(UUID institutionalPageId) {
        SearchInstitutionalPageModel searchFilters = SearchInstitutionalPageModel.builder()
                .ids(List.of(institutionalPageId))
                .build();
        List<InstitutionalPageEntity> institutionalPages = retrieveAllWithFilter(
                PageRequest.of(0, 2),
                searchFilters,
                null,
                false
        ).getContent();
        if (institutionalPages.isEmpty()) {
            throw new InstitutionalPageNotFoundException(institutionalPageId);
        } else if (institutionalPages.size() == 1) {
            return institutionalPages.getFirst();
        } else {
            return institutionalPages.stream()
                    .filter(institutionalPage -> institutionalPage.getOriginalInstitutionalPageId() != null)
                    .findFirst()
                    .orElseThrow(() -> foundNotRelatedIPsError(institutionalPageId, institutionalPages));
        }
    }

    // original and latest for member by id

    public InstitutionalPageEntity retrieveLatestForMemberById(UUID userId, UUID institutionalPageId) {
        SearchForMemberInstitutionalPageModel searchFilters = SearchForMemberInstitutionalPageModel.builder()
                .ids(List.of(institutionalPageId))
                .build();
        List<InstitutionalPageEntity> institutionalPages = retrieveAllForMemberWithFilter(
                userId,
                PageRequest.of(0, 2),
                searchFilters,
                null,
                false
        ).getContent();
        if (institutionalPages.isEmpty()) {
            throw new InstitutionalPageNotFoundException(institutionalPageId);
        } else if (institutionalPages.size() == 1) {
            return institutionalPages.getFirst();
        } else {
            return institutionalPages.stream()
                    .filter(institutionalPage -> institutionalPage.getOriginalInstitutionalPageId() != null)
                    .findFirst()
                    .orElseThrow(() -> foundNotRelatedIPsError(institutionalPageId, institutionalPages));
        }
    }

    public InstitutionalPageEntity retrieveOriginalForMemberById(UUID userId, UUID institutionalPageId) {
        SearchForMemberInstitutionalPageModel searchFilters = SearchForMemberInstitutionalPageModel.builder()
                .ids(List.of(institutionalPageId))
                .build();
        return retrieveAllForMemberWithFilter(
                userId,
                PageRequest.of(0, 1),
                searchFilters,
                null,
                true
        )
                .stream()
                .findFirst()
                .orElseThrow(() -> new InstitutionalPageNotFoundException(institutionalPageId));
    }

    // approved for member

    public Page<InstitutionalPageEntity> retrieveAllApprovedForMember(UUID userId, Pageable pageable) {
        SearchForMemberInstitutionalPageModel searchFilters = SearchForMemberInstitutionalPageModel.builder().build();
        return retrieveAllApprovedForMemberWithFilter(userId, pageable, searchFilters);
    }

    public Page<InstitutionalPageEntity> retrieveAllApprovedForMemberWithFilter(
            UUID userId,
            Pageable pageable,
            SearchForMemberInstitutionalPageModel searchFilters
    ) {
        return retrieveAllForMemberWithFilter(
                userId,
                pageable,
                searchFilters,
                List.of(ModerationStatus.APPROVED),
                true
        );
    }

    public InstitutionalPageEntity retrieveApprovedForMemberById(UUID userId, UUID institutionalPageId) {
        SearchForMemberInstitutionalPageModel searchFilters = SearchForMemberInstitutionalPageModel.builder()
                .includePublishedAndNotMember(true) // to let the user see details of published IPs, even if not member
                .ids(List.of(institutionalPageId))
                .build();
        return retrieveAllApprovedForMemberWithFilter(userId, PageRequest.of(0, 1), searchFilters)
                .stream()
                .findFirst()
                .orElseThrow(() -> new InstitutionalPageNotFoundException(institutionalPageId));
    }

    public Page<InstitutionalPageEntity> retrieveAllApprovedHierarchyForMemberByRootId(
            UUID userId,
            Pageable pageable,
            UUID rootId
    ) {
        return institutionalPageRepository.findHierarchyForMemberFiltered(
                pageable,
                userId,
                rootId,
                true,
                List.of(ModerationStatus.APPROVED.getLabel()),
                true
        );
    }

    public Page<InstitutionalPageEntity> retrieveAllOriginalInHierarchyByRootId(
            Pageable pageable,
            UUID rootId
    ) {
        return institutionalPageRepository.findHierarchyFiltered(
                pageable,
                rootId,
                false,
                null,
                true
        );
    }

    // pending for member

    public Page<InstitutionalPageEntity> retrieveAllPendingForMember(UUID userId, Pageable pageable) {
        SearchForMemberInstitutionalPageModel searchFilters = SearchForMemberInstitutionalPageModel.builder().build();
        return retrieveAllPendingForMemberWithFilter(userId, pageable, searchFilters, null);
    }

    public Page<InstitutionalPageEntity> retrieveAllPendingForMemberWithFilter(
            UUID userId,
            Pageable pageable,
            SearchForMemberInstitutionalPageModel searchFilters,
            String moderationStatus
    ) {
        List<ModerationStatus> moderationStatuses = moderationStatus != null
                ? List.of(ModerationStatus.ofLabel(moderationStatus))
                : List.of(ModerationStatus.PENDING, ModerationStatus.REJECTED);
        return retrieveAllForMemberWithFilter(
                userId,
                pageable,
                searchFilters,
                moderationStatuses,
                false
        );
    }

    public InstitutionalPageEntity retrievePendingForMemberById(UUID userId, UUID institutionalPageId) {
        SearchForMemberInstitutionalPageModel searchFilters = SearchForMemberInstitutionalPageModel.builder()
                .ids(List.of(institutionalPageId))
                .build();
        return retrieveAllPendingForMemberWithFilter(userId, PageRequest.of(0, 1), searchFilters, null)
                .stream()
                .findFirst()
                .orElseThrow(() -> new InstitutionalPageNotFoundException(institutionalPageId));
    }

    // all pending

    public Page<InstitutionalPageEntity> retrieveAllPending(Pageable pageable) {
        SearchInstitutionalPageModel searchFilters = SearchInstitutionalPageModel.builder().build();
        return retrieveAllPendingWithFilter(pageable, searchFilters, null);
    }

    public Page<InstitutionalPageEntity> retrieveAllPendingWithFilter(
            Pageable pageable,
            SearchInstitutionalPageModel searchFilters,
            String moderationStatus
    ) {
        List<ModerationStatus> moderationStatuses = moderationStatus != null
                ? List.of(ModerationStatus.ofLabel(moderationStatus))
                : List.of(ModerationStatus.PENDING, ModerationStatus.REJECTED);
        return retrieveAllWithFilter(
                pageable,
                searchFilters,
                moderationStatuses,
                false
        );
    }

    public InstitutionalPageEntity retrievePendingById(UUID institutionalPageId) {
        SearchInstitutionalPageModel searchFilters = SearchInstitutionalPageModel.builder()
                .ids(List.of(institutionalPageId))
                .build();
        return retrieveAllPendingWithFilter(PageRequest.of(0, 1), searchFilters, null)
                .stream()
                .findFirst()
                .orElseThrow(() -> new InstitutionalPageNotFoundException(institutionalPageId));
    }

    // published and approved

    public Page<InstitutionalPageEntity> retrieveAllPublished(Pageable pageable) {
        SearchInstitutionalPageModel searchFilters = SearchInstitutionalPageModel.builder()
                .published(true)
                .build();
        return retrieveAllApprovedWithFilter(pageable, searchFilters);
    }

    public InstitutionalPageEntity retrievePublishedById(UUID institutionalPageId) {
        SearchInstitutionalPageModel searchFilters = SearchInstitutionalPageModel.builder()
                .ids(List.of(institutionalPageId))
                .published(true)
                .build();
        return retrieveAllApprovedWithFilter(PageRequest.of(0, 1), searchFilters)
                .stream()
                .findFirst()
                .orElseThrow(() -> new InstitutionalPageNotFoundException(institutionalPageId));
    }

    public List<InstitutionalPageEntity> retrieveAllApprovedByIds(
            List<UUID> institutionalPageIds
    ) {
        SearchInstitutionalPageModel searchFilters = SearchInstitutionalPageModel.builder()
                .ids(institutionalPageIds)
                .build();
        return retrieveAllApprovedWithFilter(Pageable.unpaged(), searchFilters).getContent();
    }

    public InstitutionalPageEntity retrieveApprovedById(UUID institutionalPageId) {
        SearchInstitutionalPageModel searchFilters = SearchInstitutionalPageModel.builder()
                .ids(List.of(institutionalPageId))
                .build();
        return retrieveAllApprovedWithFilter(PageRequest.of(0, 1), searchFilters)
                .stream()
                .findFirst()
                .orElseThrow(() -> new InstitutionalPageNotFoundException(institutionalPageId));
    }

    public Page<InstitutionalPageEntity> retrieveAllApprovedWithFilter(
            Pageable pageable,
            SearchInstitutionalPageModel searchFilters
    ) {
        return retrieveAllWithFilter(
                pageable,
                searchFilters,
                List.of(ModerationStatus.APPROVED),
                true
        );
    }

    // navigate relations
    // always retrieve approved version

    public UUID retrieveRootInstitutionalPageId(UUID institutionalPageId) {
        InstitutionalPageEntity institutionalPage = retrieveInstitutionalPage(institutionalPageId);
        return getRootInstitutionalPageId(institutionalPage);
    }

    public InstitutionalPageEntity retrieveRootForMemberById(UUID userId, UUID institutionalPageId) {
        InstitutionalPageEntity institutionalPage = retrieveOriginalForMemberById(userId, institutionalPageId);
        return retrieveRootInstitutionalPage(institutionalPage);
    }

    public List<InstitutionalPageEntity> retrieveOriginalRootsInstitutionalPages(List<InstitutionalPageEntity> institutionalPages) {
        List<UUID> institutionalPageIds = new LinkedList<>();
        List<UUID> rootInstitutionalPageIds = new LinkedList<>();
        // map containing institutional pages received as input, which are already root
        Map<UUID, InstitutionalPageEntity> existingRootInstitutionalPages = new HashMap<>();
        for (InstitutionalPageEntity institutionalPage : institutionalPages) {
            UUID institutionalPageId = institutionalPage.getId();
            UUID rootId = InstitutionalPageHelper.getRootInstitutionalPageId(institutionalPage);
            // avoid loading again the same ips
            if (rootId.equals(institutionalPageId) || existingRootInstitutionalPages.containsKey(institutionalPageId)) {
                existingRootInstitutionalPages.put(institutionalPageId, institutionalPage);
            } else {
                institutionalPageIds.add(institutionalPageId);
                rootInstitutionalPageIds.add(rootId);
            }
        }
        if (rootInstitutionalPageIds.isEmpty()) {
            return new LinkedList<>(existingRootInstitutionalPages.values());
        }
        List<InstitutionalPageEntity> foundRoots = new LinkedList<>(retrieveAllApprovedByIds(rootInstitutionalPageIds));
        verifyAllRootFoundOrThrow(institutionalPageIds, foundRoots, rootInstitutionalPageIds);
        // add the root IPs already received as input
        foundRoots.addAll(existingRootInstitutionalPages.values());
        return foundRoots;
    }

    public List<InstitutionalPageEntity> retrieveApprovedParentsInstitutionalPages(List<InstitutionalPageEntity> institutionalPages) {
        List<UUID> institutionalPageIds = new LinkedList<>();
        List<UUID> parentInstitutionalPageIds = new LinkedList<>();
        for (InstitutionalPageEntity institutionalPage : institutionalPages) {
            institutionalPageIds.add(institutionalPage.getId());
            Optional.ofNullable(institutionalPage.getParentInstitutionalPageId())
                    .ifPresent(parentId -> parentInstitutionalPageIds.add(parentId));
        }
        if (parentInstitutionalPageIds.isEmpty()) {
            return new LinkedList<>();
        }
        List<InstitutionalPageEntity> foundParents = retrieveAllApprovedByIds(parentInstitutionalPageIds);
        verifyAllParentFoundOrThrow(institutionalPageIds, foundParents, parentInstitutionalPageIds);
        return foundParents;
    }

    public List<InstitutionalPageEntity> retrieveApprovedChildInstitutionalPages(InstitutionalPageEntity institutionalPage) {
        return retrieveApprovedChildInstitutionalPages(List.of(institutionalPage));
    }

    public List<InstitutionalPageEntity> retrieveApprovedChildInstitutionalPages(List<InstitutionalPageEntity> institutionalPages) {
        List<UUID> originalInstitutionalPageIds = institutionalPages.stream()
                .map(institutionalPage -> getOriginalInstitutionalPageId(institutionalPage))
                .toList();
        return institutionalPageRepository.findAllByParentInstitutionalPageIdInAndModerationStatusIn(
                originalInstitutionalPageIds,
                List.of(ModerationStatus.APPROVED)
        );
    }

    public List<InstitutionalPageEntity> retrieveOriginalChildInstitutionalPages(
            InstitutionalPageEntity institutionalPage
    ) {
        UUID originalInstitutionalPageId = getOriginalInstitutionalPageId(institutionalPage);
        return institutionalPageRepository
                .findAllByParentInstitutionalPageIdAndOriginalInstitutionalPageIdIsNull(originalInstitutionalPageId);
    }

    public InstitutionalPageEntity retrieveOriginalInstitutionalPage(InstitutionalPageEntity institutionalPage) {
        InstitutionalPageEntity originalInstitutionalPage;
        if (institutionalPage.getOriginalInstitutionalPageId() != null) {
            originalInstitutionalPage = retrieveInstitutionalPageOrThrow(
                    institutionalPage.getOriginalInstitutionalPageId(),
                    () -> new WP2BusinessException(getOriginalIPNotFoundErrorMessage(
                            institutionalPage.getId(),
                            institutionalPage.getOriginalInstitutionalPageId()
                    ))
            );
        } else {
            originalInstitutionalPage = institutionalPage;
        }
        return originalInstitutionalPage;
    }

    public InstitutionalPageEntity retrieveRootInstitutionalPage(InstitutionalPageEntity institutionalPage) {
        InstitutionalPageEntity rootInstitutionalPage;
        if (!CollectionUtils.isEmpty(institutionalPage.getAncestorInstitutionalPageIds())) {
            UUID rootInstitutionalPageId = institutionalPage.getAncestorInstitutionalPageIds().getFirst();
            rootInstitutionalPage = retrieveInstitutionalPageOrThrow(
                    rootInstitutionalPageId,
                    () -> new WP2BusinessException(getRootIPNotFoundErrorMessage(
                            institutionalPage.getId(),
                            rootInstitutionalPageId
                    ))
            );
        } else {
            rootInstitutionalPage = retrieveOriginalInstitutionalPage(institutionalPage);
        }
        return rootInstitutionalPage;
    }

    //

    public static UUID getOriginalInstitutionalPageId(InstitutionalPageEntity institutionalPageEntity) {
        return Optional
                .ofNullable(institutionalPageEntity.getOriginalInstitutionalPageId())
                .orElse(institutionalPageEntity.getId());
    }

    public static UUID getRootInstitutionalPageId(InstitutionalPageEntity institutionalPageEntity) {
        UUID rootInstitutionalPageId;
        if (!CollectionUtils.isEmpty(institutionalPageEntity.getAncestorInstitutionalPageIds())) {
            rootInstitutionalPageId = institutionalPageEntity.getAncestorInstitutionalPageIds().getFirst();
        } else {
            rootInstitutionalPageId = getOriginalInstitutionalPageId(institutionalPageEntity);
        }
        return rootInstitutionalPageId;
    }

    // private

    private Page<InstitutionalPageEntity> retrieveAllWithFilter(
            Pageable pageable,
            SearchInstitutionalPageModel searchFilters,
            List<ModerationStatus> moderationStatuses,
            Boolean originalVersionOnly
    ) {
        boolean applyModerationStatusesFilter = !CollectionUtils.isEmpty(moderationStatuses);
        List<String> moderationStatusLabels = applyModerationStatusesFilter
                ? toStringList(moderationStatuses)
                : List.of("__IGNORED__");
        boolean applyInstitutionalPageIdsFilter = !CollectionUtils.isEmpty(searchFilters.getIds());
        List<UUID> institutionalPageIds = applyInstitutionalPageIdsFilter
                ? searchFilters.getIds()
                : List.of(UUID.fromString("00000000-0000-0000-0000-000000000000"));
        String searchText = Optional.ofNullable(searchFilters.getSearchText())
                .map(String::toLowerCase)
                .orElse(null);

        return institutionalPageRepository.findAllFiltered(
                pageable,
                applyModerationStatusesFilter,
                moderationStatusLabels,
                applyInstitutionalPageIdsFilter,
                institutionalPageIds,
                searchFilters.getPublished(),
                searchFilters.getCategory(),
                searchFilters.getRootLevelOnly() != null && searchFilters.getRootLevelOnly(),
                originalVersionOnly != null && originalVersionOnly,
                searchText
        );
    }

    private Page<InstitutionalPageEntity> retrieveAllForMemberWithFilter(
            UUID userId,
            Pageable pageable,
            SearchForMemberInstitutionalPageModel searchFilters,
            List<ModerationStatus> moderationStatuses,
            Boolean originalVersionOnly
    ) {
        boolean applyModerationStatusesFilter = !CollectionUtils.isEmpty(moderationStatuses);
        List<String> moderationStatusLabels = applyModerationStatusesFilter
                ? toStringList(moderationStatuses)
                : List.of("__IGNORED__");
        boolean applyInstitutionalPageIdsFilter = !CollectionUtils.isEmpty(searchFilters.getIds());
        List<UUID> institutionalPageIds = applyInstitutionalPageIdsFilter
                ? searchFilters.getIds()
                : List.of(UUID.fromString("00000000-0000-0000-0000-000000000000"));
        String searchText = Optional.ofNullable(searchFilters.getSearchText())
                .map(String::toLowerCase)
                .orElse(null);
        return institutionalPageRepository.findAllForMemberFiltered(
                pageable,
                userId,
                searchFilters.getWpLeaderOnly() != null && searchFilters.getWpLeaderOnly(),
                applyModerationStatusesFilter,
                moderationStatusLabels,
                applyInstitutionalPageIdsFilter,
                institutionalPageIds,
                searchFilters.getIncludePrivateAndMember() == null || searchFilters.getIncludePrivateAndMember(),
                searchFilters.getIncludePublishedAndMember() == null || searchFilters.getIncludePublishedAndMember(),
                searchFilters.getIncludePublishedAndNotMember() != null && searchFilters.getIncludePublishedAndNotMember(),
                searchFilters.getCategory(),
                searchFilters.getRootLevelOnly() != null && searchFilters.getRootLevelOnly(),
                originalVersionOnly != null && originalVersionOnly,
                searchText
        );
    }

    //

    private static void verifyAllRootFoundOrThrow(
            List<UUID> institutionalPageIds,
            List<InstitutionalPageEntity> foundRoots,
            List<UUID> requestedRootIds
    ) {
        Set<UUID> foundRootsIds = foundRoots.stream()
                .map(InstitutionalPageEntity::getId)
                .collect(Collectors.toSet());
        Set<UUID> notFoundRootIds = new HashSet<>(requestedRootIds);
        notFoundRootIds.removeAll(foundRootsIds);
        if (!notFoundRootIds.isEmpty()) {
            throw new WP2BusinessException(getRootsErrorMessage(institutionalPageIds, notFoundRootIds));
        }
    }

    private static void verifyAllParentFoundOrThrow(
            List<UUID> institutionalPageIds,
            List<InstitutionalPageEntity> foundParents,
            List<UUID> requestedParentIds
    ) {
        Set<UUID> foundParentsIds = foundParents.stream()
                .map(InstitutionalPageEntity::getId)
                .collect(Collectors.toSet());
        Set<UUID> notFoundParentIds = new HashSet<>(requestedParentIds);
        notFoundParentIds.removeAll(foundParentsIds);
        if (!notFoundParentIds.isEmpty()) {
            throw new WP2BusinessException(getParentsErrorMessage(institutionalPageIds, notFoundParentIds));
        }
    }

    // errors

    private static WP2BusinessException foundNotRelatedIPsError(
            UUID institutionalPageId,
            List<InstitutionalPageEntity> institutionalPages
    ) {
        List<UUID> institutionalPageIds = institutionalPages
                .stream()
                .map(institutionalPageEntity -> institutionalPageEntity.getId())
                .toList();
        return new WP2BusinessException(
                "found more institutional pages for same id '%s' not related to each other, ids: '%s'"
                        .formatted(institutionalPageId, institutionalPageIds));
    }

    private static String getRootIPNotFoundErrorMessage(
            UUID institutionalPageId,
            UUID rootInstitutionalPageId
    ) {
        return "InstitutionalPage with id: '%s', rootInstitutionalPage with id: '%s' not found"
                .formatted(institutionalPageId, rootInstitutionalPageId);
    }

    private static String getRootsErrorMessage(List<UUID> institutionalPageIds, Set<UUID> rootInstitutionalPageIds) {
        return "InstitutionalPage with ids: '%s', some rootInstitutionalPage with ids: '%s' not found"
                .formatted(institutionalPageIds, rootInstitutionalPageIds);
    }

    private static String getParentsErrorMessage(List<UUID> institutionalPageIds, Set<UUID> parentInstitutionalPageIds) {
        return "InstitutionalPage with ids: '%s', some parentInstitutionalPage with ids: '%s' not found"
                .formatted(institutionalPageIds, parentInstitutionalPageIds);
    }

    private static String getOriginalIPNotFoundErrorMessage(
            UUID institutionalPageId,
            UUID originalInstitutionalPageId
    ) {
        return "InstitutionalPage with id: '%s', originalInstitutionalPage with id: '%s' not found"
                .formatted(institutionalPageId, originalInstitutionalPageId);
    }

    //

    private static @NotNull List<String> toStringList(List<ModerationStatus> moderationStatuses) {
        return moderationStatuses.stream()
                .map(moderationStatus -> moderationStatus.getLabel())
                .toList();
    }

}
