package com.finconsgroup.itserr.marketplace.institutionalpage.dm.entity;

import com.finconsgroup.itserr.marketplace.core.entity.AbstractUUIDEntity;
import com.finconsgroup.itserr.marketplace.institutionalpage.dm.enums.ModerationStatus;
import com.finconsgroup.itserr.marketplace.institutionalpage.dm.model.ButtonModel;
import jakarta.persistence.CascadeType;
import jakarta.persistence.Column;
import jakarta.persistence.Entity;
import jakarta.persistence.FetchType;
import jakarta.persistence.OneToMany;
import jakarta.persistence.OrderColumn;
import jakarta.persistence.PrePersist;
import jakarta.persistence.Table;
import lombok.Data;
import lombok.EqualsAndHashCode;
import lombok.NoArgsConstructor;
import lombok.ToString;
import lombok.experimental.SuperBuilder;
import org.hibernate.annotations.JdbcTypeCode;
import org.hibernate.type.SqlTypes;

import java.time.Instant;
import java.time.temporal.ChronoUnit;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.UUID;

import static com.finconsgroup.itserr.marketplace.institutionalpage.dm.util.Constants.MAX_LONG_TEXT_LENGTH;

/**
 * This entity represents a soft deleted {@link InstitutionalPageEntity}.
 */
@Entity
@Table(name = "archived_institutional_page")
@Data
@SuperBuilder
@EqualsAndHashCode(callSuper = true)
@NoArgsConstructor
public class ArchivedInstitutionalPageEntity extends AbstractUUIDEntity {

    /**
     * The name of the institutional page.
     */
    @Column(name = "name", nullable = false, updatable = false)
    private String name;

    /**
     * ID of the parent institutional page.
     */
    @Column(name = "parent_institutional_page_id")
    private UUID parentInstitutionalPageId;

    /**
     * list of institutional page ancestors.
     */
    @Column(name = "ancestor_institutional_page_ids", updatable = false)
    private List<UUID> ancestorInstitutionalPageIds;

    /**
     * ID of the workspace folder related to the institutional page.
     */
    @Column(name = "workspace_folder_id", nullable = false, updatable = false)
    private UUID workspaceFolderId;

    /**
     * The members (wp leads included) related to the institutional page.
     */
    @OneToMany(fetch = FetchType.LAZY, cascade = CascadeType.ALL, orphanRemoval = true, mappedBy = "institutionalPage")
    @OrderColumn(name = "member_order")
    @ToString.Exclude
    private List<ArchivedMemberEntity> members;

    /**
     * The paragraphs related to the institutional page.
     */
    @OneToMany(fetch = FetchType.LAZY, cascade = CascadeType.ALL, orphanRemoval = true, mappedBy = "institutionalPage")
    @OrderColumn(name = "paragraph_order")
    @ToString.Exclude
    private List<ArchivedParagraphEntity> paragraphs;

    /**
     * The category of the institutional page (e.g. tool, project).
     */
    @Column(name = "category", nullable = false, updatable = false)
    private String category;

    /**
     * Indicates whether the institutional page is published
     */
    @Column(name = "published", nullable = false)
    private boolean published;

    /**
     * Indicates whether the institutional page should be deleted
     */
    @Column(name = "to_delete", nullable = false)
    private boolean toDelete;

    /**
     * The moderation status of the institutional page.
     */
    @Column(name = "moderation_status", nullable = false, updatable = false)
    private ModerationStatus moderationStatus;

    /**
     * The publication time of the institutional page.
     */
    @Column(name = "publication_time", updatable = false)
    private Instant publicationTime;

    /**
     * Message related to rejection of publish process.
     */
    @Column(name = "rejection_message", updatable = false)
    private String rejectionMessage;

    /**
     * ID of the original institutional page version.
     */
    @Column(name = "original_institutional_page_id")
    private UUID originalInstitutionalPageId;

    /**
     * ID of the updated institutional page version.
     */
    @Column(name = "updated_institutional_page_id")
    private UUID updatedInstitutionalPageId;

    /**
     * The abstract of the institutional page.
     */
    @Column(name = "abstract_content", columnDefinition = "text", nullable = false, updatable = false, length = MAX_LONG_TEXT_LENGTH)
    private String abstractContent;

    /**
     * Copyright text to display.
     */
    @Column(name = "copyright_text", nullable = false, updatable = false)
    private String copyrightText;

    /**
     * Copyright year.
     */
    @Column(name = "copyright_year", nullable = false, updatable = false)
    private int copyrightYear;

    /**
     * ID of the maintainer of the page.
     */
    @Column(name = "maintainer")
    private UUID maintainer;

    /**
     * list of tags.
     */
    @Column(name = "tags")
    private List<String> tags;

    /**
     * The image URL.
     */
    @Column(name = "image_url")
    private String imageUrl;

    /**
     * The user manual URL.
     */
    @Column(name = "user_manual_url")
    private String userManualUrl;

    /**
     * The FAQs page URL.
     */
    @Column(name = "faqs_url")
    private String faqsUrl;

    /**
     * The application page version.
     */
    @Column(name = "application_version")
    private String applicationVersion;

    /**
     * The application type (e.g. Desktop Application).
     */
    @Column(name = "application_type")
    private String applicationType;

    /**
     * The License name.
     */
    @Column(name = "license")
    private String license;

    /**
     * The supported macOS version or description.
     */
    @Column(name = "supported_os_mac_os")
    private String supportedOsMac;

    /**
     * The supported Linux version or description.
     */
    @Column(name = "supported_os_linux")
    private String supportedOsLinux;

    /**
     * The supported Windows version or description.
     */
    @Column(name = "supported_os_windows")
    private String supportedOsWindows;

    /**
     * URL of the source repository.
     */
    @Column(name = "repo_name")
    private String repoName;

    /**
     * URL of the website.
     */
    @Column(name = "website")
    private String website;

    /**
     * The JSON structure for extra fields.
     */
    @JdbcTypeCode(SqlTypes.JSON)
    @Column(name = "additional_fields", columnDefinition = "json")
    private LinkedHashMap<String, String> additionalFields;

    /**
     * The JSON structure for external links.
     */
    @JdbcTypeCode(SqlTypes.JSON)
    @Column(name = "external_links", columnDefinition = "json")
    private LinkedHashMap<String, String> externalLinks;

    /**
     * The JSON structure for buttons.
     */
    @JdbcTypeCode(SqlTypes.JSON)
    @Column(name = "buttons", columnDefinition = "json")
    private List<ButtonModel> buttons;

    /**
     * ID of the user who is currently updating the entity.
     */
    @Column(name = "update_locked_by")
    private UUID updateLockedBy;

    /**
     * The ID of the user who last modified the entity.
     */
    @Column(name = "last_modified_by", nullable = false)
    private UUID lastModifiedBy;

    /**
     * The timestamp when the institutionalPage was created
     * in the original (non-archived) table.
     */
    @Column(name = "creation_time", nullable = false, updatable = false)
    private Instant creationTime;

    /**
     * The timestamp when the institutionalPage was last updated
     * in the original (non-archived) table.
     */
    @Column(name = "update_time", nullable = false, updatable = false)
    private Instant updateTime;

    /**
     * The timestamp when this record was inserted into the archive table,
     * representing the moment the institutionalPage was archived.
     */
    @Column(name = "archival_time", nullable = false, updatable = false)
    private Instant archivalTime;

    /**
     * Sets {@code archivalTime} before the entity is persisted.
     * <p>
     * The timestamp is stored in UTC to ensure consistent and timezone-safe timestamps.
     * The result is truncated to microseconds to match PostgreSQL's default precision
     * for {@code TIMESTAMPTZ} columns (6 digits).
     * Avoids using Hibernate's {@code @CreationTimestamp} to ensure timestamps
     * are immediately available after {@code JpaRepository.save()},
     * without requiring an explicit {@code JpaRepository.flush()}.
     * </p>
     */
    @PrePersist
    public void onCreate() {
        archivalTime = Instant.now().truncatedTo(ChronoUnit.MICROS);
    }

}
