package com.finconsgroup.itserr.marketplace.institutionalpage.dm.service;

import com.finconsgroup.itserr.marketplace.institutionalpage.dm.dto.*;
import com.finconsgroup.itserr.marketplace.institutionalpage.dm.exception.InstitutionalPageNotFoundException;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.lang.NonNull;

import java.util.Set;
import java.util.UUID;

/**
 * Service for handling business logic related to InstitutionalPages.
 * This interface provides methods to retrieve, create, update, and delete InstitutionalPages,
 * as well as manage membership, paragraphs, and publication workflow.
 *
 * <p>Example usage:
 * <pre>
 * Page<OutputInstitutionalPageDto> institutionalPages = institutionalPageService.
 *          findAllInstitutionalPages(userId, associationsToLoad, pageable);
 * </pre>
 * </p>
 */
public interface InstitutionalPageService {

    /**
     * Creates a new InstitutionalPage.
     *
     * @param userId                          the unique identifier of the authenticated user
     * @param inputCreateInstitutionalPageDto the input DTO containing InstitutionalPage details
     * @return a {@link OutputInstitutionalPageDto} representing the created InstitutionalPage
     * @throws InstitutionalPageNotFoundException if the specified parent InstitutionalPage does not exist
     */
    @NonNull
    OutputInstitutionalPageDto createInstitutionalPage(
            @NonNull UUID userId,
            @NonNull InputCreateInstitutionalPageDto inputCreateInstitutionalPageDto
    );

    /**
     * Retrieves a paginated list of all InstitutionalPages the user is contributing to.
     *
     * @param userId             the unique identifier of the authenticated user
     * @param associationsToLoad the associations to be returned on the response
     * @param pageable           the pagination information
     * @return a page of {@link OutputInstitutionalPageDto} representing the InstitutionalPages
     */
    @NonNull
    Page<OutputInstitutionalPageDto> findAllInstitutionalPages(
            @NonNull UUID userId,
            @NonNull Set<String> associationsToLoad,
            @NonNull Pageable pageable
    );

    /**
     * Retrieves a paginated list of all InstitutionalPages the user is contributing to, matching the search criteria.
     *
     * @param userId                                   the unique identifier of the authenticated user
     * @param associationsToLoad                       the associations to be returned on the response
     * @param pageable                                 the pagination information
     * @param inputSearchForMemberInstitutionalPageDto the input DTO containing search criteria
     * @return a page of {@link OutputInstitutionalPageDto} representing the InstitutionalPages
     */
    @NonNull
    Page<OutputInstitutionalPageDto> searchInstitutionalPages(
            @NonNull UUID userId,
            @NonNull Set<String> associationsToLoad,
            @NonNull Pageable pageable,
            @NonNull InputSearchForMemberInstitutionalPageDto inputSearchForMemberInstitutionalPageDto
    );

    /**
     * Retrieves a specific InstitutionalPage by id, only if the user is a member.
     *
     * @param userId              the unique identifier of the authenticated user
     * @param institutionalPageId the id of the InstitutionalPage to retrieve
     * @return a {@link OutputInstitutionalPageDto} representing the InstitutionalPage
     * @throws InstitutionalPageNotFoundException if no InstitutionalPage is found with the given id
     */
    @NonNull
    OutputInstitutionalPageDto findInstitutionalPageById(
            @NonNull UUID userId,
            @NonNull UUID institutionalPageId
    );

    /**
     * Updates a specific InstitutionalPage by id, only if the user is a member.
     *
     * @param userId                          the unique identifier of the authenticated user
     * @param institutionalPageId             the id of the InstitutionalPage to update
     * @param inputUpdateInstitutionalPageDto the input DTO containing updated InstitutionalPage details
     * @return a {@link OutputInstitutionalPageDto} representing the updated InstitutionalPage
     * @throws InstitutionalPageNotFoundException if no InstitutionalPage is found with the given id
     */
    @NonNull
    OutputInstitutionalPageDto updateInstitutionalPageById(
            @NonNull UUID userId,
            @NonNull UUID institutionalPageId,
            @NonNull InputUpdateInstitutionalPageDto inputUpdateInstitutionalPageDto
    );

    /**
     * Deletes a specific InstitutionalPage by id, only if the user is a member.
     *
     * @param userId              the unique identifier of the authenticated user
     * @param institutionalPageId the id of the InstitutionalPage to delete
     * @return a {@link OutputInstitutionalPageDto} representing the updated InstitutionalPage
     * @throws InstitutionalPageNotFoundException if no InstitutionalPage is found with the given id
     */
    @NonNull
    OutputInstitutionalPageDto deleteInstitutionalPageById(
            @NonNull UUID userId,
            @NonNull UUID institutionalPageId
    );

    /**
     * Retrieves hierarchy of InstitutionalPages by rootInstitutionalPage id, only if the user is a member.
     *
     * @param userId                  the unique identifier of the authenticated user
     * @param associationsToLoad      the associations to be returned on the response
     * @param pageable                the pagination information
     * @param rootInstitutionalPageId the id of the rootInstitutionalPage
     * @return a page of {@link OutputInstitutionalPageDto} representing the hierarchy of InstitutionalPages
     * @throws InstitutionalPageNotFoundException if no InstitutionalPage is found with the given id
     */
    @NonNull
    Page<OutputInstitutionalPageDto> findInstitutionalPagesHierarchyByRootId(
            @NonNull UUID userId,
            @NonNull Set<String> associationsToLoad,
            @NonNull Pageable pageable,
            @NonNull UUID rootInstitutionalPageId
    );

    /**
     * Updates the folder ID associated with an institutional page, only if the user is a member.
     * Normally, the folder ID should never change, but this endpoint is required to handle
     * a specific d4science behavior where the workspace folder ID is updated after a folder is unshared.
     *
     * @param userId              the unique identifier of the authenticated user
     * @param institutionalPageId the id of the InstitutionalPage to update
     * @param newFolderId         the new id of the folder
     * @return a {@link OutputInstitutionalPageDto} representing the updated InstitutionalPage
     * @throws InstitutionalPageNotFoundException if no InstitutionalPage is found with the given id
     */
    @NonNull
    OutputInstitutionalPageDto updateFolderId(
            @NonNull UUID userId,
            @NonNull UUID institutionalPageId,
            @NonNull UUID newFolderId
    );

    /**
     * Retrieves a paginated list of all Published InstitutionalPages.
     *
     * @param associationsToLoad the associations to be returned on the response
     * @param pageable           the pagination information
     * @return a page of {@link OutputInstitutionalPageDto} representing Published InstitutionalPages
     */
    @NonNull
    Page<OutputInstitutionalPageDto> findAllPublishedInstitutionalPages(
            @NonNull Set<String> associationsToLoad,
            @NonNull Pageable pageable
    );

    /**
     * Retrieves a paginated list of all Published InstitutionalPages, matching the search criteria.
     *
     * @param associationsToLoad                       the associations to be returned on the response
     * @param pageable                                 the pagination information
     * @param inputSearchPublishedInstitutionalPageDto the DTO containing search criteria
     * @return a page of {@link OutputInstitutionalPageDto} representing Published InstitutionalPages
     */
    @NonNull
    Page<OutputInstitutionalPageDto> searchPublishedInstitutionalPages(
            @NonNull Set<String> associationsToLoad,
            @NonNull Pageable pageable,
            @NonNull InputSearchPublishedInstitutionalPageDto inputSearchPublishedInstitutionalPageDto
    );

    /**
     * Retrieves a specific Published InstitutionalPage by id.
     *
     * @param institutionalPageId the id of the published InstitutionalPage
     * @return a {@link OutputInstitutionalPageDto} representing the Published InstitutionalPage
     */
    @NonNull
    OutputInstitutionalPageDto findPublishedInstitutionalPageById(
            @NonNull UUID institutionalPageId
    );

    /**
     * Requests publish of an InstitutionalPage.
     *
     * @param userId              the unique identifier of the authenticated user
     * @param institutionalPageId the id of the InstitutionalPage to publish
     * @return a {@link OutputInstitutionalPageDto} representing the InstitutionalPage requested for Publish
     */
    @NonNull
    OutputInstitutionalPageDto requestPublishInstitutionalPage(
            @NonNull UUID userId,
            @NonNull UUID institutionalPageId
    );

    /**
     * Retrieves all latest version Pending InstitutionalPages.
     *
     * @param userId             the unique identifier of the authenticated user
     * @param associationsToLoad the associations to be returned on the response
     * @param pageable           the pagination information
     * @return a page of {@link OutputInstitutionalPageDto} representing Pending InstitutionalPages
     */
    @NonNull
    Page<OutputInstitutionalPageDto> findAllLatestVersionPendingInstitutionalPages(
            @NonNull UUID userId,
            @NonNull Set<String> associationsToLoad,
            @NonNull Pageable pageable
    );

    /**
     * Retrieves a paginated list of all latest version Pending InstitutionalPages the user is contributing to, matching the search criteria.
     *
     * @param userId                                 the unique identifier of the authenticated user
     * @param associationsToLoad                     the associations to be returned on the response
     * @param pageable                               the pagination information
     * @param inputSearchPendingInstitutionalPageDto the input DTO containing search criteria
     * @return a page of {@link OutputInstitutionalPageDto} representing the InstitutionalPages
     */
    @NonNull
    Page<OutputInstitutionalPageDto> searchLatestVersionPendingInstitutionalPages(
            @NonNull UUID userId,
            @NonNull Set<String> associationsToLoad,
            @NonNull Pageable pageable,
            @NonNull InputSearchPendingInstitutionalPageDto inputSearchPendingInstitutionalPageDto
    );

    /**
     * Retrieves a latest version Pending InstitutionalPage by id (admin).
     *
     * @param userId              the unique identifier of the authenticated user
     * @param institutionalPageId the id of the pending InstitutionalPage
     * @return a {@link OutputInstitutionalPageDto} representing the Pending InstitutionalPage
     */
    @NonNull
    OutputInstitutionalPageDto findLatestVersionPendingInstitutionalPageById(
            @NonNull UUID userId,
            @NonNull UUID institutionalPageId
    );

    /**
     * Accepts/Reject a pending InstitutionalPage (admin).
     *
     * @param userId                           the unique identifier of the authenticated user
     * @param institutionalPageId              the id of the pending InstitutionalPage
     * @param inputPatchIPModerationRequestDto the input DTO containing  accept/reject details
     * @return a {@link OutputModerationDto}   containing the operation type and institutional page
     */
    @NonNull
    OutputModerationDto acceptOrRejectPendingInstitutionalPage(
            @NonNull UUID userId,
            @NonNull UUID institutionalPageId,
            @NonNull InputPatchIPModerationRequestDto inputPatchIPModerationRequestDto
    );

    /**
     * Request update for a specific InstitutionalPage by id, only if the user is a member.
     *
     * @param userId              the unique identifier of the authenticated user
     * @param institutionalPageId the id of the pending InstitutionalPage
     * @return a {@link OutputRequestUpdateDto} indicating whether the lock for update has been obtained
     */
    @NonNull
    OutputRequestUpdateDto requestUpdateInstitutionalPageById(
            @NonNull UUID userId,
            @NonNull UUID institutionalPageId
    );

    /**
     * Release lock-for-update of a specific InstitutionalPage by id, only if the user is a member.
     *
     * @param userId              the unique identifier of the authenticated user
     * @param institutionalPageId the id of the pending InstitutionalPage
     */
    void cancelUpdateInstitutionalPageById(
            @NonNull UUID userId,
            @NonNull UUID institutionalPageId
    );

}
