package com.finconsgroup.itserr.marketplace.metadata.bs.api;

import com.finconsgroup.itserr.marketplace.metadata.bs.dto.InputCreateMetadataDto;
import com.finconsgroup.itserr.marketplace.metadata.bs.dto.OutputMetadataDto;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.media.Content;
import io.swagger.v3.oas.annotations.responses.ApiResponse;
import io.swagger.v3.oas.annotations.security.SecurityRequirement;
import io.swagger.v3.oas.annotations.tags.Tag;
import jakarta.validation.Valid;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.web.bind.annotation.DeleteMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.ResponseStatus;

import java.util.UUID;

@Tag(name = "Metadata", description = "Operations related to metadata")
@SecurityRequirement(name = "BearerAuth")
public interface MetadataApi {

    @Operation(
            summary = "Creates a new metadata",
            description = "Creates a new metadata.",
            responses = {
                    @ApiResponse(responseCode = "201", description = "Created"),
                    @ApiResponse(responseCode = "409", description = "Another metadata exists with the same name", content = @Content),
                    @ApiResponse(responseCode = "503", description = "Service unavailable", content = @Content)
            }
    )
    @ResponseStatus(HttpStatus.CREATED)
    @PostMapping(path = "/metadata", consumes = MediaType.APPLICATION_JSON_VALUE, produces = MediaType.APPLICATION_JSON_VALUE)
    OutputMetadataDto create(
            @Valid @RequestBody InputCreateMetadataDto request);

    /**
     * Deletes a metadata by id.
     * @param metadataId the id of the metadata to delete
     * @return ResponseEntity containing the OutputMetadataDto and HTTP status 200 (OK)
     */
    @Operation(
            summary = "Deletes a metadata by id",
            responses = {
                    @ApiResponse(responseCode = "200", description = "Deleted"),
                    @ApiResponse(responseCode = "404", description = "Metadata was not found", content = @Content),
                    @ApiResponse(responseCode = "503", description = "Service unavailable", content = @Content)
            }
    )
    @ResponseStatus(HttpStatus.OK)
    @DeleteMapping(value = "/metadata/{metadataId}", produces = MediaType.APPLICATION_JSON_VALUE)
    OutputMetadataDto deleteById(@PathVariable("metadataId") UUID metadataId);

}
