/**
 * Global exception handler for REST controllers in the inventory system. This class handles custom exceptions related to product and category operations and
 * returns appropriate HTTP error responses.
 *
 * <p>Example usage:
 * <pre>
 * When a ProductNotFoundException is thrown, this handler returns a 404 NOT FOUND response.
 * </pre>
 * </p>
 */
package com.finconsgroup.itserr.marketplace.metadata.bs.exception;

import com.finconsgroup.itserr.marketplace.core.web.exception.ErrorResponseDto;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.bind.annotation.RestControllerAdvice;
import org.springframework.web.context.request.WebRequest;

import static com.finconsgroup.itserr.marketplace.core.web.exception.ExceptionUtils.createResponseEntityError;
import static com.finconsgroup.itserr.marketplace.core.web.exception.ExceptionUtils.logError;
import static org.springframework.http.HttpStatus.SERVICE_UNAVAILABLE;

/**
 * Custom exception handler.
 */
@RestControllerAdvice
public class CustomExceptionHandler {

    /**
     * <p>Handles a {@link MetadataExistsException}, which means that a name collision exists with an existing metadata.</p>
     * <p>A 409 Conflict is returned.</p>
     * @param ex Thrown exception.
     * @param request HTTP request.
     * @return Error response.
     */
    @ExceptionHandler(MetadataExistsException.class)
    @ResponseBody
    public ResponseEntity<ErrorResponseDto> handleCategoryNotFoundException(
            final MetadataExistsException ex,
            final WebRequest request) {
        logError(ex, request);
        return createResponseEntityError(HttpStatus.CONFLICT, ex.getMessage());
    }

    /**
     * Handles {@link WP2ServiceUnavailableException} exceptions, returning a 503 Service Unavailable.
     * @param ex Exception.
     * @param request HTTP request.
     * @return Response.
     */
    @ExceptionHandler(WP2ServiceUnavailableException.class)
    public ResponseEntity<ErrorResponseDto> handleWP2ServiceUnavailable(final WP2ServiceUnavailableException ex, final WebRequest request) {
        logError(ex, request);
        return createResponseEntityError(SERVICE_UNAVAILABLE, "Service unavailable");
    }

}
