package com.finconsgroup.itserr.marketplace.search.dm.util;

import io.micrometer.common.util.StringUtils;
import org.springframework.data.domain.Sort;
import org.springframework.lang.NonNull;
import org.springframework.lang.Nullable;

import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.atomic.AtomicReference;

/**
 * Utility operations for Sort related features
 */
public final class SortUtils {
    private static final String MULTI_SORT_SEPARATOR = ",";

    private SortUtils() {
        throw new UnsupportedOperationException("SortUtils cannot be instantiated");
    }

    /**
     * Calculates the sort order based on the list of sort field strings.
     * The string should either contain the field name and direction separated by semicolon(:)
     * or just the field name (in which case the default direction is considered to be ascending)
     * e.g. _source:DESC or _source
     *
     * @param sortFields      the list of sort field strings
     * @param separator       the separator for sort field and direction
     * @param propertyNameMap the map containing name that should be used for queries instead of the property name
     * @return {@link Sort} based on the sort fields and order
     */
    @NonNull
    public static Sort buildSort(@NonNull List<String> sortFields,
                                 @NonNull String separator,
                                 @Nullable Map<String, String> propertyNameMap) {
        final AtomicReference<Sort> sortResult = new AtomicReference<>(Sort.unsorted());
        sortFields.stream()
                .map(s -> s.split(separator, 2))
                .map(arr -> {
                    String propertyName = mapSortProperty(propertyNameMap, arr[0]);
                    if (arr.length == 1) {
                        return Sort.by(Sort.Direction.ASC, propertyName);
                    } else {
                        return Sort.by(Sort.Direction.fromString(arr[1]), propertyName);
                    }
                })
                .forEach(sort -> sortResult.getAndAccumulate(sort, Sort::and));
        return sortResult.get();
    }

    /**
     * Calculates the sort order based on the provided input i.e. sort, direction and multiSort.
     * If multiSort has value, then it takes preference. And it should have the field name and direction
     * combinations separated by separator e.g. semicolon(:) or just the field name (in which case the
     * default direction is considered to be ascending)
     * e.g. _source:DESC or _source
     *
     * @param sort            the sort field
     * @param direction       the direction to sort
     * @param multiSort       the sort string containing multiple fields to sort on
     * @param separator       the separator for sort field and direction
     * @param propertyNameMap the map containing name that should be used for queries instead of the property name
     * @return {@link Sort} based on the sort fields and order
     */
    @NonNull
    public static Sort buildSort(@NonNull String sort,
                                 @NonNull Sort.Direction direction,
                                 String multiSort,
                                 @NonNull String separator,
                                 @Nullable Map<String, String> propertyNameMap) {
        Sort sortResult;
        if (StringUtils.isNotBlank(multiSort)) {
            List<String> sortFields = Arrays.asList(multiSort.split(MULTI_SORT_SEPARATOR));
            sortResult = SortUtils.buildSort(sortFields, separator, propertyNameMap);
        } else {
            sortResult = Sort.by(direction, mapSortProperty(propertyNameMap, sort));
        }
        return sortResult;
    }

    /**
     * Maps the sort property name to the name that can be used to sort the documents and
     * returns the new sort.
     *
     * @param propertyMap the map containing name that should be used for queries instead of the property name
     * @param sort        the original sort
     * @return Sort with property name mapped if necessary
     */
    @NonNull
    public static Sort mapSortProperty(@Nullable Map<String, String> propertyMap, @NonNull Sort sort) {
        final AtomicReference<Sort> sortResult = new AtomicReference<>(Sort.unsorted());
        sort
                .stream().sequential()
                .map(o -> Sort.by(o.getDirection(), mapSortProperty(propertyMap, o.getProperty())))
                .forEach(newSort -> sortResult.getAndAccumulate(newSort, Sort::and));
        return sortResult.get();
    }

    /*
     * Maps the sort property name to name that can be used to sort the documents.
     * e.g. text fields cannot be used to sort directly, and instead we need to use
     * the field data of type keyword
     *
     * @param propertyMap the map containing name that should be used for queries instead of the property name
     * @param property    the property name to be mapped
     * @return the mapped property name, if found, otherwise the provided property name
     */
    @NonNull
    private static String mapSortProperty(@Nullable Map<String, String> propertyMap, @NonNull String property) {
        return Optional.ofNullable(propertyMap)
                .map(m -> m.getOrDefault(property, property))
                .orElse(property);
    }
}
