package com.finconsgroup.itserr.marketplace.search.dm.service.impl;

import com.finconsgroup.itserr.marketplace.core.web.exception.WP2BusinessException;
import com.finconsgroup.itserr.marketplace.core.web.exception.WP2ResourceNotFoundException;
import com.finconsgroup.itserr.marketplace.search.dm.bean.SearchRequest;
import com.finconsgroup.itserr.marketplace.search.dm.config.DefaultSearchProperties;
import com.finconsgroup.itserr.marketplace.search.dm.config.EventSearchProperties;
import com.finconsgroup.itserr.marketplace.search.dm.config.SearchProperties;
import com.finconsgroup.itserr.marketplace.search.dm.dto.InputEventDto;
import com.finconsgroup.itserr.marketplace.search.dm.dto.OutputEventDto;
import com.finconsgroup.itserr.marketplace.search.dm.dto.OutputEventLocalSearchDto;
import com.finconsgroup.itserr.marketplace.search.dm.dto.OutputGlobalSearchAutoCompleteDataDto;
import com.finconsgroup.itserr.marketplace.search.dm.dto.OutputGlobalSearchAutoCompleteDto;
import com.finconsgroup.itserr.marketplace.search.dm.dto.OutputGlobalSearchDataDto;
import com.finconsgroup.itserr.marketplace.search.dm.dto.OutputGlobalSearchDto;
import com.finconsgroup.itserr.marketplace.search.dm.entity.Event;
import com.finconsgroup.itserr.marketplace.search.dm.entity.UserProfile;
import com.finconsgroup.itserr.marketplace.search.dm.enums.Category;
import com.finconsgroup.itserr.marketplace.search.dm.event.UserProfileUpdatedEvent;
import com.finconsgroup.itserr.marketplace.search.dm.mapper.EventMapper;
import com.finconsgroup.itserr.marketplace.search.dm.repository.CustomAggregationRepository;
import com.finconsgroup.itserr.marketplace.search.dm.repository.CustomQueryRepository;
import com.finconsgroup.itserr.marketplace.search.dm.repository.EventRepository;
import com.finconsgroup.itserr.marketplace.search.dm.service.EventService;
import lombok.extern.slf4j.Slf4j;
import org.springframework.context.ApplicationListener;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.lang.NonNull;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.function.Function;

/**
 * Default implementation of {@link EventService} to perform search and document related operations
 * by connecting to an OpenSearch instance.
 */
@Service
@Slf4j
public class DefaultEventService implements EventService, ApplicationListener<UserProfileUpdatedEvent> {

    private final EventRepository eventRepository;
    private final EventMapper eventMapper;
    private final EventSearchProperties eventSearchProperties;
    private final Map<String, String> sortFilterPropertyMap;

    public DefaultEventService(EventRepository eventRepository,
                               EventMapper eventMapper,
                               EventSearchProperties eventSearchProperties,
                               DefaultSearchProperties defaultSearchProperties) {
        this.eventRepository = eventRepository;
        this.eventMapper = eventMapper;
        this.eventSearchProperties = eventSearchProperties;
        this.sortFilterPropertyMap = buildSortFilterPropertyMap(eventSearchProperties.search(),
                defaultSearchProperties.search().sortFilterPropertyMap());
    }

    @Override
    @Transactional
    @NonNull
    public OutputEventDto upsertDocument(@NonNull InputEventDto dto) {
        Event event = eventMapper.toEntity(dto);
        Event savedEvent = eventRepository.save(event);
        return eventMapper.toDto(savedEvent);
    }

    @Override
    @Transactional(readOnly = true)
    @NonNull
    public OutputEventDto getDocument(@NonNull String id) {
        Event savedEvent = eventRepository
                .findById(id)
                .orElseThrow(() -> new WP2ResourceNotFoundException(
                        "search_dm_event_not_found"));
        return eventMapper.toDto(savedEvent);
    }

    @Override
    @Transactional
    public void deleteDocument(@NonNull String id) {
        if (!eventRepository.existsById(id)) {
            throw new WP2ResourceNotFoundException("search_dm_event_not_found");
        }

        eventRepository.deleteById(id);
    }

    @Override
    public void deleteAll() {
        if (!eventSearchProperties.search().enableDeleteAll()) {
            throw new WP2BusinessException("search_dm_event_delete_all_not_enabled");
        }

        eventRepository.deleteAll();
    }

    @Override
    @NonNull
    @Transactional(readOnly = true)
    public List<OutputGlobalSearchAutoCompleteDto> getAutoCompletions(@NonNull String terms) {
        Page<OutputGlobalSearchAutoCompleteDataDto> resultPage = search(
                SearchRequest.builder().terms(terms).build(),
                eventSearchProperties.search().autoCompletion().sourceFields(),
                eventMapper::toAutoCompleteDataDto,
                PageRequest.of(0, eventSearchProperties.search().autoCompletion().topHitsLimit())
        );
        if (resultPage.isEmpty()) {
            return List.of();
        } else {
            return List.of(OutputGlobalSearchAutoCompleteDto
                    .builder()
                    .category(Category.EVENT.getId())
                    .data(resultPage.getContent())
                    .build());
        }
    }


    @NonNull
    @Override
    @Transactional(readOnly = true)
    public Page<OutputEventLocalSearchDto> getLocalSearch(String terms, String filters,
                                                          @NonNull Pageable pageable) {
        SearchRequest searchRequest = SearchRequest
                .builder()
                .terms(terms)
                .queryFilters(buildQueryFilters(filters, sortFilterPropertyMap))
                .build();
        Pageable sortedPageable = applySort(pageable, eventSearchProperties.search(),
                sortFilterPropertyMap);
        return search(searchRequest,
                eventSearchProperties.search().local().sourceFields(),
                eventMapper::toLocalSearchDto,
                sortedPageable);
    }

    @Override
    @NonNull
    @Transactional(readOnly = true)
    public List<OutputGlobalSearchDto> getSearch(@NonNull String terms) {
        Map<String, List<OutputGlobalSearchDataDto>> searchResultAggregatedByType = searchByAggregation(
                SearchRequest.builder().terms(terms).build(),
                eventSearchProperties.search().globalSearch().sourceFields(),
                eventSearchProperties.search().globalSearch().topHitsLimit(),
                eventSearchProperties.search().globalSearch().aggregation(),
                eventMapper::toGlobalSearchDataDto);

        if (searchResultAggregatedByType.isEmpty()) {
            return List.of();
        } else {
            List<OutputGlobalSearchDto> globalSearchResults = new ArrayList<>();
            searchResultAggregatedByType.forEach((type, results) ->
                    globalSearchResults.add(OutputGlobalSearchDto
                            .builder()
                            .category(Category.EVENT.getId())
                            .type(type)
                            .data(results)
                            .build()));
            return globalSearchResults;
        }
    }

    @Override
    public void onApplicationEvent(@NonNull UserProfileUpdatedEvent event) {
        try {
            if (event.isUserProfileMinimalUpdated()) {
                syncDocumentsForUserProfile(event.getUserProfile());
            }
        } catch (Exception ex) {
            log.error("Error occurred while syncing events for user profile with id - {}",
                    event.getUserProfile().getId(), ex);
        }
    }

    @NonNull
    @Override
    public Page<Event> searchDocumentsForUserProfileSync(@NonNull SearchRequest searchRequest, @NonNull Pageable pageable) {
        return search(searchRequest, null, Function.identity(), pageable);
    }

    @Override
    public void syncDocumentForUserProfile(@NonNull Event event,
                                           @NonNull UserProfile userProfile) {
        boolean updated = false;
        if (event.getSubscribedParticipants() != null) {
            for (Event.SubscribedParticipant subscribedParticipant : event.getSubscribedParticipants()) {
                if (Objects.equals(subscribedParticipant.getUserId(), userProfile.getId()) &&
                        !isSubscriberUserProfileUpdated(subscribedParticipant, userProfile)) {
                    updateSubscriberUserProfile(subscribedParticipant, userProfile);
                    updated = true;
                }
            }
        }
        if (event.getMaintainer() != null &&
                Objects.equals(event.getMaintainer().getId(), userProfile.getId()) &&
                !isUserProfileMinimalUpdated(event.getMaintainer(), userProfile)) {
            updateUserProfileMinimal(event.getMaintainer(), userProfile);
            updated = true;
        }

        if (event.getEventPlanner() != null &&
                Objects.equals(event.getEventPlanner().getId(), userProfile.getId()) &&
                !isUserProfileMinimalUpdated(event.getEventPlanner(), userProfile)) {
            updateUserProfileMinimal(event.getEventPlanner(), userProfile);
            updated = true;
        }

        if (updated) {
            eventRepository.save(event);
        }
    }

    @NonNull
    @Override
    public Class<Event> getDocumentClass() {
        return Event.class;
    }

    @NonNull
    @Override
    public SearchProperties getSearchProperties() {
        return eventSearchProperties.search();
    }

    @NonNull
    @Override
    public CustomQueryRepository getCustomQueryRepository() {
        return eventRepository;
    }

    @NonNull
    @Override
    public CustomAggregationRepository getCustomAggregationRepository() {
        return eventRepository;
    }

    private boolean isSubscriberUserProfileUpdated(@NonNull Event.SubscribedParticipant subscribedParticipant,
                                                   @NonNull UserProfile userProfile) {
        return !Objects.equals(subscribedParticipant.getFirstName(), userProfile.getFirstName())
                || !Objects.equals(subscribedParticipant.getLastName(), userProfile.getLastName())
                || !Objects.equals(subscribedParticipant.getEmail(), userProfile.getEmail())
                || !Objects.equals(subscribedParticipant.getOrcid(), userProfile.getOrcid());
    }

    private void updateSubscriberUserProfile(@NonNull Event.SubscribedParticipant subscribedParticipant,
                                             @NonNull UserProfile userProfile) {
        subscribedParticipant.setFirstName(userProfile.getFirstName());
        subscribedParticipant.setLastName(userProfile.getLastName());
        subscribedParticipant.setEmail(userProfile.getEmail());
        subscribedParticipant.setOrcid(userProfile.getOrcid());
    }

}
