package com.finconsgroup.itserr.marketplace.search.dm.service.impl;

import com.finconsgroup.itserr.marketplace.core.web.exception.WP2BusinessException;
import com.finconsgroup.itserr.marketplace.core.web.exception.WP2ResourceNotFoundException;
import com.finconsgroup.itserr.marketplace.search.dm.bean.SearchRequest;
import com.finconsgroup.itserr.marketplace.search.dm.config.DefaultSearchProperties;
import com.finconsgroup.itserr.marketplace.search.dm.config.NewsSearchProperties;
import com.finconsgroup.itserr.marketplace.search.dm.config.SearchProperties;
import com.finconsgroup.itserr.marketplace.search.dm.dto.InputNewsDto;
import com.finconsgroup.itserr.marketplace.search.dm.dto.OutputGlobalSearchAutoCompleteDataDto;
import com.finconsgroup.itserr.marketplace.search.dm.dto.OutputGlobalSearchAutoCompleteDto;
import com.finconsgroup.itserr.marketplace.search.dm.dto.OutputGlobalSearchDataDto;
import com.finconsgroup.itserr.marketplace.search.dm.dto.OutputGlobalSearchDto;
import com.finconsgroup.itserr.marketplace.search.dm.dto.OutputNewsDto;
import com.finconsgroup.itserr.marketplace.search.dm.dto.OutputNewsLocalSearchDto;
import com.finconsgroup.itserr.marketplace.search.dm.entity.News;
import com.finconsgroup.itserr.marketplace.search.dm.entity.RegisteredAuthor;
import com.finconsgroup.itserr.marketplace.search.dm.entity.UserProfile;
import com.finconsgroup.itserr.marketplace.search.dm.enums.Category;
import com.finconsgroup.itserr.marketplace.search.dm.event.UserProfileUpdatedEvent;
import com.finconsgroup.itserr.marketplace.search.dm.mapper.NewsMapper;
import com.finconsgroup.itserr.marketplace.search.dm.repository.CustomAggregationRepository;
import com.finconsgroup.itserr.marketplace.search.dm.repository.CustomQueryRepository;
import com.finconsgroup.itserr.marketplace.search.dm.repository.NewsRepository;
import com.finconsgroup.itserr.marketplace.search.dm.service.NewsService;
import lombok.extern.slf4j.Slf4j;
import org.springframework.context.ApplicationListener;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.lang.NonNull;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.function.Function;

/**
 * Default implementation of {@link NewsService} to perform search and document related operations
 * by connecting to an OpenSearch instance.
 */
@Service
@Slf4j
public class DefaultNewsService implements NewsService, ApplicationListener<UserProfileUpdatedEvent> {

    private final NewsRepository newsRepository;
    private final NewsMapper newsMapper;
    private final NewsSearchProperties newsSearchProperties;
    private final Map<String, String> sortFilterPropertyMap;

    public DefaultNewsService(NewsRepository newsRepository,
                              NewsMapper newsMapper,
                              NewsSearchProperties newsSearchProperties,
                              DefaultSearchProperties defaultSearchProperties) {
        this.newsRepository = newsRepository;
        this.newsMapper = newsMapper;
        this.newsSearchProperties = newsSearchProperties;
        this.sortFilterPropertyMap = buildSortFilterPropertyMap(newsSearchProperties.search(),
                defaultSearchProperties.search().sortFilterPropertyMap());
    }

    @Override
    @Transactional
    @NonNull
    public OutputNewsDto upsertDocument(@NonNull InputNewsDto dto) {
        News news = newsMapper.toEntity(dto);
        News savedNews = newsRepository.save(news);
        return newsMapper.toDto(savedNews);
    }

    @Override
    @Transactional(readOnly = true)
    @NonNull
    public OutputNewsDto getDocument(@NonNull String id) {
        News savedNews = newsRepository
                .findById(id)
                .orElseThrow(() -> new WP2ResourceNotFoundException(
                        "search_dm_news_not_found"));
        return newsMapper.toDto(savedNews);
    }

    @Override
    @Transactional
    public void deleteDocument(@NonNull String id) {
        if (!newsRepository.existsById(id)) {
            throw new WP2ResourceNotFoundException("search_dm_news_not_found");
        }

        newsRepository.deleteById(id);
    }

    @Override
    public void deleteAll() {
        if (!newsSearchProperties.search().enableDeleteAll()) {
            throw new WP2BusinessException("search_dm_news_delete_all_not_enabled");
        }

        newsRepository.deleteAll();
    }

    @Override
    @NonNull
    @Transactional(readOnly = true)
    public List<OutputGlobalSearchAutoCompleteDto> getAutoCompletions(@NonNull String terms) {
        Page<OutputGlobalSearchAutoCompleteDataDto> resultPage = search(
                SearchRequest.builder().terms(terms).build(),
                newsSearchProperties.search().autoCompletion().sourceFields(),
                newsMapper::toAutoCompleteDataDto,
                PageRequest.of(0, newsSearchProperties.search().autoCompletion().topHitsLimit())
        );
        if (resultPage.isEmpty()) {
            return List.of();
        } else {
            return List.of(OutputGlobalSearchAutoCompleteDto
                    .builder()
                    .category(Category.NEWS.getId())
                    .data(resultPage.getContent())
                    .build());
        }
    }


    @NonNull
    @Override
    @Transactional(readOnly = true)
    public Page<OutputNewsLocalSearchDto> getLocalSearch(String terms, String filters,
                                                         @NonNull Pageable pageable) {
        SearchRequest searchRequest = SearchRequest
                .builder()
                .terms(terms)
                .queryFilters(buildQueryFilters(filters, sortFilterPropertyMap))
                .build();
        Pageable sortedPageable = applySort(pageable, newsSearchProperties.search(),
                sortFilterPropertyMap);
        return search(searchRequest,
                newsSearchProperties.search().local().sourceFields(),
                newsMapper::toLocalSearchDto,
                sortedPageable);
    }

    @Override
    @NonNull
    @Transactional(readOnly = true)
    public List<OutputGlobalSearchDto> getSearch(@NonNull String terms) {
        Map<String, List<OutputGlobalSearchDataDto>> searchResultAggregatedByType = searchByAggregation(
                SearchRequest.builder().terms(terms).build(),
                newsSearchProperties.search().globalSearch().sourceFields(),
                newsSearchProperties.search().globalSearch().topHitsLimit(),
                newsSearchProperties.search().globalSearch().aggregation(),
                newsMapper::toGlobalSearchDataDto);

        if (searchResultAggregatedByType.isEmpty()) {
            return List.of();
        } else {
            List<OutputGlobalSearchDto> globalSearchResults = new ArrayList<>();
            searchResultAggregatedByType.forEach((type, results) ->
                    globalSearchResults.add(OutputGlobalSearchDto
                            .builder()
                            .category(Category.NEWS.getId())
                            .type(type)
                            .data(results)
                            .build()));
            return globalSearchResults;
        }
    }

    @Override
    public void onApplicationEvent(@NonNull UserProfileUpdatedEvent event) {
        try {
            if (event.isUserProfileMinimalUpdated() || event.isLocationUpdated()) {
                syncDocumentsForUserProfile(event.getUserProfile());
            }
        } catch (Exception ex) {
            log.error("Error occurred while syncing news for user profile with id - {}",
                    event.getUserProfile().getId(), ex);
        }
    }

    @NonNull
    @Override
    public Page<News> searchDocumentsForUserProfileSync(@NonNull SearchRequest searchRequest, @NonNull Pageable pageable) {
        return search(searchRequest, null, Function.identity(), pageable);
    }

    @Override
    public void syncDocumentForUserProfile(@NonNull News news,
                                           @NonNull UserProfile userProfile) {
        boolean updated = false;
        if (news.getCreator() != null &&
                Objects.equals(news.getCreator().getId(), userProfile.getId()) &&
                isRegisteredAuthorUserProfileUpdated(news.getCreator(), userProfile)) {
            updateRegisteredAuthorUserProfile(news.getCreator(), userProfile);
            updated = true;
        }
        if (news.getAuthors() != null) {
            for (RegisteredAuthor registeredAuthor : news.getAuthors()) {
                if (Objects.equals(registeredAuthor.getId(), userProfile.getId()) &&
                        isRegisteredAuthorUserProfileUpdated(registeredAuthor, userProfile)) {
                    updateRegisteredAuthorUserProfile(registeredAuthor, userProfile);
                    updated = true;
                }
            }
        }

        if (updated) {
            newsRepository.save(news);
        }
    }

    @NonNull
    @Override
    public Class<News> getDocumentClass() {
        return News.class;
    }

    @NonNull
    @Override
    public SearchProperties getSearchProperties() {
        return newsSearchProperties.search();
    }

    @NonNull
    @Override
    public CustomQueryRepository getCustomQueryRepository() {
        return newsRepository;
    }

    @NonNull
    @Override
    public CustomAggregationRepository getCustomAggregationRepository() {
        return newsRepository;
    }

}
