package com.finconsgroup.itserr.marketplace.userprofile.bs.api;

import com.finconsgroup.itserr.marketplace.core.web.exception.ErrorResponseDto;
import com.finconsgroup.itserr.marketplace.userprofile.bs.dto.OutputEndorsementAcknowledgementDto;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.media.Content;
import io.swagger.v3.oas.annotations.media.Schema;
import io.swagger.v3.oas.annotations.responses.ApiResponse;
import io.swagger.v3.oas.annotations.security.SecurityRequirement;
import io.swagger.v3.oas.annotations.tags.Tag;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.web.bind.annotation.DeleteMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.ResponseStatus;

import java.util.UUID;


/**
 * Defines the REST API contract for managing user endorsements.
 *
 * <p>
 * This interface provides endpoints for endorsing and removing endorsements
 * on a user's expertise within their profile.
 * </p>
 *
 * <p>Example usage:</p>
 * <pre>
 * POST   /profiles/{profileId}/endorsements/{expertiseId} - Add an endorsement to a user's expertise
 * DELETE /profiles/{profileId}/endorsements/{expertiseId} - Remove an endorsement from a user's expertise
 * </pre>
 */

@Tag(
        name = "Endorsement",
        description = "The Endorsement API: it provides endpoints for " +
                "endorsing expertise."
)
@SecurityRequirement(name = "BearerAuth")
public interface EndorsementApi {

    /**
     * Endorses a specific expertise of a user profile.
     *
     * @param profileId   the ID of the user profile receiving the endorsement
     * @param expertiseId the ID of the expertise being endorsed
     * @return a {@link OutputEndorsementAcknowledgementDto} representing the endorsement acknowledgement
     */
    @Operation(
            summary = "Endorse a user's expertise",
            description = "Adds an endorsement to a user's specific expertise by expertise ID.",
            responses = {
                    @ApiResponse(responseCode = "200", description = "Endorsement added successfully"),
                    @ApiResponse(responseCode = "400", description = "Bad Request",
                            content = {@Content(
                                    mediaType = MediaType.APPLICATION_JSON_VALUE,
                                    schema = @Schema(implementation = ErrorResponseDto.class)
                            )}
                    ),
                    @ApiResponse(responseCode = "404", description = "User profile or expertise not found",
                            content = {@Content(
                                    mediaType = MediaType.APPLICATION_JSON_VALUE,
                                    schema = @Schema(implementation = ErrorResponseDto.class)
                            )}
                    ),
                    @ApiResponse(responseCode = "409", description = "Conflict",
                            content = {@Content(
                                    mediaType = MediaType.APPLICATION_JSON_VALUE,
                                    schema = @Schema(implementation = ErrorResponseDto.class)
                            )}
                    )
            }
    )
    @PostMapping(value = "/profiles/{profileId}/expertise/{expertiseId}/confirm", produces = MediaType.APPLICATION_JSON_VALUE)
    @ResponseStatus(HttpStatus.OK)
    OutputEndorsementAcknowledgementDto addEndorsement(@PathVariable("profileId") UUID profileId, @PathVariable("expertiseId") UUID expertiseId);

    /**
     * Delete an existing endorsement.
     *
     * @param profileId   the ID of the user profile receiving the endorsement
     * @param expertiseId the ID of the expertise being endorsed
     */
    @Operation(
            summary = "Delete an existing endorsement",
            description = "Adds an endorsement to a user's specific expertise by expertise ID.",
            responses = {
                    @ApiResponse(responseCode = "200", description = "Endorsement deleted successfully"),
                    @ApiResponse(responseCode = "404", description = "User profile or expertise not found",
                            content = @Content(
                                    mediaType = MediaType.APPLICATION_JSON_VALUE,
                                    schema = @Schema(implementation = ErrorResponseDto.class)
                            )
                    )
            }
    )
    @DeleteMapping(value = "/profiles/{profileId}/expertise/{expertiseId}/confirm", produces = MediaType.APPLICATION_JSON_VALUE)
    @ResponseStatus(HttpStatus.OK)
    void removeEndorsement(@PathVariable("profileId") UUID profileId, @PathVariable("expertiseId") UUID expertiseId);

}
