package com.finconsgroup.itserr.marketplace.userprofile.bs.api;

import com.finconsgroup.itserr.marketplace.core.web.exception.ErrorResponseDto;
import com.finconsgroup.itserr.marketplace.userprofile.bs.dto.InputUpdateUserProfileDto;
import com.finconsgroup.itserr.marketplace.userprofile.bs.dto.InputUserProfileDto;
import com.finconsgroup.itserr.marketplace.userprofile.bs.dto.OutputUserProfileDto;
import com.finconsgroup.itserr.marketplace.userprofile.bs.dto.OutputUserSettingDto;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.media.Content;
import io.swagger.v3.oas.annotations.media.Schema;
import io.swagger.v3.oas.annotations.responses.ApiResponse;
import io.swagger.v3.oas.annotations.security.SecurityRequirement;
import io.swagger.v3.oas.annotations.tags.Tag;
import jakarta.validation.Valid;
import jakarta.validation.constraints.NotBlank;
import org.springframework.http.MediaType;
import org.springframework.http.HttpStatus;
import org.springframework.web.bind.annotation.DeleteMapping;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.PutMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RequestPart;
import org.springframework.web.bind.annotation.ResponseStatus;
import org.springframework.web.bind.annotation.RequestBody;

import org.springframework.web.multipart.MultipartFile;

import java.net.URI;
import java.util.List;

/**
 * Interface for managing REST operations on UserProfile.
 *
 * <p>
 * Provides endpoints for uploading user profile image and CV.
 * </p>
 */
@Tag(
        name = "UserProfile",
        description = "API for managing user profiles"
)
@SecurityRequirement(name = "BearerAuth")
public interface UserProfileApi {

    /**
     * Creates a new userprofile.
     *
     * @param inputUserProfileDto the input data transfer object containing userprofile details
     * @return the created {@link OutputUserProfileDto} and HTTP status 201 (Created)
     */
    @Operation(
            summary = "create userprofile for the authenticated user",
            responses = {
                    @ApiResponse(responseCode = "201", description = "Created"),
                    @ApiResponse(responseCode = "409", description = "Another metadata exists with the same name", content = @Content),
                    @ApiResponse(responseCode = "503", description = "Service unavailable", content = @Content)
            }
    )
    @PostMapping(
            value = "/profile",
            consumes = MediaType.APPLICATION_JSON_VALUE,
            produces = MediaType.APPLICATION_JSON_VALUE
    )
    @ResponseStatus(HttpStatus.CREATED)
    OutputUserProfileDto create(@Valid @RequestBody InputUserProfileDto inputUserProfileDto);

    /**
     * Updates an existing userprofile by id.
     *
     * @param inputUpdateUserProfileDto the input data transfer object containing updated userprofile details
     * @return the updated {@link OutputUserProfileDto} and HTTP status 200 (OK)
     */
    @Operation(
            summary = "update userprofile by id",
            responses = {
                    @ApiResponse(responseCode = "200", description = "OK"),
                    @ApiResponse(responseCode = "404", description = "Not Found"),
            }
    )
    @PutMapping(
            value = "/profile",
            consumes = MediaType.APPLICATION_JSON_VALUE,
            produces = MediaType.APPLICATION_JSON_VALUE
    )
    @ResponseStatus(HttpStatus.OK)
    OutputUserProfileDto updateById(
            @Valid @RequestBody InputUpdateUserProfileDto inputUpdateUserProfileDto
    );

    /**
     * Uploads the profile image of an existing user.
     *
     * @param image the image file to upload (PNG, JPEG, etc.)
     * @return the URL where the uploaded image can be accessed
     * @throws org.springframework.web.multipart.MaxUploadSizeExceededException if the file is too large
     */
    @Operation(
            summary = "Upload the profile image of a user by ID",
            responses = {
                    @ApiResponse(responseCode = "200", description = "Profile image updated"),
                    @ApiResponse(responseCode = "404", description = "Profile not found")
            }
    )
    @PostMapping(
            value = "/profile/image",
            consumes = MediaType.MULTIPART_FORM_DATA_VALUE,
            produces = MediaType.APPLICATION_JSON_VALUE
    )
    URI uploadImage(
            @RequestPart("file") MultipartFile image
    );

    /**
     * Uploads the CV of an existing user.
     *
     * @param cvFile the CV file to upload (PDF, DOC, etc.)
     * @return the URL where the uploaded CV can be accessed
     * @throws org.springframework.web.multipart.MaxUploadSizeExceededException if the file is too large
     */
    @Operation(
            summary = "Upload the profile CV of a user by ID",
            responses = {
                    @ApiResponse(responseCode = "200", description = "CV updated"),
                    @ApiResponse(responseCode = "404", description = "Profile not found")
            }
    )
    @PostMapping(
            value = "/profile/cv",
            consumes = MediaType.MULTIPART_FORM_DATA_VALUE,
            produces = MediaType.APPLICATION_JSON_VALUE
    )
    URI uploadCv(
            @RequestPart("file") MultipartFile cvFile
    );

    /**
     * Change the preferred language of an existing user.
     *
     * @param language the user preferred language (EN, IT, etc.)
     * @return the updated user language
     */
    @Operation(
            summary = "Change the preferred language of an existing user",
            responses = {
                    @ApiResponse(responseCode = "200", description = "Language updated"),
                    @ApiResponse(responseCode = "404", description = "Profile not found")
            }
    )
    @PutMapping(
            value = "/profile/language/{language}",
            produces = MediaType.APPLICATION_JSON_VALUE
    )
    OutputUserSettingDto updateLanguage(@PathVariable("language") String language);

    /**
     * Get the user profiles matching the provided role.
     *
     * @param roleName the role name to search for
     * @return the list of matching {@link OutputUserProfileDto} and HTTP status 200 (OK)
     */
    @Operation(
            summary = "get user profiles by role name",
            responses = {
                    @ApiResponse(responseCode = "200", description = "OK")
            },
            hidden = true // internal api to be used by other services only
    )
    @GetMapping(value = "/profiles-by-role", produces = MediaType.APPLICATION_JSON_VALUE)
    @ResponseStatus(HttpStatus.OK)
    List<OutputUserProfileDto> getUserProfilesByRole(
            @NotBlank @RequestParam("roleName") String roleName
    );

    /**
     * activate the user profile.
     */
    @Operation(
            summary = "activate a user profile by ID",
            responses = {
                    @ApiResponse(responseCode = "200", description = "OK"),
                    @ApiResponse(responseCode = "404", description = "Not Found",
                            content = {
                                    @Content(
                                            mediaType = MediaType.APPLICATION_JSON_VALUE,
                                            schema = @Schema(implementation = ErrorResponseDto.class)
                                    )
                            }),
            }
    )
    @PostMapping(value = "/profile/activate", produces = MediaType.APPLICATION_JSON_VALUE)
    @ResponseStatus(HttpStatus.OK)
    void activate();

    /**
     * deactivate the user profile.
     */
    @Operation(
            summary = "deactivate a user profile by ID",
            responses = {
                    @ApiResponse(responseCode = "200", description = "OK"),
                    @ApiResponse(responseCode = "404", description = "Not Found",
                            content = {
                                    @Content(
                                            mediaType = MediaType.APPLICATION_JSON_VALUE,
                                            schema = @Schema(implementation = ErrorResponseDto.class)
                                    )
                            }),
            }
    )
    @DeleteMapping(value = "/profile/activate", produces = MediaType.APPLICATION_JSON_VALUE)
    @ResponseStatus(HttpStatus.OK)
    void deactivate();

    /**
     * Change the home panel options of an existing user.
     *
     * @param hidePanel the user preferred setting (true, false)
     * @return the updated user flag value
     */
    @Operation(
            summary = "Change the home panel options of an existing user",
            responses = {
                    @ApiResponse(responseCode = "200", description = "Panel flag updated"),
                    @ApiResponse(responseCode = "404", description = "Profile not found")
            }
    )
    @PutMapping(
            value = "/profile/hidePanel/{hidePanel}",
            produces = MediaType.APPLICATION_JSON_VALUE
    )
    Boolean updateHidePanel(@PathVariable("hidePanel") Boolean hidePanel);
}
