package com.finconsgroup.itserr.marketplace.userprofile.bs.controller;

import com.finconsgroup.itserr.marketplace.core.web.exception.WP2ResourceNotFoundException;
import com.finconsgroup.itserr.marketplace.userprofile.bs.api.DiagnosticsApi;
import com.finconsgroup.itserr.marketplace.userprofile.bs.dto.OutputUserProfileDto;
import com.finconsgroup.itserr.marketplace.userprofile.bs.messaging.ResourceProducer;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.core.LoggerContext;
import org.apache.logging.log4j.core.appender.RollingFileAppender;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.core.io.FileSystemResource;
import org.springframework.core.io.Resource;
import org.springframework.http.HttpHeaders;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.RestController;

import java.io.File;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.HashMap;
import java.util.Map;
import java.util.UUID;

/**
 * REST controller class for handling debugging/analysis related API requests.
 *
 * <p>Implements the {@link DiagnosticsApi} interface.</p>
 */
@Slf4j
@RestController
@RequiredArgsConstructor
public class DiagnosticsController implements DiagnosticsApi {

    private final ResourceProducer resourceProducer;

    // For storing rabbitmq test messages
    static Map<String, OutputUserProfileDto> userProfileDtoHashMap = new HashMap<>();

    // The name of the RollingFile Log4j2 component
    // It is not the name of the log file
    // We use this to dynamically retrieve the name of the log file.
    @Value("${log.log4j2-rolling-file-name}")
    private String log4j2RollingFileName;

    @Override
    public ResponseEntity<Resource> downloadLogs() {
        log.debug("call to DiagnosticsController - downloadLogs");
        Path filePath = Paths.get(getLogFilePathFromLog4j2());
        log.debug("Trying to retrieve log file from: {}", filePath);
        File logFile = filePath.toFile();
        if (!logFile.exists() || !logFile.isFile()) {
            throw new WP2ResourceNotFoundException("Log file not found: %s".formatted(logFile.getAbsolutePath()));
        }
        Resource resource = new FileSystemResource(logFile);
        return ResponseEntity.ok()
                .header(HttpHeaders.CONTENT_DISPOSITION, getContentDispositionHeaderValue(logFile.getName()))
                .contentType(MediaType.APPLICATION_OCTET_STREAM)
                .body(resource);
    }

    @Override
    public ResponseEntity<String> publishMessage(boolean isCreated, boolean isUpdated, boolean isDeleted, String name) {
        log.debug("call to DiagnosticsController - publishMessage, name: %s".formatted(name));
        if (!(isCreated || isUpdated || isDeleted)) {
            return ResponseEntity.badRequest()
                    .body("One between isCreated, isUpdated, isDeleted should be true");
        }
        OutputUserProfileDto outputUserProfileDto;
        if (isCreated) {
            outputUserProfileDto = OutputUserProfileDto.builder()
                    .id(UUID.randomUUID())
                    .firstName(name)
                    .socialProfile("the CREATED social profile")
                    .shortBio("the CREATED short bio")
                    .build();
            log.info("sending created message, outputUserProfileDto: %s".formatted(outputUserProfileDto));
            userProfileDtoHashMap.put(outputUserProfileDto.getFirstName(), outputUserProfileDto);
            resourceProducer.publishCreatedResource(outputUserProfileDto);
        } else if (isUpdated) {
            outputUserProfileDto = userProfileDtoHashMap.get(name);
            if (outputUserProfileDto == null) {
                return ResponseEntity.badRequest()
                        .body("no test item created for name: %s".formatted(name));
            }
            outputUserProfileDto.setSocialProfile("the UPDATED social profile");
            outputUserProfileDto.setShortBio("the UPDATED short bio");
            log.info("sending updated message, outputUserProfileDto: %s".formatted(outputUserProfileDto));
            resourceProducer.publishUpdatedResource(outputUserProfileDto);
        } else {
            outputUserProfileDto = userProfileDtoHashMap.get(name);
            if (outputUserProfileDto == null) {
                return ResponseEntity.badRequest()
                        .body("no test item created for name: %s".formatted(name));
            }
            log.info("sending deleted message, outputUserProfileDto: %s".formatted(outputUserProfileDto));
            userProfileDtoHashMap.remove(outputUserProfileDto.getFirstName());
            resourceProducer.publishDeletedResource(outputUserProfileDto);
        }
        return ResponseEntity.ok("message sent correctly, outputUserProfileDto: %s".formatted(outputUserProfileDto));
    }

    // private
    public String getLogFilePathFromLog4j2() {
        LoggerContext context = (LoggerContext) LogManager.getContext(false);
        RollingFileAppender appender = context.getConfiguration().getAppender(log4j2RollingFileName);
        return appender.getFileName();
    }

    private static String getContentDispositionHeaderValue(String fileName) {
        return "attachment; filename=\"%s\"".formatted(fileName);
    }

}