package com.finconsgroup.itserr.marketplace.userprofile.bs.messaging;

import com.fasterxml.jackson.core.type.TypeReference;
import com.finconsgroup.itserr.marketplace.userprofile.bs.client.UserProfileDmClient;
import com.finconsgroup.itserr.marketplace.userprofile.bs.client.dto.InputFindUserProfilesByInterestsDto;
import com.finconsgroup.itserr.marketplace.userprofile.bs.config.properties.MessagingCatalogBsProperties;
import com.finconsgroup.itserr.marketplace.userprofile.bs.config.properties.UserProfileBsConfigurationProperties;
import com.finconsgroup.itserr.marketplace.userprofile.bs.dto.UserProfileInterestsMessagingAdditionalDataDto;
import com.finconsgroup.itserr.marketplace.userprofile.bs.messaging.dto.CatalogItemMessagingStatusAdditionalDataDto;
import com.finconsgroup.itserr.messaging.consumer.CloudEventConsumer;
import com.finconsgroup.itserr.messaging.dto.MessagingEventDto;
import io.cloudevents.CloudEvent;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Component;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Optional;
import java.util.UUID;

@Component("catalogEventConsumer")
@Slf4j
public class CatalogEventConsumer extends CloudEventConsumer {

    private static final String OPERATION_TYPE_CREATE = "create";
    private static final String STATUS_APPROVED = "approved";

    private final UserProfileDmClient userProfileDmClient;
    private final EventProducer eventProducer;

    /**
     * Constructs the CatalogEventConsumer
     *
     * @param userProfileBsConfigurationProperties the application configuration properties
     * @param userProfileDmClient                  user profile dm client
     * @param eventProducer                        event producer
     */
    public CatalogEventConsumer(UserProfileBsConfigurationProperties userProfileBsConfigurationProperties,
                                UserProfileDmClient userProfileDmClient,
                                EventProducer eventProducer) {
        MessagingCatalogBsProperties messagingCatalogBsProperties = userProfileBsConfigurationProperties.getMessaging().getCatalogBs();
        this.addHandler(messagingCatalogBsProperties.getResourceStatusChangeEventType(), new TypeReference<>() {
        }, this::handleResourceStatusChange);
        this.userProfileDmClient = userProfileDmClient;
        this.eventProducer = eventProducer;
    }

    private void handleResourceStatusChange(MessagingEventDto<CatalogItemMessagingStatusAdditionalDataDto> eventPayload,
                                            CloudEvent cloudEvent) {
        CatalogItemMessagingStatusAdditionalDataDto data = eventPayload.getAdditionalData();

        // when the item creation is approved for the first time
        // generate the matching interests event
        if (STATUS_APPROVED.equalsIgnoreCase(eventPayload.getStatus()) && OPERATION_TYPE_CREATE.equalsIgnoreCase(data.getOperationType())) {
            List<String> matchingInterestsToCheck = new ArrayList<>(Optional.ofNullable(data.getTags()).orElse(List.of()));
            matchingInterestsToCheck.add(eventPayload.getTitle());
            matchingInterestsToCheck.add(data.getContent());

            List<UUID> userProfileIdsWithInterests = userProfileDmClient.findMatchingInterests(
                    InputFindUserProfilesByInterestsDto.builder().stringsToCheck(matchingInterestsToCheck).build());

            // did not find any matching user profiles
            if (userProfileIdsWithInterests == null || userProfileIdsWithInterests.isEmpty()) {
                return;
            }

            MessagingEventDto<UserProfileInterestsMessagingAdditionalDataDto> messagingEventDto =
                    new MessagingEventDto<>();

            messagingEventDto.setId(eventPayload.getId());
            messagingEventDto.setUser(eventPayload.getUser());
            messagingEventDto.setName(eventPayload.getName());
            messagingEventDto.setTitle(eventPayload.getTitle());
            messagingEventDto.setCategory(eventPayload.getCategory());
            messagingEventDto.setStatus(eventPayload.getStatus());
            messagingEventDto.setMessage(eventPayload.getMessage());
            messagingEventDto.setTimestamp(eventPayload.getTimestamp());

            UserProfileInterestsMessagingAdditionalDataDto additionalData = new UserProfileInterestsMessagingAdditionalDataDto();
            additionalData.setNotifyUserIds(new HashSet<>(userProfileIdsWithInterests));
            messagingEventDto.setAdditionalData(additionalData);

            eventProducer.publishCatalogInterestsEvent(messagingEventDto);
        }
    }

}