package com.finconsgroup.itserr.marketplace.userprofile.bs.messaging;

import com.fasterxml.jackson.core.type.TypeReference;
import com.finconsgroup.itserr.marketplace.userprofile.bs.client.dto.InputAddProjectToUserProfilesDto;
import com.finconsgroup.itserr.marketplace.userprofile.bs.client.dto.InputPatchUserProfileProjectDto;
import com.finconsgroup.itserr.marketplace.userprofile.bs.client.dto.InputRemoveProjectFromUserProfilesDto;
import com.finconsgroup.itserr.marketplace.userprofile.bs.config.properties.MessagingInstitutionalPageBsProperties;
import com.finconsgroup.itserr.marketplace.userprofile.bs.config.properties.UserProfileBsConfigurationProperties;
import com.finconsgroup.itserr.marketplace.userprofile.bs.dto.OutputProjectDto;
import com.finconsgroup.itserr.marketplace.userprofile.bs.dto.OutputUserProfileDto;
import com.finconsgroup.itserr.marketplace.userprofile.bs.messaging.dto.EventDataWrapper;
import com.finconsgroup.itserr.marketplace.userprofile.bs.messaging.dto.InstitutionalPageStatusChangeNotificationData;
import com.finconsgroup.itserr.marketplace.userprofile.bs.messaging.dto.JoinRequestStatusChangeNotificationData;
import com.finconsgroup.itserr.marketplace.userprofile.bs.messaging.dto.MemberInvitationStatusChangeNotificationData;
import com.finconsgroup.itserr.marketplace.userprofile.bs.messaging.dto.MembershipChangeNotificationData;
import com.finconsgroup.itserr.marketplace.userprofile.bs.messaging.dto.MembershipRemovalNotificationData;
import com.finconsgroup.itserr.marketplace.userprofile.bs.messaging.enums.OperationType;
import com.finconsgroup.itserr.marketplace.userprofile.bs.messaging.helper.MessagingHelper;
import com.finconsgroup.itserr.marketplace.userprofile.bs.service.UserProfileService;
import com.finconsgroup.itserr.messaging.consumer.CloudEventConsumer;
import com.finconsgroup.itserr.messaging.dto.UntypedMessagingEventDto;
import io.cloudevents.CloudEvent;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Component;

import java.util.List;
import java.util.Map;
import java.util.UUID;

@Component("institutionalPageEventConsumer")
@Slf4j
public class InstitutionalPageEventConsumer extends CloudEventConsumer {

    private final UserProfileService userProfileService;
    private final MessagingHelper messagingHelper;
    private final String resourceStatusChangeEventType;

    /**
     * Constructs the InstitutionalPageEventConsumer
     *
     * @param userProfileBsConfigurationProperties the application configuration properties
     * @param userProfileService                   user profile service
     * @param messagingHelper                      messaging helper for event subscriber
     */
    public InstitutionalPageEventConsumer(UserProfileBsConfigurationProperties userProfileBsConfigurationProperties, UserProfileService userProfileService, MessagingHelper messagingHelper) {
        MessagingInstitutionalPageBsProperties messagingInstitutionalPageBsProperties = userProfileBsConfigurationProperties.getMessaging().getInstitutionalPageBs();
        this.addHandler(messagingInstitutionalPageBsProperties.getResourceStatusChangeEventType(), new TypeReference<>() {
        }, this::handleResourceStatusChange);
        this.addHandler(messagingInstitutionalPageBsProperties.getJoinRequestStatusChangeEventType(), new TypeReference<>() {
        }, this::handleJoinRequestStatusChange);
        this.addHandler(messagingInstitutionalPageBsProperties.getMemberInvitationStatusChangeEventType(), new TypeReference<>() {
        }, this::handleInvitationStatusChange);
        this.addHandler(messagingInstitutionalPageBsProperties.getMembershipChangeEventType(), new TypeReference<>() {
        }, this::handleMembershipChange);
        this.addHandler(messagingInstitutionalPageBsProperties.getMembershipRemovalEventType(), new TypeReference<>() {
        }, this::handleMembershipRemoval);
        this.userProfileService = userProfileService;
        this.messagingHelper = messagingHelper;
        this.resourceStatusChangeEventType = messagingInstitutionalPageBsProperties.getResourceStatusChangeEventType();
    }

    private void handleResourceStatusChange(UntypedMessagingEventDto resourcePayload, CloudEvent cloudEvent) {
        InstitutionalPageStatusChangeNotificationData data =
                messagingHelper.convertToInstitutionalPageStatusChangeNotificationData(resourcePayload.getAdditionalData());

        processUserGroups(cloudEvent.getType(), data.getCurrentWpLeaders(), resourcePayload, data, true);
        processUserGroups(cloudEvent.getType(), data.getCurrentMembers(), resourcePayload, data, false);
    }

    private void processUserGroups(String eventType,
                                   List<UUID> userIds,
                                   UntypedMessagingEventDto resourcePayload,
                                   InstitutionalPageStatusChangeNotificationData data,
                                   boolean isWpLeader) {
        if (!userIds.isEmpty()) {
            EventDataWrapper eventDataWrapper = new EventDataWrapper(
                    resourcePayload,
                    eventType,
                    data,
                    userIds,
                    isWpLeader);
            processApprovedEvent(eventDataWrapper);
        }
    }

    private void handleJoinRequestStatusChange(UntypedMessagingEventDto resourcePayload, CloudEvent cloudEvent) {
        JoinRequestStatusChangeNotificationData data =
                messagingHelper.convertToJoinRequestStatusChangeNotificationData(resourcePayload.getAdditionalData());

        if (!data.getRequestingUsers().isEmpty()) {
            EventDataWrapper eventDataWrapper = new EventDataWrapper(
                    resourcePayload,
                    cloudEvent.getType(),
                    data,
                    data.getRequestingUsers(),
                    false);

            processApprovedEvent(eventDataWrapper);
        }
    }

    private void handleInvitationStatusChange(UntypedMessagingEventDto resourcePayload, CloudEvent cloudEvent) {
        MemberInvitationStatusChangeNotificationData data =
                messagingHelper.convertToMemberInvitationStatusChangeNotificationData(resourcePayload.getAdditionalData());

        if (data.getInvitedUser() != null) {
            EventDataWrapper eventDataWrapper = new EventDataWrapper(
                    resourcePayload,
                    cloudEvent.getType(),
                    data,
                    List.of(data.getInvitedUser()),
                    data.getWpLeader());

            processApprovedEvent(eventDataWrapper);
        }
    }

    private void handleMembershipChange(UntypedMessagingEventDto resourcePayload, CloudEvent cloudEvent) {
        MembershipChangeNotificationData data =
                messagingHelper.convertToMembershipChangeNotificationData(resourcePayload.getAdditionalData());

        UUID userId = data.getMemberUserId();

        EventDataWrapper eventDataWrapper = new EventDataWrapper(
                resourcePayload,
                cloudEvent.getType(),
                data,
                List.of(userId),
                data.getWpLeader());

        if (Boolean.TRUE.equals(data.getWpLeader())) {
            addWpLeaderToProjectHierarchy(userId, eventDataWrapper);
        } else {
            removeWpLeaderFromProjectHierarchy(userId, eventDataWrapper);
        }
    }


    private void handleMembershipRemoval(UntypedMessagingEventDto resourcePayload, CloudEvent cloudEvent) {
        MembershipRemovalNotificationData data =
                messagingHelper.convertToMembershipRemovalNotificationData(resourcePayload.getAdditionalData());
        if (Boolean.TRUE.equals(data.getWpLeader())) {
            // Remove from entire hierarchy
            data.getHierarchyInstitutionalPages().keySet().forEach(institutionalPageId -> {
                InputRemoveProjectFromUserProfilesDto removeDto = messagingHelper.buildInputRemoveProjectFromUserProfilesDto(
                        institutionalPageId,
                        List.of(data.getMemberUserId()));

                userProfileService.removeProjectFromUserProfiles(removeDto);
            });
        } else {
            // remove from institutional page
            InputRemoveProjectFromUserProfilesDto removeDto = messagingHelper.buildInputRemoveProjectFromUserProfilesDto(
                    UUID.fromString(resourcePayload.getId()),
                    List.of(data.getMemberUserId()));

            userProfileService.removeProjectFromUserProfiles(removeDto);
        }
    }

    private void processApprovedEvent(EventDataWrapper eventDataWrapper) {
        if (!Boolean.TRUE.equals(eventDataWrapper.getApproved())) {
            log.debug("Event Request not approved for processing, id: {}, type: {}", eventDataWrapper.getResourceId(), eventDataWrapper.getEventType());
            return;
        }

        if (resourceStatusChangeEventType.equals(eventDataWrapper.getEventType())) {
            // Handle status change specifically due to CREATE/DELETE operations
            handleStatusChangeOperation(eventDataWrapper);
        } else {
            // Other events (join request, invitation), only add projects
            handleEventsAsPerMembership(eventDataWrapper);
        }
    }

    private void handleStatusChangeOperation(EventDataWrapper eventDataWrapper) {
        OperationType operationType = eventDataWrapper.getOperationType();
        if (operationType == null) {
            log.warn("No operation type in event wrapper for resourceId: {}", eventDataWrapper.getResourceId());
            return;
        }

        switch (operationType) {
            case CREATE:
                userProfileService.addProjectToUserProfiles(messagingHelper.buildInputAddProjectToUserProfilesDto(eventDataWrapper));
                break;
            case DELETE:
                userProfileService.removeProjectFromUserProfiles(messagingHelper.buildInputRemoveProjectFromUserProfilesDto(eventDataWrapper));
                break;
            default:
                log.debug("Unsupported operation type: {} for resourceId: {}", operationType, eventDataWrapper.getResourceId());
        }
    }

    private void handleEventsAsPerMembership(EventDataWrapper eventDataWrapper) {
        if (Boolean.TRUE.equals(eventDataWrapper.getWpLeader())) {
            // handling member invitation event case, since a member can also be invited as wpLeader
            eventDataWrapper.getUserIds().forEach(userId -> addWpLeaderToProjectHierarchy(userId, eventDataWrapper));
        } else {
            InputAddProjectToUserProfilesDto addDto = messagingHelper.buildInputAddProjectToUserProfilesDto(eventDataWrapper);
            userProfileService.addProjectToUserProfiles(addDto);
        }
    }

    private void addWpLeaderToProjectHierarchy(UUID userId, EventDataWrapper eventDataWrapper) {
        // retrieve all institutional pages ids in the hierarchy
        Map<UUID, String> hierarchyInstitutionalPages = eventDataWrapper.getHierarchyInstitutionalPages();
        OutputUserProfileDto userProfile = userProfileService.getById(userId);

        // Get projects user is already a member of
        List<UUID> existingProjectIds = userProfile.getProjects().stream()
                .map(OutputProjectDto::getProjectId)
                .filter(hierarchyInstitutionalPages::containsKey)
                .toList();

        // patch user profile by setting wpLeader = true
        if (!existingProjectIds.isEmpty()) {
            InputPatchUserProfileProjectDto patchDto = messagingHelper.buildInputPatchUserProfileProjectDto(
                    existingProjectIds,
                    List.of(userId),
                    true);

            userProfileService.patchUserProfileProject(patchDto);
        }

        // Add new memberships for projects user isn't part of
        List<UUID> projectIdsToAdd = hierarchyInstitutionalPages.keySet().stream()
                .filter(projectId -> !existingProjectIds.contains(projectId))
                .toList();

        for (UUID projectId : projectIdsToAdd) {
            String projectName = hierarchyInstitutionalPages.get(projectId);
            InputAddProjectToUserProfilesDto inputAddProjectToUserProfilesDto = messagingHelper.buildInputAddProjectToUserProfilesDto(
                    projectId,
                    projectName,
                    userId,
                    eventDataWrapper.getRootInstitutionalPageId(),
                    eventDataWrapper.getRootInstitutionalPageName(),
                    true);

            userProfileService.addProjectToUserProfiles(inputAddProjectToUserProfilesDto);
        }
    }

    private void removeWpLeaderFromProjectHierarchy(UUID userId, EventDataWrapper eventDataWrapper) {
        // retrieve all institutional pages ids in the hierarchy
        OutputUserProfileDto userProfile = userProfileService.getById(userId);
        List<UUID> projectInHierarchy = userProfile.getProjects().stream()
                .filter(project -> project.getRootProjectId().equals(eventDataWrapper.getRootInstitutionalPageId()))
                .map(OutputProjectDto::getProjectId)
                .toList();

        for (UUID projectId : projectInHierarchy) {
            if (projectId.equals(eventDataWrapper.getRootInstitutionalPageId())) {
                // If the institutional page is root then demote from wpLeader but keep membership
                InputPatchUserProfileProjectDto patchDto = messagingHelper.buildInputPatchUserProfileProjectDto(
                        List.of(projectId), List.of(userId), false);
                userProfileService.patchUserProfileProject(patchDto);
            } else {
                // Remove the user from child projects
                InputRemoveProjectFromUserProfilesDto inputRemoveProjectFromUserProfilesDto = InputRemoveProjectFromUserProfilesDto.builder()
                        .userIds(List.of(userId))
                        .projectId(projectId)
                        .build();
                userProfileService.removeProjectFromUserProfiles(inputRemoveProjectFromUserProfilesDto);
            }
        }
    }
}