package com.finconsgroup.itserr.marketplace.userprofile.bs.service;

import com.finconsgroup.itserr.marketplace.core.web.exception.WP2DuplicateResourceException;
import com.finconsgroup.itserr.marketplace.core.web.exception.WP2ResourceNotFoundException;
import com.finconsgroup.itserr.marketplace.core.web.exception.WP2ValidationException;
import com.finconsgroup.itserr.marketplace.userprofile.bs.bean.UserProfileInterests;
import com.finconsgroup.itserr.marketplace.userprofile.bs.client.dto.InputAddProjectToUserProfilesDto;
import com.finconsgroup.itserr.marketplace.userprofile.bs.client.dto.InputAdminPatchUserProfileDto;
import com.finconsgroup.itserr.marketplace.userprofile.bs.client.dto.InputPatchUserProfileProjectDto;
import com.finconsgroup.itserr.marketplace.userprofile.bs.client.dto.InputRemoveProjectFromUserProfilesDto;
import com.finconsgroup.itserr.marketplace.userprofile.bs.client.dto.OutputAdminPatchUserProfileDto;
import com.finconsgroup.itserr.marketplace.userprofile.bs.dto.InputUpdateUserProfileDto;
import com.finconsgroup.itserr.marketplace.userprofile.bs.dto.OutputEndorsementAcknowledgementDto;
import com.finconsgroup.itserr.marketplace.userprofile.bs.dto.OutputUserProfileDto;
import com.finconsgroup.itserr.marketplace.userprofile.bs.dto.InputUserProfileDto;
import com.finconsgroup.itserr.marketplace.userprofile.bs.dto.OutputUserSettingDto;
import com.finconsgroup.itserr.marketplace.userprofile.bs.dto.WorkspaceDto;
import com.finconsgroup.itserr.marketplace.userprofile.bs.dto.FolderDto;
import org.springframework.lang.NonNull;
import org.springframework.lang.Nullable;
import org.springframework.web.multipart.MultipartFile;

import java.io.IOException;
import java.net.URI;

import java.util.List;
import java.util.Set;
import java.util.UUID;

/**
 * Service interface for handling business logic related to user profiles.
 * <p>
 * Provides operations for uploading profile-related resources, such as
 * profile images and curriculum vitae (CV).
 * </p>
 *
 * <p>Example usage:</p>
 * <pre>
 * URI imageUri = userProfileService.uploadImage(imageFile);
 * URI cvUri    = userProfileService.uploadCv(cvFile);
 * </pre>
 */
public interface UserProfileService {

    /**
     * Uploads a user's profile image to the dedicated storage.
     *
     * @param image the image file (PNG, JPEG, etc.) to upload; must not be {@code null}
     * @return a {@link URI} pointing to the uploaded image resource; never {@code null}
     * @throws org.springframework.web.multipart.MaxUploadSizeExceededException if the file exceeds the configured size limit
     * @throws IllegalArgumentException                                         if the file is empty or not a valid image
     */
    @NonNull
    URI uploadImage(@NonNull MultipartFile image);

    /**
     * Uploads a user's curriculum vitae (CV) to the dedicated storage.
     *
     * @param cvFile the CV file (PDF, DOC, etc.) to upload; must not be {@code null}
     * @return a {@link URI} pointing to the uploaded CV resource; never {@code null}
     * @throws IOException                                                      if an I/O error occurs while saving the file
     * @throws org.springframework.web.multipart.MaxUploadSizeExceededException if the file exceeds the configured size limit
     * @throws IllegalArgumentException                                         if the file is empty or not a valid document
     */
    @NonNull
    URI uploadCv(@NonNull MultipartFile cvFile);

    /**
     * Creates a new userprofile using the provided input data.
     *
     * @param inputUserProfileDto the input DTO containing userprofile details
     * @return an OutputUserProfileDto representing the created userprofile
     */
    @NonNull
    OutputUserProfileDto create(@NonNull UUID userId,
                                @NonNull InputUserProfileDto inputUserProfileDto);

    /**
     * Retrieves an userprofile by id
     *
     * @param userId the id of the userprofile to update
     * @return the found {@link OutputUserProfileDto} and HTTP status 200 (OK)
     */
    @NonNull
    OutputUserProfileDto getById(@NonNull UUID userId);

    /**
     * Updates an existing userprofile by id with the provided data.
     *
     * @param userId                    the id of the userprofile to update
     * @param inputUpdateUserProfileDto the input DTO containing updated userprofile details
     * @return an OutputUserProfileDto representing the updated userprofile
     */
    @NonNull
    OutputUserProfileDto updateById(@NonNull UUID userId, @NonNull InputUpdateUserProfileDto inputUpdateUserProfileDto);

    /**
     * get a users workspace folder
     */
    @NonNull
    WorkspaceDto getUserWorkspace();

    /**
     * get a user profile folder
     *
     * @param folderId userprofile folder id
     */
    @NonNull
    FolderDto getUserProfileFolderId(@NonNull String folderId);

    /**
     * get a user profile folder
     *
     * @param parentFolderId userprofile folder id
     */
    @NonNull
    String createUserProfileFolder(@NonNull String parentFolderId);

    /**
     * update a user language
     *
     * @param language userprofile language
     * @return OutputUserSettingDto that represents a user setting
     */
    @NonNull
    OutputUserSettingDto updateLanguage(String language);

    /**
     * Get the user profiles matching the provided role.
     *
     * @param roleName the role name to search for
     * @return the list of matching {@link OutputUserProfileDto}
     */
    @NonNull
    List<OutputUserProfileDto> getUserProfilesByRole(String roleName);

    /**
     * add projects to existing userprofile for given ids.
     *
     * @param inputAddProjectToUserProfilesDto contains the list of userIds and object containing updated userprofile project details
     */
    @NonNull
    void addProjectToUserProfiles(InputAddProjectToUserProfilesDto inputAddProjectToUserProfilesDto);

    /**
     * removes projects for existing userprofile for given ids.
     *
     * @param inputRemoveProjectFromUserProfilesDto contains the list of userIds and object containing updated userprofile project details
     */
    @NonNull
    void removeProjectFromUserProfiles(InputRemoveProjectFromUserProfilesDto inputRemoveProjectFromUserProfilesDto);

    /**
     * update project details for existing userprofile for given ids.
     *
     * @param inputPatchUserProfileProjectDto contains the list of userIds and patch updates for userprofile project details
     */
    @NonNull
    void patchUserProfileProject(InputPatchUserProfileProjectDto inputPatchUserProfileProjectDto);

    /**
     * update the user profile info.
     *
     * @param profileId                     id of the user profile
     * @param inputAdminPatchUserProfileDto which contains information that needs to be patched for an existing entity
     * @return the found {@link OutputAdminPatchUserProfileDto} and HTTP status 200 (OK)
     */
    @NonNull
    OutputAdminPatchUserProfileDto patchUserProfile(UUID profileId, InputAdminPatchUserProfileDto inputAdminPatchUserProfileDto);

    /**
     * activate/deactivate a user profile.
     */
    @NonNull
    void processUserProfileStatusChange(UUID userId, Boolean active);

    /**
     * Publish resource messages for all user profiles
     */
    void publishAllProfilesMessages();

    /**
     * Publish the events for user profiles who have matching interests for the provided strings to check.
     *
     * @param userProfileInterests the user profile interests object built based on the source
     * @param stringsToCheck       the strings to check for matching interests
     * @param userIdsToExclude     the user ids to exclude from sending the notification
     */
    void publishInterestsEvent(@NonNull UserProfileInterests userProfileInterests,
                               @Nullable List<String> stringsToCheck,
                               @NonNull Set<UUID> userIdsToExclude);

    /**
     * update hide panel flag value
     *
     * @param hidePanel the user preferred value (true, false)
     * @return the updated user flag value
     */
    @NonNull
    Boolean updateHidePanel(Boolean hidePanel);

    /**
     * Endorses a specific expertise of a user profile.
     *
     * @param profileId   the id of the endorsed user
     * @param expertiseId the id of the expertise
     * @throws WP2ResourceNotFoundException  if no user profile is found for the given user id
     * @throws WP2DuplicateResourceException if user has already endorsed to the program
     * @throws WP2ValidationException        if user tries to endorse his own expertise
     */
    @NonNull
    OutputEndorsementAcknowledgementDto addEndorsement(UUID profileId, UUID expertiseId);

    /**
     * Delete an existing endorsement.
     *
     * @param profileId   the id of the endorsed user
     * @param expertiseId the id of the expertise
     * @throws WP2ResourceNotFoundException if no user profile is found for the given user id
     */
    @NonNull
    void removeEndorsement(UUID profileId, UUID expertiseId);
}
