package eu.dnetlib.data.transformation.manager;

import java.io.IOException;
import java.io.StringReader;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.annotation.Resource;
import javax.xml.ws.wsaddressing.W3CEndpointReference;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.dom4j.Document;
import org.dom4j.DocumentException;
import org.dom4j.Node;
import org.dom4j.io.SAXReader;
import org.springframework.beans.factory.annotation.Required;

import com.google.common.collect.Lists;

import eu.dnetlib.data.information.DataSink;
import eu.dnetlib.data.information.DataSinkResolver;
import eu.dnetlib.data.information.DataSource;
import eu.dnetlib.data.information.DataSourceResolver;
import eu.dnetlib.enabling.is.lookup.rmi.ISLookUpException;
import eu.dnetlib.enabling.resultset.MappedResultSetFactory;
import eu.dnetlib.enabling.resultset.XSLTMappedResultSetFactory;
import eu.dnetlib.enabling.tools.SplittedQueryExecutor;
import eu.dnetlib.enabling.tools.blackboard.AbstractBlackboardNotificationHandler;
import eu.dnetlib.enabling.tools.blackboard.BlackboardJob;
import eu.dnetlib.enabling.tools.blackboard.BlackboardServerHandler;
import groovy.lang.GroovyShell;
import groovy.util.GroovyScriptEngine;

public class TransformationManagerServiceMockNotificationHandler extends AbstractBlackboardNotificationHandler<BlackboardServerHandler> {

	private static final Log log = LogFactory.getLog(TransformationManagerServiceMockNotificationHandler.class); // NOPMD by marko on 11/24/08 5:02 PM

	/**
	 * splitted query executor utility.
	 */
	@Resource
	private SplittedQueryExecutor splittedQueryExecutor;

	/**
	 * Used to take the resultset EPR from a data source.
	 */
	private DataSourceResolver dataSourceResolver;

	/**
	 * Used to take the resultset EPR from a data source.
	 */
	private DataSinkResolver dataSinkResolver;

	/**
	 * used to transform the records using Groovy.
	 */
	private MappedResultSetFactory mappedResultSetFactory;

	/**
	 * used to transform the records using XSLT.
	 */
	private XSLTMappedResultSetFactory xsltResultSetFactory;

	/**
	 * default xslt to apply to each record.
	 */
	private org.springframework.core.io.Resource xslt;

	@Override
	protected void processJob(BlackboardJob job) {
		log.info("PROCESSING MOCK TMS JOB " + job);
		log.info("ACTION " + job.getAction());

		if (job.getAction().equals("TRANSFORM")) {

			final String tdsId = job.getParameters().get("id");
			final DataSource dataSource = getDataSource(tdsId);
			final DataSink dataSink = dataSinkResolver.resolve(job.getParameters().get("dataSink"));
			final Map<String, String> params = createParameters(tdsId);

			String query = "for $x in collection('/db/DRIVER/TransformationDSResources/TransformationDSResourceType') "
					+ "for $y in collection('/db/DRIVER/GroovyProcessingDSResource/GroovyProcessingDSResourceType') " + "where  "
					+ "$x//RESOURCE_IDENTIFIER/@value = '" + tdsId + "' " + "and $x//TRANSFORMATION_RULE_DS_IDENTIFIER = $y//RESOURCE_IDENTIFIER/@value "
					+ "return concat($y//GROOVY_CLASSPATH, ':-:', $y//GROOVY_DNETCLASS)";

			List<Map<String, String>> list = Lists.newArrayList(splittedQueryExecutor.query(query, "groovyClasspath", "groovyDnetClass"));

			// COPY from source to sink
			try {

				W3CEndpointReference epr = null;
				if (list.size() == 1) {
					epr = transformGroovy(dataSource.retrieve(), list.get(0).get("groovyClasspath"), list.get(0).get("groovyDnetClass"), params);
				} else {
					String queryForXSLT = "for $x in collection('/db/DRIVER/TransformationDSResources/TransformationDSResourceType') "
							+ "for $y in collection('/db/DRIVER/TransformationRuleDSResources/TransformationRuleDSResourceType') " + "where  "
							+ "$x//RESOURCE_IDENTIFIER/@value = '" + tdsId + "' "
							+ "and $x//TRANSFORMATION_RULE_DS_IDENTIFIER = $y//RESOURCE_IDENTIFIER/@value "
							+ "return $y//CODE/*[local-name()='stylesheet']";

					List<String> results = splittedQueryExecutor.getLookupLocator().getService().quickSearchProfile(queryForXSLT);
					if (results.size() == 0)
						epr = transformXslt(dataSource.retrieve(), params);
					else
						epr = transformXslt(dataSource.retrieve(), params, results.get(0));
				}
				dataSink.store(epr);
			} catch (Exception e) {
				getBlackboardHandler().failed(job, e);
				throw new IllegalStateException("cannot copy from source to sink", e);
			}
		}
		getBlackboardHandler().done(job);
	}

	private W3CEndpointReference transformGroovy(W3CEndpointReference source, String groovyClasspath, String groovyDnetClass, Map<String, String> params)
			throws ClassNotFoundException, IOException {

		GroovyScriptEngine gse = new GroovyScriptEngine(groovyClasspath);
		gse.getGroovyClassLoader().loadClass(groovyDnetClass);
		log.info("***********************************************");
		log.info("Loaded Groovy classes:");
		for (Class<?> c : gse.getGroovyClassLoader().getLoadedClasses()) {
			log.info(c.getCanonicalName());
		}
		log.info("***********************************************");
		GroovyShell groovyShell = new GroovyShell(gse.getGroovyClassLoader());

		Object go = groovyShell.evaluate("new " + groovyDnetClass + "()");
		if (go instanceof GroovyUnaryFunction) {
			GroovyUnaryFunction groovyUnaryFunction = (GroovyUnaryFunction) go;
			groovyUnaryFunction.setParams(params);
			return mappedResultSetFactory.createMappedResultSet(source, groovyUnaryFunction);
		} else {
			throw new RuntimeException("Groovy object " + go + " is not supported");
		}
	}

	private W3CEndpointReference transformXslt(W3CEndpointReference source, Map<String, String> parameters) {
		return xsltResultSetFactory.createMappedResultSet(source, xslt, parameters);
	}

	private W3CEndpointReference transformXslt(W3CEndpointReference source, Map<String, String> parameters, String xsltCode) {
		log.debug("XSLT CODE:");
		log.debug(xsltCode);
		return xsltResultSetFactory.createMappedResultSet(source, xsltCode, parameters);
	}

	protected DataSource getDataSource(String tdsId) {
		String query = "//RESOURCE_PROFILE[.//RESOURCE_IDENTIFIER/@value='" + tdsId + "']//DATA_SOURCE/string()";
		return dataSourceResolver.resolve(splittedQueryExecutor.queryFirst(query));
	}

	protected Map<String, String> createParameters(String tdsId) {
		HashMap<String, String> theMap = new HashMap<String, String>();
		String query = "//RESOURCE_PROFILE[.//RESOURCE_IDENTIFIER/@value=(//RESOURCE_PROFILE[.//RESOURCE_IDENTIFIER/@value='" + tdsId
				+ "']//REPOSITORY_SERVICE_IDENTIFIER/text())]";
		try {
			String result = splittedQueryExecutor.getLookupLocator().getService().getResourceProfileByQuery(query);
			Document doc = new SAXReader().read(new StringReader(result));

			theMap.put("repositoryCountry", doc.selectSingleNode("//COUNTRY").getText());
			theMap.put("repositoryName", doc.selectSingleNode("//OFFICIAL_NAME").getText());
			theMap.put("repositoryLink", doc.selectSingleNode("//REPOSITORY_WEBPAGE").getText());
			theMap.put("repositoryInstitution", doc.selectSingleNode("//REPOSITORY_INSTITUTION").getText());
			//TODO  this is a change usefull only for Openaire
			// 		MUST BE REFACTORED
			Node datasourceId = doc.selectSingleNode("//FIELD[./key='OpenAireDataSourceId']/value");
			if (datasourceId!=null)
				theMap.put("dataSourceId", datasourceId.getText());			
			//END OF CODE TO BE REFACTORED
			
			

		} catch (ISLookUpException e) {
			throw new RuntimeException(e);
		} catch (DocumentException e) {
			throw new RuntimeException(e);
		}
		return theMap;
	}

	public DataSourceResolver getDataSourceResolver() {
		return dataSourceResolver;
	}

	@Required
	public void setDataSourceResolver(DataSourceResolver dataSourceResolver) {
		this.dataSourceResolver = dataSourceResolver;
	}

	public DataSinkResolver getDataSinkResolver() {
		return dataSinkResolver;
	}

	@Required
	public void setDataSinkResolver(DataSinkResolver dataSinkResolver) {
		this.dataSinkResolver = dataSinkResolver;
	}

	public XSLTMappedResultSetFactory getXsltResultSetFactory() {
		return xsltResultSetFactory;
	}

	@Required
	public void setXsltResultSetFactory(XSLTMappedResultSetFactory xsltResultSetFactory) {
		this.xsltResultSetFactory = xsltResultSetFactory;
	}

	public org.springframework.core.io.Resource getXslt() {
		return xslt;
	}

	@Required
	public void setXslt(org.springframework.core.io.Resource xslt) {
		this.xslt = xslt;
	}

	public MappedResultSetFactory getMappedResultSetFactory() {
		return mappedResultSetFactory;
	}

	@Required
	public void setMappedResultSetFactory(MappedResultSetFactory mappedResultSetFactory) {
		this.mappedResultSetFactory = mappedResultSetFactory;
	}

}
