package eu.dnetlib.data.information.oai.publisher;

import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Locale;
import java.util.Map;
import java.util.Map.Entry;
import java.util.TimeZone;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.apache.commons.lang.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.beans.factory.annotation.Required;
import org.springframework.ui.ModelMap;
import org.springframework.web.bind.annotation.ModelAttribute;
import org.springframework.web.bind.annotation.RequestMapping;

import eu.dnetlib.data.information.oai.publisher.core.AbstractOAICore;
import eu.dnetlib.data.information.oai.publisher.info.ListRecordsInfo;

/**
 * OAI Servlet.
 * 
 * @author michele
 * 
 */
public final class OAIController {

	public enum OAI_VERBS {
		IDENTIFY, LIST_IDENTIFIERS, LIST_RECORDS, LIST_METADATA_FORMATS, LIST_SETS, GET_RECORD, UNSUPPORTED_VERB;

		public static OAI_VERBS getVerb(final String theVerb) {
			if (StringUtils.isBlank(theVerb)) return UNSUPPORTED_VERB;
			if (theVerb.equalsIgnoreCase("Identify")) return IDENTIFY;
			if (theVerb.equalsIgnoreCase("ListIdentifiers")) return LIST_IDENTIFIERS;
			if (theVerb.equalsIgnoreCase("ListRecords")) return LIST_RECORDS;
			if (theVerb.equalsIgnoreCase("ListMetadataFormats")) return LIST_METADATA_FORMATS;
			if (theVerb.equalsIgnoreCase("ListSets")) return LIST_SETS;
			if (theVerb.equalsIgnoreCase("GetRecord")) return GET_RECORD;
			if (theVerb.equalsIgnoreCase("listidentifiers")) return LIST_IDENTIFIERS;
			return UNSUPPORTED_VERB;

		}
	}

	public enum DELETED_SUPPORT {
		NO, TRANSIENT, PERSISTENT;
	}

	/**
	 * logger.
	 */
	private static final Log log = LogFactory.getLog(OAIController.class); // NOPMD by marko on 11/24/08 5:02 PM

	/**
	 * Default content type.
	 */
	private static final String DEFAULT_CONTENT_TYPE = "text/xml;charset=utf-8";

	/**
	 * optional base url. If present it overrides the X-Forwarded-Url.
	 */
	private String baseUrl;

	/**
	 * forwarded url header name, default "X-Forwarded-Url".
	 */
	private String forwardedUrlHeaderName = "X-Forwarded-Url";

	/**
	 * error messages map.
	 */
	private static Map<String, String> errMessages = new HashMap<String, String>();

	/**
	 * OAI servlet core. Most of the logic is here
	 */
	private AbstractOAICore core;

	private String repoName = "Driver Service for supporting Open Archive Initiative requests";
	private String repoEmail = "artini@isti.cnr.it";
	private String earliestDatestamp = "1970-01-01";
	private DELETED_SUPPORT deletedRecordSupport;
	private String dateGranularity = "YYYY-MM-DD";

	static {
		OAIController.errMessages
				.put("badArgument",
						"The request includes illegal arguments, is missing required arguments, includes a repeated argument, or values for arguments have an illegal syntax.");
		OAIController.errMessages.put("badVerb",
				"Value of the verb argument is not a legal OAI-PMH verb, the verb argument is missing, or the verb argument is repeated.");
		OAIController.errMessages.put("cannotDisseminateFormat",
				"The metadata format identified by the value given for the metadataPrefix argument is not supported by the item or by the repository.");
		OAIController.errMessages.put("idDoesNotExist", "The value of the identifier argument is unknown or illegal in this repository.");
		OAIController.errMessages.put("noMetadataFormats", "There are no metadata formats available for the specified item.");
		OAIController.errMessages.put("noSetHierarchy", "The repository does not support sets.");
		OAIController.errMessages.put("noRecordsMatch",
				"The combination of the values of the from, until, set and metadataPrefix arguments results in an empty list.");
		OAIController.errMessages.put("badResumptionToken", "The value of the resumptionToken argument is invalid or expired.");
	}

	@RequestMapping("oai.do")
	public String oai(final ModelMap map, final HttpServletRequest request, final HttpServletResponse response) throws Exception {
		response.setContentType(OAIController.DEFAULT_CONTENT_TYPE);

		try {
			final Map<String, String> params = cleanParameters(request.getParameterMap());
			if (params == null) return oaiError("badArgument", map);
			String theVerb = params.get("verb");
			OAI_VERBS requestedVerb = OAI_VERBS.getVerb(theVerb);
			switch (requestedVerb) {
			case IDENTIFY:
				return oaiIdentify(params, map);
			case LIST_METADATA_FORMATS:
				return oaiListMetadataFormats(params, map);
			case LIST_SETS:
				return oaiListSets(params, map);
			case GET_RECORD:
				return oaiGetRecord(params, map);
			case LIST_IDENTIFIERS:
				return oaiListIdentifiersOrRecords(params, map);
			case LIST_RECORDS:
				return oaiListIdentifiersOrRecords(params, map);
			default:
				return oaiError("badVerb", map);
			}
		} catch (final Exception e) {
			OAIController.log.error("ERROR", e);
			return oaiError(e, map);
		}
	}

	private Map<String, String> cleanParameters(final Map<?, ?> startParams) {
		final HashMap<String, String> params = new HashMap<String, String>();
		final Iterator<?> iter = startParams.entrySet().iterator();
		while (iter.hasNext()) {
			final Entry<?, ?> entry = (Entry<?, ?>) iter.next();
			final String key = entry.getKey().toString();
			final String[] arr = (String[]) entry.getValue();
			if (arr.length == 0) return null;
			final String value = arr[0];
			if (key.equals("verb")) {
				params.put("verb", value);
			} else if (key.equals("from")) {
				params.put("from", value);
			} else if (key.equals("until")) {
				params.put("until", value);
			} else if (key.equals("metadataPrefix")) {
				params.put("metadataPrefix", value);
			} else if (key.equals("identifier")) {
				params.put("identifier", value);
			} else if (key.equals("set")) {
				params.put("set", value);
			} else if (key.equals("resumptionToken")) {
				params.put("resumptionToken", value);
			} else return null;
		}
		return params;
	}

	private String oaiIdentify(final Map<String, String> params, final ModelMap map) throws Exception {
		String verb = null;
		if (params.containsKey("verb")) {
			verb = params.get("verb");
			params.remove("verb");
		}
		if (params.entrySet().size() > 0) return oaiError("badArgument", verb, map);
		return "oai/OAI_Identify";
	}

	private String oaiGetRecord(final Map<String, String> params, final ModelMap map) throws Exception {
		String verb = null;
		String prefix = null;
		String identifier = null;
		if (params.containsKey("verb")) {
			verb = params.get("verb");
			params.remove("verb");
		}
		if (params.containsKey("metadataPrefix")) {
			prefix = params.get("metadataPrefix");
			params.remove("metadataPrefix");
		} else return oaiError("badArgument", verb, map);
		if (params.containsKey("identifier")) {
			identifier = params.get("identifier");
			params.remove("identifier");
		} else return oaiError("badArgument", verb, map);
		if (params.entrySet().size() > 0) return oaiError("badArgument", verb, map);

		map.addAttribute("record", core.getInfoRecord(identifier, prefix));

		return "oai/OAI_GetRecord";
	}

	private String oaiListIdentifiersOrRecords(final Map<String, String> params, final ModelMap map) throws Exception {
		OAI_VERBS verb = null;
		String metadataPrefix = null;
		String resumptionToken = null;
		String set = null;
		String from = null;
		String until = null;

		if (params.containsKey("verb")) {
			verb = OAI_VERBS.getVerb(params.get("verb"));
			params.remove("verb");
		}

		if (params.containsKey("resumptionToken")) {
			resumptionToken = params.get("resumptionToken");
			params.remove("resumptionToken");
		} else {
			if (params.containsKey("metadataPrefix")) {
				metadataPrefix = params.get("metadataPrefix");
				params.remove("metadataPrefix");
			} else return oaiError("badArgument", verb.toString(), map);
			if (params.containsKey("from")) {
				from = params.get("from");
				params.remove("from");
			}
			if (params.containsKey("until")) {
				until = params.get("until");
				params.remove("until");
			}
			if (params.containsKey("set")) {
				set = params.get("set");
				params.remove("set");
			}
		}
		if (params.entrySet().size() > 0) return oaiError("badArgument", verb.toString(), map);

		boolean onlyIdentifiers = true;
		if (verb == OAI_VERBS.LIST_RECORDS) {
			onlyIdentifiers = false;
		}

		ListRecordsInfo infos;
		if (StringUtils.isBlank(resumptionToken)) {
			infos = core.listRecords(onlyIdentifiers, metadataPrefix, set, from, until);
		} else {
			infos = core.listRecords(onlyIdentifiers, resumptionToken);
		}

		map.addAttribute("info", infos);

		if (verb == OAI_VERBS.LIST_RECORDS) return "oai/OAI_ListRecords";

		return "oai/OAI_ListIdentifiers";
	}

	private String oaiListSets(final Map<String, String> params, final ModelMap map) throws Exception {
		String verb = null;
		if (params.containsKey("verb")) {
			verb = params.get("verb");
			params.remove("verb");
		}
		if (params.entrySet().size() > 0) return oaiError("badArgument", verb, map);
		map.addAttribute("sets", core.listSets());
		return "oai/OAI_ListSets";
	}

	private String oaiListMetadataFormats(final Map<String, String> params, final ModelMap map) throws Exception {
		String id = null;
		String verb = null;
		if (params.containsKey("verb")) {
			verb = params.get("verb");
			params.remove("verb");
		}
		if (params.containsKey("identifier")) {
			id = params.get("identifier");
			params.remove("identifier");
		}
		if (params.entrySet().size() > 0) return oaiError("badArgument", verb, map);

		map.addAttribute("formats", core.listMetadataFormats());
		if (id != null) {
			map.addAttribute("identifier", id);
			return "oai/OAI_ListMetadataFormats_withid";
		}
		return "oai/OAI_ListMetadataFormats";
	}

	private String oaiError(final String errCode, final String verb, final ModelMap map) throws Exception {
		if ((verb == null) || (verb.equals(""))) return oaiError(errCode, map);
		map.addAttribute("verb", verb);
		map.addAttribute("errcode", errCode);
		map.addAttribute("errmsg", OAIController.errMessages.get(errCode));
		return "oai/OAI_Error";
	}

	private String oaiError(final String errCode, final ModelMap map) throws Exception {
		map.addAttribute("errcode", errCode);
		map.addAttribute("errmsg", OAIController.errMessages.get(errCode));
		return "oai/OAI_Error_noverb";
	}

	private String oaiError(final Exception e, final ModelMap map) throws Exception {
		map.addAttribute("errcode", "InternalException");
		map.addAttribute("errmsg", e.getMessage());
		return "oai/OAI_Error_noverb";
	}

	/*
	 * Default Attribute
	 */
	@ModelAttribute("url")
	public String url(final HttpServletRequest request) {
		String forwardedUrl = request.getHeader(forwardedUrlHeaderName);
		if (baseUrl != null) return baseUrl;
		else if (forwardedUrl != null) return forwardedUrl;
		else return request.getRequestURL() + "";
	}

	/*
	 * Default Attribute
	 */
	@ModelAttribute("date")
	public String date() {
		SimpleDateFormat formatter = new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ssZ", Locale.US);
		formatter.setTimeZone(TimeZone.getTimeZone("GMT"));
		String date = formatter.format(new Date());
		return date.replace("+0000", "Z");
	}

	@ModelAttribute("repoName")
	public String getRepoName() {
		return repoName;
	}

	@ModelAttribute("email")
	public String getRepoEmail() {
		return repoEmail;
	}

	@ModelAttribute("earliestDatestamp")
	public String getEarliestDatestamp() {
		return earliestDatestamp;
	}

	public void setRepoName(final String repoName) {
		this.repoName = repoName;
	}

	public void setRepoEmail(final String repoEmail) {
		this.repoEmail = repoEmail;
	}

	public void setEarliestDatestamp(final String earliestDatestamp) {
		this.earliestDatestamp = earliestDatestamp;
	}

	public String getBaseUrl() {
		return baseUrl;
	}

	public void setBaseUrl(final String baseUrl) {
		this.baseUrl = baseUrl;
	}

	public AbstractOAICore getCore() {
		return core;
	}

	@Required
	public void setCore(final AbstractOAICore core) {
		this.core = core;
	}

	public String getForwardedUrlHeaderName() {
		return forwardedUrlHeaderName;
	}

	public void setForwardedUrlHeaderName(final String forwardedUrlHeaderName) {
		this.forwardedUrlHeaderName = forwardedUrlHeaderName;
	}

	@ModelAttribute("deletedRecord")
	public DELETED_SUPPORT getDeletedRecordSupport() {
		return deletedRecordSupport;
	}

	public void setDeletedRecordSupport(final DELETED_SUPPORT deletedRecordSupport) {
		this.deletedRecordSupport = deletedRecordSupport;
	}

	@ModelAttribute("granularity")
	public String getDateGranularity() {
		return dateGranularity;
	}

	public void setDateGranularity(final String dateGranularity) {
		this.dateGranularity = dateGranularity;
	}
}
