package eu.dnetlib.data.information.oai.publisher.store;

import java.util.List;

import eu.dnetlib.data.information.oai.publisher.OaiPublisherException;

public interface PublisherStoreDAO<X extends PublisherStore<T>, T extends Cursor> {

	/**
	 * Lists all PublisherStore.
	 * 
	 * @return a List of PublisherStore instances.
	 */
	List<X> listPublisherStores();

	/**
	 * Gets the store with the given identifier.
	 * 
	 * @param storeId
	 *            identifier of the store to retrieve
	 * @return a PublisherStore instance or null if there is no store with the given id.
	 */
	X getStore(String storeId);

	/**
	 * Gets the store with the given properties.
	 * 
	 * @param mdfName
	 *            name of the metadata format
	 * @param mdfInterpretation
	 *            name of the metadata interpretation
	 * @param mdfLayout
	 *            name of the metadata layout
	 * @return a PublisherStore instance or null if there is no store with the given properties.
	 */
	X getStore(String mdfName, String mdfInterpretation, String mdfLayout);

	/**
	 * Gets the store to be used as source to deliver records with the given metadata prefix.
	 * 
	 * @param targetMetadataPrefix
	 *            prefix of the metadata format deliverable through this store
	 * @return a PublisherStore instance or null if there is no store serving the given metadata prefix.
	 */
	X getStoreFor(String targetMetadataPrefix);

	/**
	 * Create a PublisherStore with the given properties.
	 * 
	 * @param mdfName
	 *            name of the metadata format
	 * @param mdfInterpretation
	 *            name of the metadata interpretation
	 * @param mdfLayout
	 *            name of the metadata layout
	 * @return a PublisherStore instance whose identifier is automatically generated.
	 * @throws OaiPublisherException
	 *             if there is already another PublisherStore with the given metadata format name, layout and interpretation.
	 */
	X createStore(String mdfName, String mdfInterpretation, String mdfLayout) throws OaiPublisherException;

	/**
	 * Deletes the store with the given identifier.
	 * 
	 * @param storeId
	 *            id of the store to delete
	 * @return true if the store was deleted successfully, false otherwise (e.g., a store with the given id does not exist).
	 */
	boolean deleteStore(String storeId);

	/**
	 * Deletes the store with the given properties.
	 * 
	 * @param mdfName
	 *            name of the metadata format
	 * @param mdfInterpretation
	 *            name of the metadata interpretation
	 * @param mdfLayout
	 *            name of the metadata layout
	 * @return true if the store was deleted successfully, false otherwise (e.g., a store with the given properties does not exist).
	 */
	boolean deleteStore(String mdfName, String mdfInterpretation, String mdfLayout);

}
