package eu.dnetlib.data.information.oai.publisher.core;

import java.util.List;

import eu.dnetlib.data.information.oai.publisher.OaiPublisherException;
import eu.dnetlib.data.information.oai.publisher.info.ListDocumentsInfo;
import eu.dnetlib.data.information.oai.publisher.info.ListRecordsInfo;
import eu.dnetlib.data.information.oai.publisher.info.MDFInfo;
import eu.dnetlib.data.information.oai.publisher.info.RecordInfo;
import eu.dnetlib.data.information.oai.publisher.info.ResumptionToken;
import eu.dnetlib.data.information.oai.publisher.info.SetInfo;

/**
 * Core OAI servlet implementation.
 * 
 * @author michele
 * 
 */
abstract public class AbstractOAICore {

	/**
	 * page size.
	 */
	protected int pageSize = 100;

	/**
	 * Returns informations about a record. It contains basic information about a record, such as "prefix", "identifier", "datestamp",
	 * "setspec", "metadata".
	 * 
	 * @param identifier
	 *            record identifier
	 * @param prefix
	 *            metadata prefix
	 * @return a recordInfo instance
	 * @throws OaiPublisherException
	 *             could happen
	 */
	public RecordInfo getInfoRecord(final String identifier, final String prefix) throws OaiPublisherException {
		MDFInfo mdf = obtainMDFInfo(prefix);
		return getRecordById(mdf, identifier);
	}

	abstract protected RecordInfo getRecordById(final MDFInfo mdf, final String id) throws OaiPublisherException;

	public abstract List<? extends SetInfo> listSets() throws OaiPublisherException;

	abstract public List<MDFInfo> listMetadataFormats() throws OaiPublisherException;

	abstract protected ListDocumentsInfo getDocuments(final boolean onlyIdentifiers,
			final String set,
			final String metadataPrefix,
			final String from,
			final String until) throws OaiPublisherException;

	abstract protected ListDocumentsInfo getDocuments(final boolean onlyIdentifiers, final String resumptionToken) throws OaiPublisherException;

	protected abstract MDFInfo obtainMDFInfo(final String metadataPrefix) throws OaiPublisherException;

	/**
	 * List records.
	 * 
	 * @param onlyIdentifiers
	 *            only return record identifiers
	 * @param metadataPrefix
	 *            metadata prefix
	 * @param set
	 *            set name
	 * @param from
	 *            from date
	 * @param until
	 *            to date
	 * @return ListRecordsInfo
	 * @throws OaiPublisherException
	 *             could happen
	 */
	public ListRecordsInfo listRecords(final boolean onlyIdentifiers, final String metadataPrefix, final String set, final String from, final String until)
			throws OaiPublisherException {

		final ListRecordsInfo res = new ListRecordsInfo();

		if (from != null) {
			res.setFrom(from);
		}
		if (until != null) {
			res.setUntil(until);
		}
		if (metadataPrefix != null) {
			res.setMetadataprefix(metadataPrefix);
		}
		if (set != null) {
			res.setSet(set);
		}

		ListDocumentsInfo documents = getDocuments(onlyIdentifiers, set, metadataPrefix, from, until);
		ResumptionToken resumptionToken = documents.getResumptionToken();
		int cursor = documents.getCursor();
		int nMaxElements = documents.getnMaxElements();

		if (onlyIdentifiers) {
			res.setIdentifiers(documents.getDocs());
		} else {
			res.setDocuments(documents.getDocs());
		}

		if ((resumptionToken != null)) {
			res.setResumptiontoken(resumptionToken.serialize());
			res.setCursor(cursor);
			res.setSize(nMaxElements);
		}
		return res;
	}

	/**
	 * List records.
	 * 
	 * @param onlyIdentifiers
	 *            only resource identifiers.
	 * @param resumptionTokenParameter
	 *            resumption token
	 * @return ListRecordsInfo
	 * @throws OaiPublisherException
	 *             could happen
	 */
	public ListRecordsInfo listRecords(final boolean onlyIdentifiers, final String resumptionToken) throws OaiPublisherException {
		ListDocumentsInfo docs = getDocuments(onlyIdentifiers, resumptionToken);

		final ListRecordsInfo res = new ListRecordsInfo();

		if (docs.getMetadataPrefix() != null) {
			res.setMetadataprefix(docs.getMetadataPrefix());
		}

		if (onlyIdentifiers) {
			res.setIdentifiers(docs.getDocs());
		} else {
			res.setDocuments(docs.getDocs());
		}

		ResumptionToken newResumptionToken = docs.getResumptionToken();
		if ((newResumptionToken != null)) {
			res.setResumptiontoken(newResumptionToken.serialize());
			res.setCursor(docs.getCursor());
			res.setSize(docs.getnMaxElements());
		}
		return res;
	}

}
