package eu.dnetlib.enabling.database.inspector;

import java.io.StringWriter;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.annotation.Resource;
import javax.servlet.ServletResponse;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.xml.ws.wsaddressing.W3CEndpointReference;

import org.apache.commons.io.IOUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.jdbc.support.rowset.SqlRowSet;
import org.springframework.ui.ModelMap;
import org.springframework.web.bind.annotation.ModelAttribute;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;

import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;

import eu.dnetlib.enabling.database.DatabaseServiceCore;
import eu.dnetlib.enabling.database.resultset.IterableRowSet;
import eu.dnetlib.enabling.inspector.EntryPointDescriptor;
import eu.dnetlib.enabling.inspector.EntryPointDescriptorGroup;
import eu.dnetlib.enabling.inspector.EntryPointDescriptorModel;
import eu.dnetlib.enabling.resultset.client.IterableResultSetClient;
import eu.dnetlib.enabling.resultset.client.utils.EPRUtils;
import eu.dnetlib.miscutils.collections.MappedCollection;
import eu.dnetlib.miscutils.functional.UnaryFunction;

public class DatabaseController extends AbstractController {

	@Resource
	private DatabaseServiceCore core;

	private static final Log log = LogFactory.getLog(DatabaseController.class); // NOPMD by marko on 11/24/08 5:02 PM
	
	@Resource
	List<EntryPointDescriptorGroup> entryPointsGroups;

	private String DB_INSPECTOR_NAME = "db";
	
	@RequestMapping("index.do")
	public void index(final ModelMap map) {	}

	@RequestMapping("listDBs.do")
	public void listDatabases(final ModelMap map) throws Exception {
		map.addAttribute("list", core.listDatabases());
	}

	@RequestMapping("listTables.do")
	public void listTables(final ModelMap map, @RequestParam(value = "db", required = true) String db) throws Exception {
		map.addAttribute("db", db);
		
		List<Map<String, Object>> tables = new ArrayList<Map<String, Object>>();
		Map<String, String> logSize = new HashMap<String, String>();
		
		StringWriter sw = new StringWriter();
		IOUtils.copy(getClass().getResourceAsStream("tables_info.sql"), sw);
		
		SqlRowSet rows = core.getDbUtils().executeSql(db, sw.toString(), SqlRowSet.class);
		while (rows.next()) {
			String tname = rows.getString("name");
			if (tname.endsWith("_log")) {
				logSize.put(tname.substring(0, tname.length() - 4), rows.getString("total"));
			} else {
				Map<String, Object> t = new HashMap<String, Object>();
				t.put("name", rows.getString("name"));
				t.put("view", rows.getString("kind").equalsIgnoreCase("v"));
				t.put("data", rows.getString("data"));
				t.put("indices", rows.getString("indices"));
				t.put("total", rows.getString("total"));
				t.put("managed", core.getDbUtils().isManagedTable(db, tname));
				t.put("logged", core.getDbUtils().isLoggedTable(db, tname));
				t.put("dnetIdentifier", core.getDbUtils().getDefaultDnetIdentifier(db, tname));
				tables.add(t);
			}
		}
		for (Map<String, Object> t : tables) {
			if (logSize.containsKey(t.get("name"))) {
				t.put("logdata", logSize.get(t.get("name")));
			} else {
				t.put("logdata", "-");
			}
		}
		
		map.addAttribute("tables", tables);
	}

	@RequestMapping("manageDB.do")
	public String manageDB(final ModelMap map,
			@RequestParam(value = "db", required = true) String db,
			@RequestParam(value = "manage", required = true) boolean b) throws Exception {
		core.changeDatabaseStatus(db, b);
		return "redirect:listDBs.do";
	}

	@RequestMapping("importEPR.do")
	public void importEPR(final ModelMap map,
			@RequestParam(value = "db", required = false) String db,
			@RequestParam(value = "epr", required = false) String epr) throws Exception {

		if (db != null && epr != null) {
			core.importFromResultset(db, (new EPRUtils()).getEpr(epr));
			map.addAttribute("message", "Import completed");
		}

		//map.addAttribute("testEPR", lookupLocator.getService().searchProfile("//RESOURCE_PROFILE[.//RESOURCE_TYPE/@value='RepositoryServiceResourceType']"));

		map.addAttribute("list", core.listDatabases());
	}
	
	
	@RequestMapping("query.do")
	public void query(final ModelMap map, 
			@RequestParam(value = "query", required = false) final String query,
			@RequestParam(value = "db", required = false) final String db,
			@RequestParam(value = "limit", required = false) Integer limit) throws Exception {

		if (limit == null) limit = 10;
		map.addAttribute("databases", core.getDbUtils().listAllDatabases());
		map.addAttribute("limit", limit);
		map.addAttribute("db", "");
		
		if (query != null && db != null && query.length() > 0 && db.length() > 0) {
			map.addAttribute("query", query);
			map.addAttribute("db", db);

			try {
				map.addAttribute("list", Lists.newArrayList(Iterables.limit(new IterableRowSet(db, query, null, core.getDbUtils()), limit)));
			} catch (Exception e) {
				map.addAttribute("message", "<b>ERROR:</b> " + e.getMessage() + " (" + e.getClass() + ")");
				log.error("Error executing query", e);
			}
		}
	}
	
	@RequestMapping("changeTableManagement.do")
	public String changeTableManagement(final ModelMap map, 
			@RequestParam(value = "db", required = true) final String db,
			@RequestParam(value = "t", required = true) final String t) throws Exception {
		
		if (core.getDbUtils().isManagedTable(db, t)) {
			core.getDbUtils().removeManagementOfTable(db, t);
		} else {
			core.getDbUtils().prepareManagementOfTable(db, t);			
		}
		
		return "redirect:listTables.do?db=" + db;
	} 

	
	@RequestMapping("changeIdentifiers.do")
	public String changeId(HttpServletResponse res, 
			@RequestParam(value = "db", required = true) final String db,
			@RequestParam(value = "t", required = false) final String t) throws Exception {
		if (t == null) {
			core.getDbUtils().reassignDefaultDnetIdentifiers(db);
			return "redirect:listDBs.do";
		} else {
			core.getDbUtils().reassignDefaultDnetIdentifiers(db, t);
			return "redirect:listTables.do?db=" + db;
		}
	} 
	
	@RequestMapping("changeTableLog.do")
	public String changeTableLog(final ModelMap map, 
			@RequestParam(value = "db", required = true) final String db,
			@RequestParam(value = "t", required = true) final String t) throws Exception {

		if (core.getDbUtils().isManagedTable(db, t)) {
			if (core.getDbUtils().isLoggedTable(db, t)) {
				core.getDbUtils().removeLogTable(db, t);
			} else {
				core.getDbUtils().addLogTable(db, t);			
			}
		}

		return "redirect:listTables.do?db=" + db;
	} 

	@RequestMapping("describeTable.do")
	public void describeTable(final ModelMap map, 
			@RequestParam(value = "db", required = true) final String db,
			@RequestParam(value = "t", required = true) final String t) throws Exception {
		map.addAttribute("db", db);
		map.addAttribute("table", t);
		map.addAttribute("description", core.getDbUtils().describeTable(db, t));
	} 

	@RequestMapping("dumpTable.do")
	public void dumpTable(final ServletResponse response, 
			@RequestParam(value = "db", required = true) final String db,
			@RequestParam(value = "t", required = true) final String t) throws Exception {
		sendFile(response, core.getDbUtils().dumpTableAsXML(db,t), CTYPE_XML);
	}

	@RequestMapping("getTableEPR.do")
	public void getTableEPR(final ServletResponse response, 
			@RequestParam(value = "db", required = true) final String db,
			@RequestParam(value = "t", required = true) final String t) throws Exception {
		sendFile(response, core.generateResultSet(db, t, null).toString(), CTYPE_XML);
	}	

	@RequestMapping("dumpEPR.do")
	public void dumpEPR(final ModelMap map, 
			@RequestParam(value = "epr", required = false) final String eprText) throws Exception {

		if (eprText != null) {
			W3CEndpointReference epr = (new EPRUtils()).getEpr(eprText);
			IterableResultSetClient client = core.getClientResultSetFactory().getClient(epr);
			map.addAttribute("epr", eprText);
			map.addAttribute("iter", client.iterator());
		}
	}

	@RequestMapping("getConditionalEPR.do")
	public void getConditionalEPR(final ServletResponse response, 
			@RequestParam(value = "db", required = true) final String db,
			@RequestParam(value = "table", required = true) final String t, 
			@RequestParam(value = "where", required = true) String where) throws Exception {
	
		log.info("Requested EPR of table " + t + " where " + where);

		sendFile(response, core.generateResultSet(db, t, where).toString(), CTYPE_XML);
	}

	@ModelAttribute("otherMenu")
	public List<EntryPointDescriptorModel> obtainOtherMenu(final HttpServletRequest request) {
		if (entryPointsGroups == null) return null;
		
		List<EntryPointDescriptorModel> menu = new ArrayList<EntryPointDescriptorModel>();
		
		final String currentRelativeUrl = request.getPathInfo().replaceAll("/inspector/(.*\\.do).*", "$1");

		UnaryFunction<EntryPointDescriptorModel, EntryPointDescriptor> mapper = new UnaryFunction<EntryPointDescriptorModel, EntryPointDescriptor>() {

			@Override
			public EntryPointDescriptorModel evaluate(EntryPointDescriptor arg) {
				return new EntryPointDescriptorModel(arg.getName(), arg.getRelativeUrl(), currentRelativeUrl.equals(arg.getRelativeUrl()));
			}
		};

		for (EntryPointDescriptorGroup group : entryPointsGroups) {
			List<EntryPointDescriptorModel> list = Lists.newArrayList(new MappedCollection<EntryPointDescriptorModel, EntryPointDescriptor>(group.getDescriptors(), mapper));
			
			for (EntryPointDescriptorModel model : list) {
				if (!model.getName().equals(DB_INSPECTOR_NAME)) {
					menu.add(model);
				}
			}
		}
	
		return menu;
	}
	
	public void setDB_INSPECTOR_NAME(String dBINSPECTORNAME) {
		DB_INSPECTOR_NAME = dBINSPECTORNAME;
	}

}
