package eu.dnetlib.enabling.manager.msro.hope;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.beans.factory.annotation.Required;

import com.googlecode.sarasvati.Arc;
import com.googlecode.sarasvati.Engine;
import com.googlecode.sarasvati.NodeToken;

import eu.dnetlib.enabling.is.lookup.rmi.ISLookUpException;
import eu.dnetlib.enabling.is.lookup.rmi.ISLookUpService;
import eu.dnetlib.enabling.tools.ServiceLocator;
import eu.dnetlib.workflow.AbstractJobNode;

/**
 * Finds a TDS given a repository id, source metadata format and target metadata format.
 * <p>
 * Metadata formats are to be passed as job node parameters.
 * </p>
 * 
 * @author alessia
 * 
 */
public class FindTDSByMdFormats extends AbstractJobNode {
	/**
	 * logger.
	 */
	private static final Log log = LogFactory.getLog(FindTDSByMdFormats.class); // NOPMD by marko on 11/24/08 5:02 PM

	/**
	 * lookup locator.
	 */
	private ServiceLocator<ISLookUpService> lookupLocator;

	private String actionIfNotFound = "failed";

	private String inputMdFormatAttrName;
	private String outputMdFormatAttrName;

	/**
	 * {@inheritDoc}
	 * 
	 * @see com.googlecode.sarasvati.mem.MemNode#execute(com.googlecode.sarasvati.Engine,
	 *      com.googlecode.sarasvati.NodeToken)
	 */
	@Override
	public void execute(final Engine engine, final NodeToken token) {
		final String repId = token.getFullEnv().getAttribute("repositoryId");
		final String inputMdFormat = token.getFullEnv().getAttribute(this.inputMdFormatAttrName);
		final String outputMdFormat = token.getFullEnv().getAttribute(this.outputMdFormatAttrName);
		log.info("FINDING TDS for repo " + repId + " inputMdFormat: " + inputMdFormat + " outputMdFormat: " + outputMdFormat);

		final String query = "//RESOURCE_PROFILE[.//RESOURCE_TYPE/@value = 'TransformationDSResourceType' and .//REPOSITORY_SERVICE_IDENTIFIER = '"
				+ repId + "' and  .//SOURCE_METADATA_FORMAT/@name='" + inputMdFormat + "' and .//SINK_METADATA_FORMAT/@name='" + outputMdFormat
				+ "']//RESOURCE_IDENTIFIER/@value/string()";

		try {
			final String tdsId = this.lookupLocator.getService().getResourceProfileByQuery(query);

			token.getEnv().setAttribute("dataDsId", tdsId);
			token.getEnv().setAttribute("tdsId", tdsId);
			//let's add also TDS data source and datasink identifiers:
			final String dataSourceId = this.lookupLocator.getService().getResourceProfileByQuery(
					"//RESOURCE_PROFILE[.//RESOURCE_IDENTIFIER/@value = '" + tdsId + "']//DATA_SOURCE/string()");
			final String dataSinkId = this.lookupLocator.getService().getResourceProfileByQuery(
					"//RESOURCE_PROFILE[.//RESOURCE_IDENTIFIER/@value = '" + tdsId + "']//DATA_SINK/string()");

			token.getEnv().setAttribute("dataSource", dataSourceId);
			token.getEnv().setAttribute("dataSink", dataSinkId);

			engine.complete(token, Arc.DEFAULT_ARC);
		} catch (final ISLookUpException e) {
			if (this.actionIfNotFound.equals("failed")) {
				token.getEnv().setAttribute("hasFailed", true);
				token.getEnv().setAttribute("errorMessage", "cannot find TDS for repo " + repId + ": " + e);
				log.warn("cannot find TDS for repository", e);
			}
			engine.complete(token, this.actionIfNotFound);
		}
	}

	public ServiceLocator<ISLookUpService> getLookupLocator() {
		return this.lookupLocator;
	}

	@Required
	public void setLookupLocator(final ServiceLocator<ISLookUpService> lookupLocator) {
		this.lookupLocator = lookupLocator;
	}

	public String getActionIfNotFound() {
		return this.actionIfNotFound;
	}

	public void setActionIfNotFound(final String actionIfNotFound) {
		this.actionIfNotFound = actionIfNotFound;
	}

	public String getInputMdFormatAttrName() {
		return this.inputMdFormatAttrName;
	}

	public void setInputMdFormatAttrName(final String inputMdFormatAttrName) {
		this.inputMdFormatAttrName = inputMdFormatAttrName;
	}

	public String getOutputMdFormatAttrName() {
		return this.outputMdFormatAttrName;
	}

	public void setOutputMdFormatAttrName(final String outputMdFormatAttrName) {
		this.outputMdFormatAttrName = outputMdFormatAttrName;
	}

}
