package eu.dnetlib.enabling.manager.msro.hope.edm;

import java.io.PrintWriter;
import java.io.StringWriter;

import javax.annotation.Resource;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.beans.factory.annotation.Required;

import com.googlecode.sarasvati.GraphProcess;
import com.googlecode.sarasvati.env.Env;
import com.googlecode.sarasvati.mem.MemGraphProcess;

import eu.dnetlib.data.information.MDStoreDataSinkSourceDescriptorGenerator;
import eu.dnetlib.enabling.is.lookup.rmi.ISLookUpDocumentNotFoundException;
import eu.dnetlib.enabling.is.lookup.rmi.ISLookUpException;
import eu.dnetlib.enabling.is.lookup.rmi.ISLookUpService;
import eu.dnetlib.enabling.manager.msro.AbstractWorkflowLauncherNotificationHandler;
import eu.dnetlib.enabling.tools.ServiceLocator;

/**
 * NotificationHandler for the TransformHopeRecords job.
 * 
 * @author alessia
 * 
 */
public class TransformEDMRecordsNotificationHandler extends AbstractWorkflowLauncherNotificationHandler {
	private static final Log log = LogFactory.getLog(TransformEDMRecordsNotificationHandler.class); // NOPMD by marko on 11/24/08 5:02 PM
	private static final String STATUS_VALUE = "TAGGED";

	private ServiceLocator<ISLookUpService> lookupLocator;

	@Resource
	protected MDStoreDataSinkSourceDescriptorGenerator sourceDescriptionGenerator;

	/**
	 * {@inheritDoc}
	 * 
	 * @see eu.dnetlib.enabling.tools.blackboard.NotificationHandler#notified(java.lang.String, java.lang.String,
	 *      java.lang.String, java.lang.String)
	 */
	@Override
	public void notified(final String subscrId, final String topic, final String rsId, final String profile) {
		log.info(this.getClass() + " notified. ID subscription: " + subscrId);
		if (!this.mustStart(topic, profile, rsId))
			return;

		final GraphProcess process = new MemGraphProcess(this.getGraph());
		this.getProcessRegistry().associateProcessWithResource(process, rsId);

		Env env = process.getEnv();
		env.setAttribute("enableableName", getEnableableName());

		log.debug("Starting workflow process for " + getEnableableName());

		try {
			prepareProcess(process, env, rsId, profile);
		} catch (RuntimeException e) {
			env.setAttribute("hasFailed", true);
			env.setAttribute("errorMessage", e.getMessage());
			final StringWriter stackTrace = new StringWriter();
			e.printStackTrace(new PrintWriter(stackTrace));

			env.setAttribute("errorDetails", stackTrace.toString());
			throw e;
		}

		this.getEngine().startProcess(process);
	}

	private boolean mustStart(String topic, String profile, String rsId) {
		String topicPrefix = getTopicPrefix();
		if (!topic.startsWith(topicPrefix)) {
			log.debug("WF must not start because of missing topic prefix: " + topicPrefix);
			return false;
		}
		String xQueryStatus = "//RESOURCE_PROFILE[.//RESOURCE_IDENTIFIER/@value = '" + rsId + "']//LAST_UPDATE_STATUS/text()";
		try {
			String statusValue = this.lookupLocator.getService().quickSearchProfile(xQueryStatus).get(0);
			if (statusValue.equals(STATUS_VALUE)) {
				return true;
			} else {
				log.debug("WF must not start because status value is : " + statusValue);
				return false;
			}

		} catch (ISLookUpException e) {
			log.debug("WF must not start because of lookup exception: " + e);
			return false;
		}

	}

	/**
	 * {@inheritDoc}
	 * 
	 * @see eu.dnetlib.enabling.manager.msro.AbstractWorkflowLauncherNotificationHandler#prepareProcess(com.googlecode.sarasvati.GraphProcess,
	 *      com.googlecode.sarasvati.env.Env, java.lang.String, java.lang.String)
	 */
	@Override
	protected void prepareProcess(GraphProcess process, Env env, String rsId, String profile) {
		//rsId is the id of the TDS from hope cleaned to EDM
		log.debug("NotificationHandler to transform Hope records into EDM records. rsId (TDS id) = " + rsId);

		String datasource = "";
		String datasink = "";
		String repId = "";
		String xsltCode = "";

		final String sourceQuery = "//RESOURCE_PROFILE[.//RESOURCE_IDENTIFIER/@value = '" + rsId + "']//DATA_SOURCE/text()";
		final String sinkQuery = "//RESOURCE_PROFILE[.//RESOURCE_IDENTIFIER/@value = '" + rsId + "']//DATA_SINK/text()";
		final String repQuery = "//RESOURCE_PROFILE[.//RESOURCE_IDENTIFIER/@value = '" + rsId + "']//REPOSITORY_SERVICE_IDENTIFIER/text()";
		final String xsltQuery = "let $ruleID := //RESOURCE_PROFILE[.//RESOURCE_IDENTIFIER/@value = '" + rsId
				+ "']//TRANSFORMATION_RULE_DS_IDENTIFIER/text() "
				+ "return //RESOURCE_PROFILE[.//RESOURCE_IDENTIFIER/@value = $ruleID]//SCRIPT/CODE/*[local-name()='stylesheet']";
		try {
			datasource = lookupLocator.getService().getResourceProfileByQuery(sourceQuery);
			datasink = lookupLocator.getService().getResourceProfileByQuery(sinkQuery);
			repId = this.lookupLocator.getService().getResourceProfileByQuery(repQuery);
			xsltCode = this.lookupLocator.getService().quickSearchProfile(xsltQuery).get(0);
		} catch (ISLookUpDocumentNotFoundException e) {
			throw new RuntimeException(e);
		} catch (ISLookUpException e) {
			throw new RuntimeException(e);
		}

		String dataSinkId = sourceDescriptionGenerator.getParameters(datasink).getId();
		String dataSourceId = sourceDescriptionGenerator.getParameters(datasource).getId();
		env.setAttribute("mdstoreSrc", dataSourceId);
		env.setAttribute("mdstoreSink", dataSinkId);
		//this is for the update of the last storage date of EDM store
		env.setAttribute("dataDsId", dataSinkId);

		env.setAttribute("repositoryId", repId);
		env.setAttribute("xsltCode", xsltCode);
	}

	public ServiceLocator<ISLookUpService> getLookupLocator() {
		return lookupLocator;
	}

	@Required
	public void setLookupLocator(ServiceLocator<ISLookUpService> lookupLocator) {
		this.lookupLocator = lookupLocator;
	}
}
