package eu.dnetlib.enabling.manager.msro.hope.thumbnails;

import java.io.File;
import java.io.IOException;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.Iterator;
import java.util.List;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;

import javax.annotation.Resource;
import javax.xml.ws.wsaddressing.W3CEndpointReference;

import org.apache.commons.io.FileUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.im4java.core.ConvertCmd;
import org.im4java.core.IM4JavaException;
import org.im4java.core.IMOperation;

import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;
import com.google.common.io.Files;
import com.google.gson.Gson;
import com.googlecode.sarasvati.Engine;
import com.googlecode.sarasvati.NodeToken;

import eu.dnetlib.enabling.resultset.client.IterableResultSetClient;
import eu.dnetlib.enabling.resultset.client.ResultSetClientFactory;
import eu.dnetlib.miscutils.datetime.DateUtils;
import eu.dnetlib.workflow.AbstractJobNode;

public class DownloaderConverterJobNode extends AbstractJobNode {

	private static final Log log = LogFactory.getLog(DownloaderConverterJobNode.class); // NOPMD by marko on 11/24/08 5:02 PM

	@Resource
	private ResultSetClientFactory resultSetClientFactory;
	private final Gson gson = new Gson();
	private int width, height;

	/**
	 * target directory for thumbnails. See property services.msro.thumbnail.dir.
	 */
	private String thumbnailDir;
	private int nThreads = 10;
	private int nTasks = 100;

	@Override
	public void execute(Engine engine, NodeToken token) {
		ExecutorService executor = Executors.newFixedThreadPool(nThreads);
		W3CEndpointReference downloadListEpr = (W3CEndpointReference) token.getEnv().getTransientAttribute("downloadListEpr");
		if (downloadListEpr == null) {
			failed(engine, token, new NullPointerException("downloadListEpr is null"));
			log.debug("downloadListEpr is null");
		} else {
			log.debug("Got epr downloadListEpr");
			IterableResultSetClient client = resultSetClientFactory.getClient(downloadListEpr);
			log.debug("Got client");
			File tmpDir = Files.createTempDir();
			log.debug("Created tmp dir at: " + tmpDir.getAbsolutePath());
			IMOperation op = this.createConversionOperation();
			List<DownloadAndConverter> tasks = Lists.newArrayList();
			Iterator<String> itemsIterator = client.iterator();
			log.debug("iterating on client");
			int total = 0;
			Iterable<Future<Boolean>> results = Lists.newArrayList();
			while (itemsIterator.hasNext()) {
				int i = 0;
				while (i < nTasks && itemsIterator.hasNext()) {
					String item = itemsIterator.next();
					total++;
					ThumbnailInfo thumb = gson.fromJson(item, ThumbnailInfo.class);
					String[] targetSplit = thumb.getThumbnailURL().split("/");
					String destFileName = targetSplit[targetSplit.length - 1];
					File target = new File(thumbnailDir + "/" + destFileName);
					if (!target.exists()) {
						DownloadAndConverter task = new DownloadAndConverter();
						task.setInputURL(thumb.getInputURL());
						task.setTargetFilePath(thumbnailDir + "/" + destFileName);
						task.setTmpDir(tmpDir);
						task.setConversionOp(op);
						tasks.add(task);
						i++;
					} else
						log.debug("Thumbnail already generated for " + thumb.getInputURL() + " in " + thumb.getThumbnailURL());
				}
				try {
					log.debug("Invoking tasks...");
					results = Iterables.concat(results, executor.invokeAll(tasks));
					//for (Future<Boolean> res : results)
					//log.debug(res.get().booleanValue());
				} catch (InterruptedException e) {
					log.debug("Some tasks might not be successfully completed, cause: " + e.getMessage());
				} catch (Exception e) {
					log.debug("Unexpectd exception: " + e.getMessage());
				}
			}
			log.debug("Waiting for tasks completion");
			for (Future<Boolean> res : results) {
				try {
					res.get();
				} catch (InterruptedException e) {
					log.debug("Some tasks might not be successfully completed, cause: " + e.getMessage());
				} catch (ExecutionException e) {
					log.debug("Some tasks might not be successfully completed, cause: " + e.getMessage());
				}
			}
			log.debug("executor shutdown ...");
			log.debug("Number of thumbnail processed: " + total);
			//waits for submitted task to complete, then gracefully shutdowns the threadpool.
			executor.shutdown();
			super.execute(engine, token);
		}
	}

	private IMOperation createConversionOperation() {
		IMOperation op = new IMOperation();
		op.addImage();
		op.define("jpeg:size=" + width * 2 + "x" + height * 2);
		op.thumbnail(width, height);
		op.unsharp(0.0, 0.5);
		op.background("transparent");
		op.gravity("center");
		op.extent(width, height);
		op.colors(256);
		op.addImage();
		return op;
	}

	class DownloadAndConverter implements Callable<Boolean> {

		private String targetFilePath;
		private String inputURL;
		private File tmpDir;
		private IMOperation conversionOp;

		@Override
		public Boolean call() throws Exception {
			File tmp = null;
			try {
				tmp = File.createTempFile("" + DateUtils.now(), "thumb", tmpDir);
				tmp.deleteOnExit();
				FileUtils.copyURLToFile(new URL(inputURL), tmp);
				ConvertCmd cmd = new ConvertCmd();
				File target = new File(targetFilePath);
				target.createNewFile();
				cmd.run(conversionOp, tmp.getAbsolutePath() + "[0]", targetFilePath);
				return true;
			} catch (MalformedURLException e) {
				log.debug("Cannot download file from: " + inputURL + ", cause: " + e.getMessage());
				return false;
			} catch (IOException e) {
				log.debug("Cannot copy file from: " + inputURL + ", cause: " + e.getMessage());
				return false;
			} catch (InterruptedException e) {
				log.debug("Cannot convert into thumbnail file from: " + inputURL + ", cause: " + e.getMessage());
				return false;
			} catch (IM4JavaException e) {
				log.debug("Cannot convert into thumbnail file from: " + inputURL + ", cause: " + e.getMessage());
				return false;
			} finally {
				if (tmp != null)
					tmp.delete();
			}
		}

		public String getTargetFilePath() {
			return targetFilePath;
		}

		public void setTargetFilePath(String targetFilePath) {
			this.targetFilePath = targetFilePath;
		}

		public String getInputURL() {
			return inputURL;
		}

		public void setInputURL(String inputURL) {
			this.inputURL = inputURL;
		}

		public File getTmpDir() {
			return tmpDir;
		}

		public void setTmpDir(File tmpDir) {
			this.tmpDir = tmpDir;
		}

		public IMOperation getConversionOp() {
			return conversionOp;
		}

		public void setConversionOp(IMOperation conversionOp) {
			this.conversionOp = conversionOp;
		}

	}

	public ResultSetClientFactory getResultSetClientFactory() {
		return resultSetClientFactory;
	}

	public void setResultSetClientFactory(ResultSetClientFactory resultSetClientFactory) {
		this.resultSetClientFactory = resultSetClientFactory;
	}

	public String getThumbnailDir() {
		return thumbnailDir;
	}

	public void setThumbnailDir(String thumbnailDir) {
		this.thumbnailDir = thumbnailDir;
	}

	public int getnThreads() {
		return nThreads;
	}

	public void setnThreads(int nThreads) {
		this.nThreads = nThreads;
	}

	public int getnTasks() {
		return nTasks;
	}

	public void setnTasks(int nTasks) {
		this.nTasks = nTasks;
	}

	public int getWidth() {
		return width;
	}

	public void setWidth(int width) {
		this.width = width;
	}

	public int getHeight() {
		return height;
	}

	public void setHeight(int height) {
		this.height = height;
	}

}
