package eu.dnetlib.data.index.inspector;

import java.io.UnsupportedEncodingException;
import java.net.URLEncoder;
import java.util.ArrayList;
import java.util.List;
import java.util.Set;

import javax.annotation.Resource;
import javax.xml.ws.wsaddressing.W3CEndpointReference;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.stereotype.Controller;
import org.springframework.ui.Model;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;

import com.google.common.collect.Lists;
import com.google.common.collect.Sets;

import eu.dnetlib.data.index.IIndexService;
import eu.dnetlib.data.index.IndexServiceException;
import eu.dnetlib.enabling.inspector.AbstractInspectorController;
import eu.dnetlib.enabling.is.lookup.rmi.ISLookUpException;
import eu.dnetlib.enabling.is.lookup.rmi.ISLookUpService;
import eu.dnetlib.enabling.resultset.rmi.ResultSetException;
import eu.dnetlib.enabling.resultset.rmi.ResultSetService;
import eu.dnetlib.enabling.tools.ServiceEnumerator;
import eu.dnetlib.enabling.tools.ServiceLocator;
import eu.dnetlib.enabling.tools.ServiceResolver;
import eu.dnetlib.enabling.tools.ServiceRunningInstance;
import eu.dnetlib.miscutils.collections.MappedCollection;
import eu.dnetlib.miscutils.datetime.HumanTime;
import eu.dnetlib.miscutils.functional.UnaryFunction;
import eu.dnetlib.miscutils.functional.string.EscapeHtml;
import eu.dnetlib.miscutils.functional.xml.IndentXmlString;
import eu.dnetlib.soap.cxf.CxfEndpointReferenceBuilder;

/**
 * Allows to perform queries to indices.
 * 
 * @author marko
 * 
 */
@Controller
public class IndexInspector extends AbstractInspectorController {

	public static class SelectOption {
		private String value;
		private boolean selected;

		public SelectOption(final String value, final boolean selected) {
			super();
			this.value = value;
			this.selected = selected;
		}

		public String getValue() {
			return value;
		}

		public void setValue(final String value) {
			this.value = value;
		}

		public boolean isSelected() {
			return selected;
		}

		public void setSelected(final boolean selected) {
			this.selected = selected;
		}

	}

	/**
	 * result page size.
	 */
	private static final int PAGE_SIZE = 10;

	/**
	 * logger.
	 */
	private static final Log log = LogFactory.getLog(IndexInspector.class); // NOPMD by marko on 11/24/08 5:02 PM

	/**
	 * index service locator.
	 */
	@Resource(name = "anyIndexLocator")
	private ServiceLocator<IIndexService> indexLocator;

	/**
	 * lookup locator.
	 */
	@Resource(name = "lookupLocator")
	private ServiceLocator<ISLookUpService> lookupLocator;

	/**
	 * service resolver.
	 */
	@Resource
	private ServiceResolver serviceResolver;

	/**
	 * index service enumerator.
	 */
	@Resource(name = "indexServiceEnumerator")
	private ServiceEnumerator<IIndexService> indexEnumerator;
	
	/**
	 * epr builder.
	 */
	@Resource
	private CxfEndpointReferenceBuilder eprBuilder;
	
	/**
	 * local index service URL
	 */
	@Resource
	private String localIndexServiceUrl;

	/**
	 * search the whole index service.
	 * 
	 * @param model
	 *            mvc model
	 * @param query
	 *            query
	 * @param format
	 *            metadata format
	 * @throws IndexServiceException
	 *             could happen
	 * @throws ResultSetException
	 *             could happen
	 * @throws ISLookUpException
	 *             could happen
	 */
	@RequestMapping(value = "/inspector/searchIndex.do")
	public void search(
			final Model model,
			@RequestParam(value = "query", required = false) final String query,
			@RequestParam(value = "format", required = false) final String format,
			@RequestParam(value = "layout", required = false) final String layout,
			@RequestParam(value = "indexService", required = false) final String indexServiceUrl,
			@RequestParam(value = "indent", required = false) final Boolean indent) 
		throws IndexServiceException, ResultSetException, ISLookUpException {

		final List<ServiceRunningInstance<IIndexService>> indexServices = indexEnumerator.getServices();
		
		if (query != null) {
			log.info("running query: " + query + " on format " + format + " with layout " + layout);

			final W3CEndpointReference indexEpr;
			
			if(indexServiceUrl == null || "".equals(indexServiceUrl)) 
				indexEpr = indexServices.get(0).getEpr();
			else 
				indexEpr = eprBuilder.getEndpointReference(indexServiceUrl, null, null, null, null, null);
			
			final IIndexService indexService = serviceResolver.getService(IIndexService.class, indexEpr);

			final W3CEndpointReference epr = indexService.indexLookup("all", query, format, layout);

			log.info("GOT EPR: " + epr);

			final ResultSetService resultSet = serviceResolver.getService(ResultSetService.class, epr);
			final String rsId = serviceResolver.getResourceIdentifier(epr);

			final int total = resultSet.getNumberOfElements(rsId);
			int pageSize = PAGE_SIZE;
			if (total < pageSize)
				pageSize = total;

			List<String> elements = resultSet.getResult(rsId, 1, pageSize, "waiting");
			if (elements == null)
				elements = new ArrayList<String>();
			
			if (indent != null && indent)
				elements = MappedCollection.listMap(elements, new IndentXmlString());
			
			final List<String> escaped = MappedCollection.listMap(elements, new EscapeHtml());
			model.addAttribute("indent", indent);
			model.addAttribute("size", total);
			model.addAttribute("results", escaped);
		}

		final List<String> formats = lookupLocator.getService().quickSearchProfile(
				"distinct-values(//RESOURCE_PROFILE[.//RESOURCE_TYPE/@value = 'IndexDSResourceType']//METADATA_FORMAT/text())");

		final List<String> layouts = lookupLocator.getService().quickSearchProfile(
				"distinct-values(for $x in //RESOURCE_PROFILE//LAYOUT/@name/string() order by $x return $x)");

		Set<String> indices = Sets.newHashSet();
		for (ServiceRunningInstance<IIndexService> index : indexServices) {
			indices.add(index.getUrl());
		}
		
		indices.add(localIndexServiceUrl);

		model.addAttribute("indexServices", selectOptions(Lists.newArrayList(indices), indexServiceUrl));

		model.addAttribute("formats", selectOptions(formats, format));
		model.addAttribute("layouts", selectOptions(layouts, layout));

		model.addAttribute("query", query);
	}

	@RequestMapping(value = "/inspector/browseIndex.do")
	public void browse(
			final Model model,
			@RequestParam(value = "query", required = false) String querySource,
			@RequestParam(value = "field", required = false) final String browseField,
			@RequestParam(value = "idxId", required = false) final String idxId,
			@RequestParam(value = "indexService", required = false) final String indexServiceUrl,			
			@RequestParam(value = "format", required = false) final String format,
			@RequestParam(value = "layout", required = false) final String layout,
			@RequestParam(value = "start", required = false) Integer startParam,
			@RequestParam(value = "indent", required = false) final Boolean indent) 
		throws IndexServiceException, ResultSetException, ISLookUpException, UnsupportedEncodingException {

		long starting = System.currentTimeMillis();
		long elapsed = starting;
		int total = 0;
		int start = 0;
		int pageSize = PAGE_SIZE;
		
		final List<ServiceRunningInstance<IIndexService>> indexServices = indexEnumerator.getServices();

		if (startParam != null)
			start = startParam;
		
		if (browseField != null) {
			log.info("browsing field: " + browseField + " on format " + format + " with layout " + layout);

			String query = "query=" + querySource + "&groupby=" + browseField;
			
			final W3CEndpointReference indexEpr;
			
			if(indexServiceUrl == null || "".equals(indexServiceUrl)) 
				indexEpr = indexServices.get(0).getEpr();
			else 
				indexEpr = eprBuilder.getEndpointReference(indexServiceUrl, null, null, null, null, null);			
						
			final IIndexService indexService = serviceResolver.getService(IIndexService.class, indexEpr);
			
			final W3CEndpointReference epr = indexService.getBrowsingStatistics(query, idxId, format, layout);

			log.debug("GOT EPR: " + epr);

			final ResultSetService resultSet = serviceResolver.getService(ResultSetService.class, epr);
			final String rsId = serviceResolver.getResourceIdentifier(epr);

			total = resultSet.getNumberOfElements(rsId);
			if (total < pageSize)
				pageSize = total;

			List<String> elements = resultSet.getResult(rsId, 1 + start, start + pageSize, "waiting");
			if (elements == null)
				elements = new ArrayList<String>();
			
			if (indent != null && indent)
				elements = MappedCollection.listMap(elements, new IndentXmlString());
			
			final List<String> escaped = MappedCollection.listMap(elements, new EscapeHtml());
			
			elapsed = System.currentTimeMillis() - starting;
			
			model.addAttribute("results", escaped);
			model.addAttribute("elapsed", HumanTime.exactly(elapsed));
		}		
		
		final List<String> fields  = lookupLocator.getService().quickSearchProfile(
				"distinct-values(//RESOURCE_PROFILE//LAYOUT//FIELD/@browsingAliasFor/string())");
		final List<String> formats = lookupLocator.getService().quickSearchProfile(
				"distinct-values(//RESOURCE_PROFILE[.//RESOURCE_TYPE/@value = 'IndexDSResourceType']//METADATA_FORMAT/text())");
		final List<String> layouts = lookupLocator.getService().quickSearchProfile(
				"distinct-values(for $x in //RESOURCE_PROFILE//LAYOUT/@name/string() order by $x return $x)");
		final List<String> idxIds  = lookupLocator.getService().quickSearchProfile(
				"//RESOURCE_IDENTIFIER[../RESOURCE_TYPE/@value = 'IndexDSResourceType']/@value/string()");
		
		idxIds.add(0, "all");
		
		Set<String> indices = Sets.newHashSet();
		for (ServiceRunningInstance<IIndexService> index : indexServices) {
			indices.add(index.getUrl());
		}
		
		indices.add(localIndexServiceUrl);		
		
		if(querySource == null || querySource.equals(""))
			querySource = "textual";
	
		model.addAttribute("indexServices", selectOptions(Lists.newArrayList(indices), indexServiceUrl));
		model.addAttribute("encodedQuerySource", URLEncoder.encode(querySource, "UTF-8"));
		model.addAttribute("querySource", querySource);
		model.addAttribute("total", total);
		model.addAttribute("start", start);
		model.addAttribute("nextPage", start + pageSize);
		model.addAttribute("prevPage", Math.max(0, start - pageSize));
		model.addAttribute("indent", indent);
		
		model.addAttribute("browseField", browseField);
		
		model.addAttribute("fields", selectOptions(fields, browseField));
		model.addAttribute("idxIds", selectOptions(idxIds, idxId));
		model.addAttribute("formats", selectOptions(formats, format));
		model.addAttribute("layouts", selectOptions(layouts, layout));

	}

	/**
	 * Given an list of values, return a list of SelectOption instances which have the "selected" boolean field set to
	 * true only for the element matching "current".
	 * 
	 * @param input
	 *            list of input strings
	 * @param current
	 *            current value to select
	 * @return
	 */
	private List<SelectOption> selectOptions(final List<String> input, final String current) {
		return MappedCollection.listMap(input, new UnaryFunction<SelectOption, String>() {
			@Override
			public SelectOption evaluate(final String value) {
				return new SelectOption(value, value.equals(current));
			}
		});
	}

	public ServiceLocator<IIndexService> getIndexLocator() {
		return indexLocator;
	}

	public void setIndexLocator(final ServiceLocator<IIndexService> indexLocator) {
		this.indexLocator = indexLocator;
	}

	public ServiceResolver getServiceResolver() {
		return serviceResolver;
	}

	public void setServiceResolver(final ServiceResolver serviceResolver) {
		this.serviceResolver = serviceResolver;
	}

	public ServiceLocator<ISLookUpService> getLookupLocator() {
		return lookupLocator;
	}

	public void setLookupLocator(final ServiceLocator<ISLookUpService> lookupLocator) {
		this.lookupLocator = lookupLocator;
	}
}
