package eu.dnetlib.data.oai.store.sync;

import java.util.List;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;

import javax.annotation.Resource;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.google.common.collect.Lists;

import eu.dnetlib.data.information.oai.publisher.conf.PublisherConfigurationReader;
import eu.dnetlib.data.information.oai.publisher.info.MDFInfo;

public class OAIStoreSynchronizer {

	private static final Log log = LogFactory.getLog(OAIStoreSynchronizer.class); // NOPMD by marko on 11/24/08 5:02 PM
	/**
	 * OAI Publisher configuration.
	 */
	@Resource
	private PublisherConfigurationReader configuration;
	@Resource
	private OAIStoreInitializerFactory initializerFactory;

	/**
	 * Number of threads to use in the pool for the synchronization.
	 */
	private int maxInitThreads = 2;

	/**
	 * Synchronizes the OAI stores according to the current configuration.
	 * <p>
	 * If forceContentSync is true, then also the content is synchronized.
	 * 
	 * @param forceContentSync
	 *            true to force the content sync, false to sync only the configuration (e.g., a new index)
	 * @param callback
	 *            callback to execute when the execution is successfull
	 * @param failCallbackcallback
	 *            to execute when the execution fails
	 */
	public void synchronize(final boolean forceContentSync, final Callable<?> callback, final Callable<?> failCallback) {
		log.info("Initializing OAI Publisher stores from its configuration profile with initializer from factory: "
				+ initializerFactory.getClass().getCanonicalName());
		if (forceContentSync) {
			log.info("Content will be synchronized as well.");
		}
		// starting new thread to avoid the "neverending" reload of the inspector page.
		new Thread() {

			@Override
			public void run() {
				try {
					checkStores(forceContentSync);
				} catch (InterruptedException e) {
					log.error(e);
					executeCallback(failCallback);
				} catch (ExecutionException e) {
					log.error(e);
					executeCallback(failCallback);
				}
				log.info("OAI Publisher configuration updated.");
				executeCallback(callback);
			}

		}.start();
	}

	protected void checkStores(final boolean forceContentSync) throws InterruptedException, ExecutionException {
		final ExecutorService executor = Executors.newFixedThreadPool(maxInitThreads);
		List<MDFInfo> metadataFormats = this.configuration.getMetadataFormatInfo();
		List<Future<?>> futures = Lists.newArrayList();
		for (MDFInfo mdf : metadataFormats) {
			futures.add(executor.submit(initializerFactory.create(mdf, forceContentSync)));
		}
		executor.shutdown();
		for (Future<?> future : futures) {
			future.get();
		}
	}

	protected void executeCallback(final Callable<?> callback) {
		if (callback != null) {
			try {
				callback.call();
			} catch (Exception e) {
				log.error("Error executing callback", e);
			}
		}
	}

	public PublisherConfigurationReader getConfiguration() {
		return configuration;
	}

	public void setConfiguration(final PublisherConfigurationReader configuration) {
		this.configuration = configuration;
	}

	public OAIStoreInitializerFactory getInitializerFactory() {
		return initializerFactory;
	}

	public void setInitializerFactory(final OAIStoreInitializerFactory initializerFactory) {
		this.initializerFactory = initializerFactory;
	}

	public int getMaxInitThreads() {
		return maxInitThreads;
	}

	public void setMaxInitThreads(final int maxInitThreads) {
		this.maxInitThreads = maxInitThreads;
	}
}
