package eu.dnetlib.data.oai.store.mongo;

import java.util.List;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

import javax.annotation.Resource;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.beans.factory.annotation.Required;

import com.google.common.collect.Lists;
import com.mongodb.BasicDBObject;
import com.mongodb.BasicDBObjectBuilder;
import com.mongodb.DB;
import com.mongodb.DBCollection;
import com.mongodb.DBCursor;
import com.mongodb.DBObject;

import eu.dnetlib.data.information.oai.publisher.OaiPublisherException;
import eu.dnetlib.data.information.oai.publisher.conf.OAIConfigurationReader;
import eu.dnetlib.data.information.oai.publisher.info.MDFInfo;
import eu.dnetlib.data.oai.store.PublisherStoreDAO;
import eu.dnetlib.data.oai.store.RecordChangeDetector;
import eu.dnetlib.data.oai.store.parser.MongoQueryParser;
import eu.dnetlib.data.oai.store.sets.MongoSetCollection;

public class MongoPublisherStoreDAO implements PublisherStoreDAO<MongoPublisherStore, MongoCursor> {

	private static final Log log = LogFactory.getLog(MongoPublisherStoreDAO.class); // NOPMD by marko on 11/24/08 5:02 PM

	/** Name of the collection with information about the OAI stores. **/
	private String metadataCollection;
	@Resource(name = "publisherMongoDB")
	private DB publisherDB;
	@Resource
	private RecordInfoGenerator recordInfoGenerator;
	@Resource
	private MetadataExtractor metadataExtractor;
	@Resource
	private MongoQueryParser queryParser;
	@Resource(name = "oaiConfigurationExistReader")
	private OAIConfigurationReader configuration;
	@Resource
	private RecordChangeDetector recordChangeDetector;
	@Resource
	private MongoSetCollection mongoSetCollection;

	/**
	 * True if we want to patch the record according to the Openaire schema at insertion/update time. FIXME: remove this as soon as the
	 * imported records are correclty generated as expected by the schema.
	 */
	private boolean patch = false;

	@Override
	public List<MongoPublisherStore> listPublisherStores() {
		List<MongoPublisherStore> stores = Lists.newArrayList();
		DBCursor cursor = this.publisherDB.getCollection(this.metadataCollection).find();
		for (DBObject storeInfo : cursor) {
			stores.add(this.createFromDBObject(storeInfo));
		}
		return stores;
	}

	@Override
	public MongoPublisherStore getStore(final String storeId) {
		DBObject storeInfo = this.publisherDB.getCollection(this.metadataCollection).findOne(new BasicDBObject("id", storeId));
		return this.createFromDBObject(storeInfo);
	}

	@Override
	public MongoPublisherStore getStore(final String mdfName, final String mdfInterpretation, final String mdfLayout) {
		return this.getStore(this.generateStoreId(mdfName, mdfInterpretation, mdfLayout));
	}

	@Override
	public MongoPublisherStore getStoreFor(final String targetMetadataPrefix) {
		MDFInfo info = this.configuration.getMetadataFormatInfo(targetMetadataPrefix);
		return this.getStore(info.getSourceFormatName(), info.getSourceFormatInterpretation(), info.getSourceFormatLayout());
	}

	@Override
	public MongoPublisherStore createStore(final String mdfName, final String mdfInterpretation, final String mdfLayout) throws OaiPublisherException {
		DBObject store = createMetadataEntry(mdfName, mdfInterpretation, mdfLayout);
		DBCollection metadata = this.publisherDB.getCollection(this.metadataCollection);
		metadata.insert(store);
		MongoPublisherStore theStore = this.createFromDBObject(store);
		theStore.ensureIndices();
		return theStore;

	}

	@Override
	public boolean deleteStore(final String storeId) {
		DBCollection metadata = this.publisherDB.getCollection(this.metadataCollection);
		DBObject storeInfo = metadata.findOne(new BasicDBObject("id", storeId));
		if (storeInfo == null) return false;
		else {
			this.publisherDB.getCollection(storeId).drop();
			metadata.remove(storeInfo);
			return true;
		}
	}

	@Override
	public boolean deleteStore(final String mdfName, final String mdfInterpretation, final String mdfLayout) {
		return this.deleteStore(this.generateStoreId(mdfName, mdfInterpretation, mdfLayout));
	}

	public void ensureIndex(final MongoPublisherStore store) {
		if (store == null) {
			log.error("Can't ensure index on null store");
			return;
		}
		Thread t = new Thread() {

			@Override
			public void run() {
				store.ensureIndices();
			}
		};
		t.start();
	}

	public void ensureIndex() {
		this.ensureIndex(1);
	}

	public void ensureIndex(final int maxThreads) {
		ExecutorService executorService = Executors.newFixedThreadPool(maxThreads);
		List<MongoPublisherStore> stores = this.listPublisherStores();
		for (final MongoPublisherStore s : stores) {
			executorService.execute(new Runnable() {

				@Override
				public void run() {
					s.ensureIndices();

				}
			});
		}
		executorService.shutdown();
	}

	private MongoPublisherStore createFromDBObject(final DBObject storeInfo) {
		if (storeInfo == null) return null;
		String storeId = (String) storeInfo.get("id");
		String mdFormat = (String) storeInfo.get("metadataFormat");
		String mdInterpreation = (String) storeInfo.get("interpretation");
		String mdLayout = (String) storeInfo.get("layout");

		MongoPublisherStore store = new MongoPublisherStore(storeId, mdFormat, mdInterpreation, mdLayout, this.publisherDB.getCollection(storeId),
				this.configuration.getFields(mdFormat, mdInterpreation, mdLayout), queryParser, recordInfoGenerator, this.configuration.getIdScheme(),
				this.configuration.getIdNamespace(), this.metadataExtractor, this.recordChangeDetector);
		store.setMongoSetCollection(mongoSetCollection);
		store.setPatch(patch);
		return store;
	}

	private DBObject createMetadataEntry(final String mdfName, final String mdfInterpretation, final String mdfLayout) {
		DBObject info = BasicDBObjectBuilder.start("id", generateStoreId(mdfName, mdfInterpretation, mdfLayout)).append("metadataFormat", mdfName)
				.append("interpretation", mdfInterpretation).append("layout", mdfLayout).get();
		return info;

	}

	private String generateStoreId(final String mdfName, final String mdfInterpretation, final String mdfLayout) {
		return mdfName + "-" + mdfInterpretation + "-" + mdfLayout;
	}

	public DB getPublisherDB() {
		return publisherDB;
	}

	public void setPublisherDB(final DB publisherDB) {
		this.publisherDB = publisherDB;
	}

	public String getMetadataCollection() {
		return metadataCollection;
	}

	@Required
	public void setMetadataCollection(final String metadataCollection) {
		this.metadataCollection = metadataCollection;
	}

	public MongoQueryParser getQueryParser() {
		return queryParser;
	}

	public void setQueryParser(final MongoQueryParser queryParser) {
		this.queryParser = queryParser;
	}

	public OAIConfigurationReader getConfiguration() {
		return configuration;
	}

	public void setConfiguration(final OAIConfigurationReader configuration) {
		this.configuration = configuration;
	}

	public RecordInfoGenerator getRecordInfoGenerator() {
		return recordInfoGenerator;
	}

	public void setRecordInfoGenerator(final RecordInfoGenerator recordInfoGenerator) {
		this.recordInfoGenerator = recordInfoGenerator;
	}

	public MetadataExtractor getMetadataExtractor() {
		return metadataExtractor;
	}

	public void setMetadataExtractor(final MetadataExtractor metadataExtractor) {
		this.metadataExtractor = metadataExtractor;
	}

	public RecordChangeDetector getRecordChangeDetector() {
		return recordChangeDetector;
	}

	public void setRecordChangeDetector(final RecordChangeDetector recordChangeDetector) {
		this.recordChangeDetector = recordChangeDetector;
	}

	public MongoSetCollection getMongoSetCollection() {
		return mongoSetCollection;
	}

	public void setMongoSetCollection(final MongoSetCollection mongoSetCollection) {
		this.mongoSetCollection = mongoSetCollection;
	}

	public boolean isPatch() {
		return patch;
	}

	public void setPatch(final boolean patch) {
		this.patch = patch;
	}

}
