package eu.dnetlib.enabling.datasources.rmi;

import java.util.List;

import javax.jws.WebParam;
import javax.jws.WebService;

import eu.dnetlib.common.rmi.BaseService;

@WebService
public interface DatasourceManagerService extends BaseService {

	/**
	 * Register a datasource.
	 * 
	 * @param ds
	 *            the new datasource
	 * @throws DatasourceManagerServiceException
	 */
	public boolean addDatasource(@WebParam(name = "ds") DatasourceDesc ds) throws DatasourceManagerServiceException;

	/**
	 * Delete a datasource.
	 * 
	 * @param dsId
	 *            the OpenAireDataSourceId (example: opendoar____::2101)
	 * @throws DatasourceManagerServiceException
	 */
	public boolean deleteDatasource(@WebParam(name = "dsId") String dsId) throws DatasourceManagerServiceException;

	/**
	 * Returns a datasource.
	 * 
	 * @param dsId
	 *            the OpenAireDataSourceId (example: opendoar____::2101)
	 * @throws DatasourceManagerServiceException
	 */
	public DatasourceDesc getDatasource(@WebParam(name = "dsId") String dsId) throws DatasourceManagerServiceException;

	/**
	 * Returns all the datasources.
	 * 
	 * @return a list of XML profiles
	 * @throws DatasourceManagerServiceException
	 */
	public List<DatasourceDesc> listAllDatasources() throws DatasourceManagerServiceException;

	/**
	 * Returns the datasources with some properties.
	 * 
	 * @param compliance
	 *            the level of compliance (or NULL to skip the test)
	 * @param fileManagementMode
	 *            the fileManagementMode property (or NULL to skip the test)
	 * @param contentDescription
	 *            the contentDescription property (or NULL to skip the test)
	 * @param iisProcessingWorkflow
	 *            the iisProcessingWorkflow property (or NULL to skip the test)
	 * @param collectedFrom
	 *            the collectedFrom property (or NULL to skip the test)
	 * @return a list of XML profiles
	 * @throws DatasourceManagerServiceException
	 */
	public List<DatasourceDesc> listDatasourcesUsingFilter(@WebParam(name = "compliance") String compliance,
			@WebParam(name = "contentDescription") String contentDescription,
			@WebParam(name = "iisProcessingWorkflow") String iisProcessingWorkflow,
			@WebParam(name = "collectedFrom") String collectedFrom) throws DatasourceManagerServiceException;

	/**
	 * updates the level of compliance of a datasource.
	 * 
	 * @param dsId
	 *            the OpenAireDataSourceId (example: opendoar____::2101)
	 * @param ifaceId
	 *            the interface id
	 * @param level
	 *            the new level of compliance
	 * @return true if the operation ends successfully
	 * @throws DatasourceManagerServiceException
	 */
	public boolean updateLevelOfCompliance(@WebParam(name = "dsId") String dsId,
			@WebParam(name = "ifaceId") String ifaceId,
			@WebParam(name = "level") String level) throws DatasourceManagerServiceException;

	/**
	 * update the baseUrl of an interface.
	 * 
	 * @param dsId
	 *            the OpenAireDataSourceId (example: opendoar____::2101)
	 * @param ifaceId
	 *            the interface id
	 * @param baseUrl
	 *            the new value
	 * @return true if the operation ends successfully
	 * @throws DatasourceManagerServiceException
	 */
	public boolean updateBaseUrl(@WebParam(name = "dsId") String dsId, @WebParam(name = "ifaceId") String ifaceId, @WebParam(name = "baseUrl") String baseUrl)
			throws DatasourceManagerServiceException;

	/**
	 * update the activation status of an interface (true = there is a related wf).
	 * 
	 * @param dsId
	 *            the OpenAireDataSourceId (example: opendoar____::2101)
	 * @param ifaceId
	 *            the interface id
	 * @param active
	 *            true if there is a related wf
	 * @return true if the operation ends successfully
	 * @throws DatasourceManagerServiceException
	 */
	public boolean updateActivationStatus(@WebParam(name = "dsId") String dsId,
			@WebParam(name = "ifaceId") String ifaceId,
			@WebParam(name = "active") boolean active) throws DatasourceManagerServiceException;

	/**
	 * update the "contentDescription" property of an interface.
	 * 
	 * @param dsId
	 *            the OpenAireDataSourceId (example: opendoar____::2101)
	 * @param ifaceId
	 *            the interface id
	 * @param desc
	 *            the new value
	 * @return true if the operation ends successfully
	 * @throws DatasourceManagerServiceException
	 */
	public boolean updateContentDescription(@WebParam(name = "dsId") String dsId,
			@WebParam(name = "ifaceId") String ifaceId,
			@WebParam(name = "desc") String desc) throws DatasourceManagerServiceException;

	/**
	 * update the "iis_processing_workflow" property of an interface.
	 * 
	 * @param dsId
	 *            the OpenAireDataSourceId (example: opendoar____::2101)
	 * @param ifaceId
	 *            the interface id
	 * @param wf
	 *            the new value
	 * @return true if the operation ends successfully
	 * @throws DatasourceManagerServiceException
	 */
	public boolean setIisProcessingWorkflow(@WebParam(name = "dsId") String dsId, @WebParam(name = "ifaceId") String ifaceId, @WebParam(name = "wf") String wf)
			throws DatasourceManagerServiceException;

	/**
	 * Set a generic extra field of an interface.
	 * 
	 * @param dsId
	 *            the OpenAireDataSourceId (example: opendoar____::2101)
	 * @param ifaceId
	 *            the interface id
	 * @param field
	 *            the field name
	 * @param value
	 *            the new value (or null for removing it)
	 * @param preserveOriginal
	 *            preserve the original value in the DB (the value is stored in the "edited" column)
	 * @return true if the operation ends successfully
	 * @throws DatasourceManagerServiceException
	 */
	public boolean updateExtraField(@WebParam(name = "dsId") String dsId,
			@WebParam(name = "ifaceId") String ifaceId,
			@WebParam(name = "field") String field,
			@WebParam(name = "value") String value,
			@WebParam(name = "preserveOriginal") boolean preserveOriginal) throws DatasourceManagerServiceException;

	/**
	 * Update an access param of an interface.
	 * 
	 * @param dsId
	 *            the OpenAireDataSourceId (example: opendoar____::2101)
	 * @param ifaceId
	 *            the interface id
	 * @param field
	 *            the field name
	 * @param value
	 *            the new value (or null for removing it)
	 * @param preserveOriginal
	 *            preserve the original value in the DB (the value is stored in the "edited" column)
	 * @return true if the operation ends successfully
	 * @throws DatasourceManagerServiceException
	 */
	public boolean updateAccessParam(@WebParam(name = "dsId") String dsId,
			@WebParam(name = "ifaceId") String ifaceId,
			@WebParam(name = "field") String field,
			@WebParam(name = "value") String value,
			@WebParam(name = "preserveOriginal") boolean preserveOriginal) throws DatasourceManagerServiceException;

	/**
	 * delete a generic extra field or an access param of an interface.
	 * 
	 * @param dsId
	 *            the OpenAireDataSourceId (example: opendoar____::2101)
	 * @param ifaceId
	 *            the interface id
	 * @param field
	 *            the field name
	 * @return true if the operation ends successfully
	 * @throws DatasourceManagerServiceException
	 */
	public boolean deleteAccessParamOrExtraField(@WebParam(name = "dsId") String dsId,
			@WebParam(name = "ifaceId") String ifaceId,
			@WebParam(name = "field") String field) throws DatasourceManagerServiceException;

	/**
	 * Add an interface to a datasource.
	 * 
	 * @param dsId
	 *            the OpenAireDataSourceId (example: opendoar____::2101)
	 * @param iface
	 *            the interface description
	 * @return true if the operation ends successfully
	 * @throws DatasourceManagerServiceException
	 */
	public boolean addInterface(@WebParam(name = "dsId") String dsId, @WebParam(name = "interface") IfaceDesc iface) throws DatasourceManagerServiceException;

	/**
	 * Delete the interface of a datasource.
	 * 
	 * @param dsId
	 *            the OpenAireDataSourceId (example: opendoar____::2101)
	 * @param ifcId
	 *            the id of the interface
	 * @return true if the operation ends successfully
	 * @throws DatasourceManagerServiceException
	 */
	public boolean deleteInterface(@WebParam(name = "dsId") String dsId, @WebParam(name = "ifaceId") String ifaceId) throws DatasourceManagerServiceException;

	/**
	 * Perform a generic SQL update on a datasource.
	 * 
	 * @param dsId
	 *            the OpenAireDataSourceId (example: opendoar____::2101)
	 * @param sql
	 *            the sql query to insert/update/delete the datasource
	 * @param delete
	 *            true if the profile must be deleted
	 * @return true if the operation ends successfully
	 * @throws DatasourceManagerServiceException
	 */
	public boolean updateSQL(@WebParam(name = "dsId") String dsId, @WebParam(name = "sql") String sql, @WebParam(name = "delete") boolean delete)
			throws DatasourceManagerServiceException;

}
