package eu.dnetlib.data.transform;

import java.io.StringReader;
import java.util.List;

import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerException;

import org.apache.commons.codec.binary.Base64;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.dom4j.Document;
import org.dom4j.Element;
import org.dom4j.Node;
import org.dom4j.io.DocumentResult;
import org.dom4j.io.DocumentSource;
import org.dom4j.io.SAXReader;

import com.google.common.base.Function;
import com.google.common.collect.Lists;

public class XsltRowTransformer implements Function<String, List<Row>> {

	private static final Log log = LogFactory.getLog(XsltRowTransformer.class); // NOPMD by marko on 11/24/08 5:02 PM

	private Transformer transformer;

	private SAXReader reader = new SAXReader();

	public XsltRowTransformer(Transformer transformer) {
		this.transformer = transformer;
	}

	private Document transform(Document doc) {
		if (transformer == null)
			return doc;
		final DocumentResult result = new DocumentResult();

		try {
			transformer.transform(new DocumentSource(doc), result);
			return result.getDocument();
		} catch (TransformerException e) {
			throw new RuntimeException("Unable to transform document:\n" + doc.asXML(), e);
		}
	}

	private Document transform(String xml) {
		try {
			return transform(reader.read(new StringReader(xml)));
		} catch (Exception e) {
			log.error("Error parsing xml:\n" + xml, e);
			return null;
		}
	}

	private List<Row> transformAsListRow(String xml) {
		final Document doc = transform(xml);

		final List<Row> rows = Lists.newArrayList();

		if (doc == null)
			return rows;

		for (Object or : doc.selectNodes("//ROW")) {
			final Element row = (Element) or;

			final String columnFamily = row.valueOf("@columnFamily");
			final String key = row.valueOf("@key");

			if (key == null || key.isEmpty())
				throw new RuntimeException("Attribute 'key' is missing in XSLT");
			
			if (columnFamily == null || columnFamily.isEmpty())
				throw new RuntimeException("Attribute 'columnFamily' is missing in XSLT");

			final List<Column<String, byte[]>> cols = Lists.newArrayList();

			for (Object of : row.selectNodes("./QUALIFIER")) {
				final Node node = (Node) of;

				final String name = node.valueOf("@name");
				final String type = node.valueOf("@type");

				final byte[] value = decode(node.getText().trim(), type);

				cols.add(new Column<String, byte[]>(name, value));
			}
			rows.add(new Row(columnFamily, key, cols));
		}
		return rows;
	}

	private byte[] decode(String value, String type) {

		if ("base64".equals(type))
			return Base64.decodeBase64(value);

		return value.getBytes();
	}

	@Override
	public List<Row> apply(String xml) {
		return transformAsListRow(xml);
	}

}
