package eu.dnetlib.iis.export.actionmanager.generator;

import java.io.IOException;
import java.io.InputStream;
import java.util.HashMap;
import java.util.Map;

import org.apache.avro.Schema;
import org.apache.avro.file.DataFileWriter;
import org.apache.avro.io.Decoder;
import org.apache.avro.io.DecoderFactory;
import org.apache.avro.specific.SpecificDatumReader;
import org.apache.avro.specific.SpecificRecord;
import org.apache.hadoop.fs.FSDataInputStream;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.LocatedFileStatus;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.fs.RemoteIterator;
import org.apache.log4j.Logger;

import eu.dnetlib.iis.core.java.HadoopContext;
import eu.dnetlib.iis.core.java.PortBindings;
import eu.dnetlib.iis.core.java.Ports;
import eu.dnetlib.iis.core.java.Process;
import eu.dnetlib.iis.core.java.io.DataStore;
import eu.dnetlib.iis.core.java.io.FileSystemPath;
import eu.dnetlib.iis.core.java.porttype.AnyPortType;
import eu.dnetlib.iis.core.java.porttype.AvroPortType;
import eu.dnetlib.iis.core.java.porttype.PortType;
import eu.dnetlib.iis.export.schemas.DataSetReferenceWithInferencedData;
import eu.dnetlib.iis.documentssimilarity.schemas.DocumentSimilarity;
import eu.dnetlib.iis.export.schemas.DocumentWithInferencedData;
import eu.dnetlib.iis.export.schemas.PersonWithInferencedData;

/**
 * Json based inferenced data generator.
 * @author mhorst
 *
 */
public class JsonBasedInferencedDataGenerator implements Process {

	private final Logger log = Logger.getLogger(this.getClass());
	
	public enum JsonInput {
		json_document_similarity,
		json_document_with_inferenced_data,
		json_dataset_with_inferenced_data,
		json_person_with_inferenced_data
	}
	
	/**
	 * Ports handled by importer processor.
	 * Directory paths should be provided as parameters, when given path is missing
	 * output will not be generated.
	 */
	private static final Ports ports;
	
	private static final Map<ExportMode, JsonInput> modeToPathMap;
	
	static {
//		preparing ports
//		input ports
		Map<String, PortType> input = new HashMap<String, PortType>();
		input.put(JsonInput.json_document_similarity.name(), 
				new AnyPortType());		
		input.put(JsonInput.json_document_with_inferenced_data.name(), 
				new AnyPortType());
		input.put(JsonInput.json_dataset_with_inferenced_data.name(), 
				new AnyPortType());
		input.put(JsonInput.json_person_with_inferenced_data.name(), 
				new AnyPortType());
//		output ports
		Map<String, PortType> output = new HashMap<String, PortType>();
		output.put(ExportMode.document_similarity.name(), 
				new AvroPortType(DocumentSimilarity.SCHEMA$));		
		output.put(ExportMode.document_with_inferenced_data.name(), 
				new AvroPortType(DocumentWithInferencedData.SCHEMA$));
		output.put(ExportMode.dataset_with_inferenced_data.name(), 
				new AvroPortType(DataSetReferenceWithInferencedData.SCHEMA$));
		output.put(ExportMode.person_with_inferenced_data.name(), 
				new AvroPortType(PersonWithInferencedData.SCHEMA$));
		ports = new Ports(input, output);
		
		modeToPathMap = new HashMap<ExportMode, JsonInput>();
		modeToPathMap.put(ExportMode.document_similarity, JsonInput.json_document_similarity);
		modeToPathMap.put(ExportMode.document_with_inferenced_data, JsonInput.json_document_with_inferenced_data);
		modeToPathMap.put(ExportMode.dataset_with_inferenced_data, JsonInput.json_dataset_with_inferenced_data);
		modeToPathMap.put(ExportMode.person_with_inferenced_data, JsonInput.json_person_with_inferenced_data);
	}	
	
	@Override
	public Map<String, PortType> getInputPorts() {
		return getStaticPorts().getInput();
	}
	
	@Override
	public Map<String, PortType> getOutputPorts() {
		return getStaticPorts().getOutput();
	}

	/**
	 * Static method for returning ports.
	 * @return ports
	 */
	public static Ports getStaticPorts() {
		return ports;
	}

	@Override
	public void run(PortBindings portBindings, HadoopContext context,
			Map<String, String> parameters) throws Exception {
		for (ExportMode currentMode : modeToPathMap.keySet()) {
			handleMode(currentMode, portBindings, context, parameters);
		}
	}

	/**
	 * Handles objects generation for given mode.
	 * @param mode
	 * @param portBindings
	 * @param context
	 * @param parameters
	 * @throws IOException
	 */
	@SuppressWarnings("unchecked")
	protected void handleMode(ExportMode mode, 
			PortBindings portBindings, HadoopContext context,
			Map<String, String> parameters) throws IOException {
		
		Path inputPath = portBindings.getInput().get(modeToPathMap.get(mode).name());
		FileSystem fs = FileSystem.get(context.getConfiguration());
		if (fs.exists(inputPath)) {
			Schema schema = ((AvroPortType)ports.getOutput().get(mode.name())).getSchema();
			@SuppressWarnings("rawtypes")
			DataFileWriter writer = null;
			try {
				if (fs.isDirectory(inputPath)) {
					RemoteIterator<LocatedFileStatus> statusIt = fs.listFiles(inputPath, true);
					while (statusIt.hasNext()) {
						if (writer==null) {
							writer = DataStore.create(
									new FileSystemPath(fs, portBindings.getOutput().get(
											mode.name())), 
									schema);
						}
						LocatedFileStatus fileStatus = statusIt.next();
						if (!fileStatus.isDirectory()) {
							FSDataInputStream inputStream = fs.open(fileStatus.getPath());
							try {
								writer.append(convertJsonToAvro(
										inputStream, schema));
							} finally {
								inputStream.close();
							}
						} else {
							log.debug("skipping directory:" + 
									fileStatus.getPath().toString());
						}
					}	
				} else {
	//				handling as single file
					FSDataInputStream inputStream = fs.open(inputPath);
					try {
						writer = DataStore.create(
								new FileSystemPath(fs, portBindings.getOutput().get(
										mode.name())), 
								schema);
						writer.append(convertJsonToAvro(
								inputStream, schema));
					} finally {
						inputStream.close();
					}
				}
			} finally {
				if (writer!=null) {
					writer.close();	
				}
			}	
		} else {
			log.warn("unable to generate data for mode " + mode + 
					", no directory found for path: " + inputPath);
		}
	}
	
	/**
	 * Converts JSON string representation to Avro object.
	 * Returns null for null JSON.
	 * @param jsonInputStream
	 * @param schema
	 * @return Avro object converted from JSON string representation
	 * @throws IOException
	 */
	protected <T extends SpecificRecord> T convertJsonToAvro(InputStream jsonInputStream, 
			Schema schema) throws IOException {
		if (jsonInputStream==null) {
			return null;
		}
		SpecificDatumReader<T> datumReader = new SpecificDatumReader<T>(
				schema);
		Decoder decoder = DecoderFactory.get().jsonDecoder(schema, jsonInputStream);
		return datumReader.read(null, decoder);
	}
}