package eu.dnetlib.iis.export.actionmanager;

import static eu.dnetlib.iis.common.WorkflowRuntimeParameters.EXPORT_ACTION_BUILDER_FACTORY_CLASSNAME;
import static eu.dnetlib.iis.common.WorkflowRuntimeParameters.EXPORT_ACTION_SETID;
import static eu.dnetlib.iis.common.WorkflowRuntimeParameters.EXPORT_ACTION_SET_PROPERTY_SEPARATOR;
import static eu.dnetlib.iis.common.WorkflowRuntimeParameters.EXPORT_SKIP_AUTHORS;
import static eu.dnetlib.iis.common.WorkflowRuntimeParameters.EXPORT_SKIP_ORIGINAL_METADATA;
import static eu.dnetlib.iis.common.WorkflowRuntimeParameters.EXPORT_TRUST_LEVEL;

import java.io.IOException;
import java.lang.reflect.Constructor;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.avro.mapred.AvroKey;
import org.apache.avro.specific.SpecificRecordBase;
import org.apache.hadoop.io.NullWritable;
import org.apache.hadoop.mapreduce.Mapper;

import eu.dnetlib.actionmanager.actions.AtomicAction;
import eu.dnetlib.actionmanager.rmi.ActionManagerException;
import eu.dnetlib.data.proto.SpecialTrustProtos.SpecialTrust;
import eu.dnetlib.iis.common.WorkflowRuntimeParameters;
import eu.dnetlib.iis.export.actionmanager.api.ActionManagerServiceFacade;
import eu.dnetlib.iis.export.actionmanager.cfg.ActionManagerConfigurationProvider;
import eu.dnetlib.iis.export.actionmanager.cfg.StaticConfigurationProvider;
import eu.dnetlib.iis.export.actionmanager.module.ActionBuilderFactory;
import eu.dnetlib.iis.export.actionmanager.module.ActionBuilderModule;
import eu.dnetlib.iis.export.actionmanager.module.ActionSetIdProvider;
import eu.dnetlib.iis.export.actionmanager.module.ActionSetMappingNotDefinedException;
import eu.dnetlib.iis.export.actionmanager.module.AlgorithmName;

/**
 * ActionManager service based exporter mapper.
 * @author mhorst
 *
 */
public abstract class AbstractActionManagerBasedExporterMapper 
extends Mapper<AvroKey<? extends SpecificRecordBase>, NullWritable, NullWritable, NullWritable> {
	
	private String predefinedTrust = SpecialTrust.NEUTRAL.name();
	
	private ActionManagerServiceFacade actionManager;
	
	private ActionManagerConfigurationProvider configProvider;
	
	private ActionBuilderModule<SpecificRecordBase> actionBuilder;
	
	/** This is the place you can access map-reduce workflow node parameters */
	@SuppressWarnings("unchecked")
	@Override
	protected void setup(Context context) throws IOException,
			InterruptedException {
		
		if (context.getConfiguration().get(
				EXPORT_TRUST_LEVEL)!=null) {
			this.predefinedTrust = context.getConfiguration().get(
					EXPORT_TRUST_LEVEL);
		}
		
		@SuppressWarnings("unused")
		boolean skipOriginalMetadata = true;
		if (context.getConfiguration().get(
				EXPORT_SKIP_ORIGINAL_METADATA)!=null) {
			skipOriginalMetadata = Boolean.valueOf(context.getConfiguration().get(
					EXPORT_SKIP_ORIGINAL_METADATA));	
		}
		
		@SuppressWarnings("unused")
		boolean skipExportingAuthors = true;
		if (context.getConfiguration().get(
				EXPORT_SKIP_AUTHORS)!=null) {
			skipExportingAuthors = Boolean.valueOf(context.getConfiguration().get(
					EXPORT_SKIP_AUTHORS));	
		}
		String moduleClassName = context.getConfiguration().get(
				EXPORT_ACTION_BUILDER_FACTORY_CLASSNAME);
		if (moduleClassName!=null) {
			try {
				ActionSetIdProvider actionSetIdProvider = provideAlgorithmToActionSetMap(context);
				Class<?> clazz = Class.forName(moduleClassName);
				Constructor<?> constructor = clazz.getConstructor();
				ActionBuilderFactory<SpecificRecordBase> actionBuilderFactory = 
						(ActionBuilderFactory<SpecificRecordBase>) constructor.newInstance();
				actionBuilder = actionBuilderFactory.instantiate(
						predefinedTrust, actionSetIdProvider.getActionSetId(
								actionBuilderFactory.getAlgorithName()));
				actionManager = buildActionManager(context);
				configProvider = new StaticConfigurationProvider(
						StaticConfigurationProvider.AGENT_DEFAULT,
						StaticConfigurationProvider.PROVENANCE_DEFAULT,
						StaticConfigurationProvider.ACTION_TRUST_0_9,
						StaticConfigurationProvider.NAMESPACE_PREFIX_DEFAULT);
				
			} catch (Exception e) {
				throw new RuntimeException(
						"unexpected exception ocurred when instantiating "
						+ "builder module: " + moduleClassName, e);
			} 
		} else {
			throw new RuntimeException("unknown action builder module instance, "
					+ "no " + EXPORT_ACTION_BUILDER_FACTORY_CLASSNAME + 
					" parameter provided!");
		}
	}

	private ActionSetIdProvider provideAlgorithmToActionSetMap(Context context) {
		String defaultActionSetId = null;
		if (context.getConfiguration().get(EXPORT_ACTION_SETID)!=null && 
				!WorkflowRuntimeParameters.UNDEFINED_NONEMPTY_VALUE.equals(
						context.getConfiguration().get(EXPORT_ACTION_SETID))) {
			defaultActionSetId = context.getConfiguration().get(
					EXPORT_ACTION_SETID);	
		}
		final Map<AlgorithmName, String> algoToActionSetMap = new HashMap<AlgorithmName, String>();
		for (AlgorithmName currentAlgorithm : AlgorithmName.values()) {
			String propertyKey = EXPORT_ACTION_SETID + EXPORT_ACTION_SET_PROPERTY_SEPARATOR + 
					currentAlgorithm.name();
			if (context.getConfiguration().get(propertyKey)!=null && 
					!WorkflowRuntimeParameters.UNDEFINED_NONEMPTY_VALUE.equals(
							context.getConfiguration().get(propertyKey))) {
				algoToActionSetMap.put(currentAlgorithm,
						context.getConfiguration().get(propertyKey));
			} else {
				algoToActionSetMap.put(currentAlgorithm,
						defaultActionSetId);
			}
		}
		return new ActionSetIdProvider() {
			@Override
			public String getActionSetId(AlgorithmName algorithmName)
					throws ActionSetMappingNotDefinedException {
				String actionSetId = algoToActionSetMap.get(algorithmName);
				if (actionSetId!=null && 
						!WorkflowRuntimeParameters.UNDEFINED_NONEMPTY_VALUE.equals(
								actionSetId)) {
					return actionSetId;
				} else {
					throw new ActionSetMappingNotDefinedException(
							"no action set identifier defined "
							+ "for algorithm: " + algorithmName.name());
				}
			}
		};
	}
	
	/**
	 * Builds action manager instance.
	 * @param context
	 * @throws IOException
	 * @return action manager instance
	 */
	protected abstract ActionManagerServiceFacade buildActionManager(Context context) throws IOException;
	
	/**
	 * Creates collection of actions for given datum
	 * @param datum
	 * @return personBuilder of actions
	 */
	protected List<AtomicAction> createActions(SpecificRecordBase datum) {
		return actionBuilder.build(datum, configProvider.provideAgent());
	}
	
	@Override
	protected void map(AvroKey<? extends SpecificRecordBase> key, NullWritable ignore, 
			Context context) throws IOException, InterruptedException {
		try {
			List<AtomicAction> actions = createActions(key.datum());
			if (actions!=null) {
					actionManager.storeAction( 
							actions,
							null, 
							null, 
							configProvider.provideNamespacePrefix());	
			}
		} catch (ActionManagerException e) {
			throw new RuntimeException(e);
		}
	}

	@Override
	protected void cleanup(Context context)
			throws IOException, InterruptedException {
		try {
			if (actionManager!=null) {
				actionManager.close();	
			}
		} catch (ActionManagerException e) {
			throw new IOException(e);
		} finally {
			super.cleanup(context);
		}
	}
	
}