package eu.dnetlib.iis.export.actionmanager.module;

import eu.dnetlib.actionmanager.actions.ActionFactory;
import eu.dnetlib.data.proto.DataInfoProtos.DataInfo;
import eu.dnetlib.data.proto.KindProtos.Kind;
import eu.dnetlib.data.proto.OafProtos.Oaf;
import eu.dnetlib.data.proto.OafProtos.OafEntity;
import eu.dnetlib.data.proto.OafProtos.OafRel;
import eu.dnetlib.data.proto.QualifierProtos.Qualifier;
import eu.dnetlib.data.proto.RelMetadataProtos.RelMetadata;
import eu.dnetlib.data.proto.RelTypeProtos.RelType;
import eu.dnetlib.iis.common.hbase.HBaseConstants;

/**
 * Abstract builder module.
 * @author mhorst
 *
 */
public abstract class AbstractBuilderModule {

	/**
	 * Predefined trust level.
	 */
	private final String predefinedTrust;
	
	/**
	 * Action set id data should be exported to.
	 */
	protected final String actionSetId;
	
	/**
	 * Action factory to be used for building actions.
	 */
	protected final ActionFactory actionFactory = new ActionFactory();
	
	/**
	 * Default constructor.
	 * @param predefinedTrust
	 */
	public AbstractBuilderModule(String predefinedTrust,
			String actionSetId) {
		this.predefinedTrust = predefinedTrust;
		this.actionSetId = actionSetId;
	}
	
	/**
	 * Builds {@link Oaf} object.
	 * @param oafEntity
	 * @return {@link Oaf} object
	 */
	protected Oaf buildOaf(OafEntity oafEntity) {
		eu.dnetlib.data.proto.OafProtos.Oaf.Builder oafBuilder = Oaf.newBuilder();
		oafBuilder.setKind(Kind.entity);
		oafBuilder.setEntity(oafEntity);
		oafBuilder.setDataInfo(buildInference());
		oafBuilder.setTimestamp(System.currentTimeMillis());
		return oafBuilder.build();
	}
	
	/**
	 * Returns inference object.
	 * @return inference object
	 */
	protected DataInfo buildInference() {
		DataInfo.Builder builder = DataInfo.newBuilder();
		builder.setInferred(true);
		builder.setTrust(predefinedTrust);
		Qualifier.Builder provenanceBuilder = Qualifier.newBuilder();
		provenanceBuilder.setClassid(HBaseConstants.SEMANTIC_CLASS_IIS);
		provenanceBuilder.setClassname(HBaseConstants.SEMANTIC_CLASS_IIS);
		provenanceBuilder.setSchemeid(HBaseConstants.SEMANTIC_SCHEME_DNET_PROVENANCE_ACTIONS);
		provenanceBuilder.setSchemename(HBaseConstants.SEMANTIC_SCHEME_DNET_PROVENANCE_ACTIONS);
		builder.setProvenanceaction(provenanceBuilder.build());
		return builder.build();
	}

	/**
	 * Builds relation metadata.
	 * @param schemaId
	 * @param classId
	 * @return relation metadata.
	 */
	protected RelMetadata buildRelMetadata(String schemaId, String classId) {
		RelMetadata.Builder relBuilder = RelMetadata.newBuilder();
		Qualifier.Builder qBuilder = Qualifier.newBuilder();
		qBuilder.setSchemeid(schemaId);
		qBuilder.setSchemename(schemaId);
		qBuilder.setClassid(classId);
		qBuilder.setClassname(classId);
		relBuilder.setSemantics(qBuilder.build());
		return relBuilder.build();
	}
	
	/**
	 * Building same object relation.
	 * @param sourceId
	 * @param targetId
	 * @param relType
	 * @return same object relation
	 */
	protected Oaf buildSameObjectRelation(String sourceId, String targetId, 
			RelType relType) {
//		TODO how to mark this relation as inference related
		OafRel.Builder relBuilder = OafRel.newBuilder();
		relBuilder.setSource(sourceId);
		relBuilder.setTarget(targetId);
		relBuilder.setChild(false);
		relBuilder.setRelType(relType);
		eu.dnetlib.data.proto.OafProtos.Oaf.Builder oafBuilder = Oaf.newBuilder();
		oafBuilder.setKind(Kind.relation);
		oafBuilder.setRel(relBuilder.build());
		oafBuilder.setDataInfo(buildInference());
		oafBuilder.setTimestamp(System.currentTimeMillis());
		return oafBuilder.build();
	}

	/**
	 * Clones builder provided as parameter, inverts relations and builds Oaf object.
	 * @param existingBuilder
	 * @return Oaf object containing relation with inverted source and target fields
	 */
	public static byte[] invertRelationAndBuild(Oaf.Builder existingBuilder) {
//		works on builder clone to prevent changes in existing builder
		if (existingBuilder.getRel()!=null) {
			if (existingBuilder.getRel().getSource()!=null &&
					existingBuilder.getRel().getTarget()!=null) {
				Oaf.Builder builder = existingBuilder.clone();
				OafRel.Builder relBuilder = builder.getRelBuilder();
				String source = relBuilder.getSource();
				String target = relBuilder.getTarget();
				relBuilder.setSource(target);
				relBuilder.setTarget(source);
				builder.setRel(relBuilder.build());
				builder.setTimestamp(System.currentTimeMillis());
				return builder.build().toByteArray();
			} else {
				throw new RuntimeException("invalid state: " +
						"either source or target relation was missing!");
			}
		} else {
			throw new RuntimeException("invalid state: " +
					"no relation object found!");
		}
	}
}
