package eu.dnetlib.iis.export.actionmanager.module;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import eu.dnetlib.actionmanager.actions.AtomicAction;
import eu.dnetlib.actionmanager.common.Agent;
import eu.dnetlib.data.proto.KindProtos.Kind;
import eu.dnetlib.data.proto.OafProtos.Oaf;
import eu.dnetlib.data.proto.OafProtos.OafRel;
import eu.dnetlib.data.proto.RelTypeProtos.RelType;
import eu.dnetlib.data.proto.SimilarityRelProtos.SimilarityRel;
import eu.dnetlib.iis.documentssimilarity.schemas.DocumentSimilarity;

/**
 * {@link DocumentSimilarity} based action builder module.
 * @author mhorst
 *
 */
public class DocumentSimilarityActionBuilderModuleFactory  
	implements ActionBuilderFactory<DocumentSimilarity> {

	
	class DocumentSimilarityActionBuilderModule extends AbstractBuilderModule 
	implements ActionBuilderModule<DocumentSimilarity> {
	
		/**
		 * Default constructor.
		 * @param predefinedTrust
		 * @param actionSetId
		 */
		public DocumentSimilarityActionBuilderModule(
				String predefinedTrust,
				String actionSetId) {
			super(predefinedTrust, actionSetId);
		}
		
		@Override
		public List<AtomicAction> build(DocumentSimilarity object, Agent agent) {
			if (object==null) {
				return Collections.emptyList();
			}
	//		setting relations in both source and target objects
			List<AtomicAction> simActions = createActions(
					object, actionSetId, agent, false);
			List<AtomicAction> reverseSimActions = createActions(
					object, actionSetId, agent, true);
			List<AtomicAction> results = new ArrayList<AtomicAction>();
			if (simActions!=null && !simActions.isEmpty()) {
				results.addAll(simActions);
			}
			if (reverseSimActions!=null && !reverseSimActions.isEmpty()) {
				results.addAll(reverseSimActions);
			}
			return results;
		}
		
		/**
		 * Creates similarity related puts.
		 * @param object
		 * @param actionSet
		 * @param agent
		 * @param backwardMode
		 * @return similarity related puts
		 */
		protected List<AtomicAction> createActions(DocumentSimilarity object, 
				String actionSet, Agent agent, boolean backwardMode) {
			Oaf oafObjectRel = buildOAFRel(
					object.getDocumentId().toString(), 
					object.getOtherDocumentId().toString(),
					object.getSimilarity(), backwardMode);
			if (oafObjectRel==null) {
				return Collections.emptyList();
			}
			List<AtomicAction> actionList = new ArrayList<AtomicAction>();
			AtomicAction currentAction = actionFactory.createAtomicAction(
					actionSet, agent, 
					backwardMode?
							object.getOtherDocumentId().toString():
								object.getDocumentId().toString(), 
					RelType.similarityRel.name(), 
					backwardMode?
							object.getDocumentId().toString():
								object.getOtherDocumentId().toString(), 
					oafObjectRel.toByteArray());
			actionList.add(currentAction);
			return actionList;
		}
		
		/**
		 * Builds OAF object.
		 * @param source
		 * @param target
		 * @param score
		 * @param invert flag indicating source and target should be inverted
		 * @return OAF object
		 */
		private Oaf buildOAFRel(String sourceId, String targetDocId, 
				float score, boolean invert) {
			OafRel.Builder relBuilder = OafRel.newBuilder();
			if (!invert) {
				relBuilder.setSource(sourceId);
				relBuilder.setTarget(targetDocId);
				
			} else {
				relBuilder.setSource(targetDocId);
				relBuilder.setTarget(sourceId);
			}
			relBuilder.setChild(false);
			relBuilder.setRelType(RelType.similarityRel);
			relBuilder.setSimilarityRel(
					SimilarityRel.newBuilder().setType(
							SimilarityRel.Type.STANDARD).setSimilarity(
							score).build());
			eu.dnetlib.data.proto.OafProtos.Oaf.Builder oafBuilder = Oaf.newBuilder();
			oafBuilder.setKind(Kind.relation);
			oafBuilder.setRel(relBuilder.build());
			oafBuilder.setDataInfo(buildInference());
			oafBuilder.setTimestamp(System.currentTimeMillis());
			return oafBuilder.build();
		}

	}
	@Override
	public AlgorithmName getAlgorithName() {
		return AlgorithmName.document_similarities_standard;
	}

	@Override
	public ActionBuilderModule<DocumentSimilarity> instantiate(
			String predefinedTrust, String actionSetId) {
		return new DocumentSimilarityActionBuilderModule(
				predefinedTrust, actionSetId);
	}
}
