package eu.dnetlib.data.collective.transformation.engine;

import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import eu.dnetlib.common.profile.ResourceDao;
import eu.dnetlib.data.collective.transformation.IDatabaseConnector;
import eu.dnetlib.data.collective.transformation.TransformationException;
import eu.dnetlib.data.collective.transformation.VocabularyRegistry;
import eu.dnetlib.data.collective.transformation.core.xsl.ext.TransformationFunctionProxy;
import eu.dnetlib.data.collective.transformation.engine.core.ITransformation;
import eu.dnetlib.data.collective.transformation.engine.functions.Convert;
import eu.dnetlib.data.collective.transformation.engine.functions.Dblookup;
import eu.dnetlib.data.collective.transformation.engine.functions.Extract;
import eu.dnetlib.data.collective.transformation.engine.functions.IFeatureExtraction;
import eu.dnetlib.data.collective.transformation.engine.functions.RegularExpression;
import eu.dnetlib.data.collective.transformation.engine.functions.RetrieveValue;
import eu.dnetlib.data.collective.transformation.rulelanguage.util.FunctionCall;

/**
 * @author jochen
 *
 */
public class SimpleTransformationEngine{

	private static Log log = LogFactory.getLog(SimpleTransformationEngine.class);
	private ITransformation transformation;
	private VocabularyRegistry vocabularyRegistry;
	private IDatabaseConnector databaseConnector;
	private ResourceDao resourceDao;
	private IFeatureExtraction featureExtraction;
	private final List<String> ignoreRecords = new LinkedList<String>();
	private final List<String> mdRecords = new LinkedList<String>();
	private long totalTransformedRecords = 0;
	private long totalIgnoredRecords = 0;
	private String mappingFile;
	
	
	/**
	 * execute any preprocessings declared in the transformation script prior starting the transformation of records
	 */
	public void preprocess(){
		for ( Map<String, String> preprocMap: this.transformation.getRuleLanguageParser().getPreprocessings()){
			Iterator<String> it = preprocMap.keySet().iterator();
			while(it.hasNext()){
				String function = it.next();
				if (function.equals("dblookup")){
					Dblookup fun = new Dblookup();
					fun.setDbConnector(databaseConnector);
					try{
						log.debug("preprocessingMap value: " + preprocMap.get(function));
						TransformationFunctionProxy.getInstance().setLookupRecord(fun.getResults(preprocMap.get(function)));
					}catch(Exception e){
						log.debug(e.getMessage());
						throw new IllegalStateException(e);
					}
				}
			}
		}
		log.debug("preprocessing done.");
	}
	
	/**
	 * transforms a source record
	 * @param sourceRecord the record to transform
	 * @return transformed record
	 */
	public String transform(String sourceRecord){
		List<String> objectRecords = new LinkedList<String>();
		objectRecords.add(sourceRecord);
		int index = 0;
		mdRecords.clear();
		ignoreRecords.clear();
		initTransformationFunction();
		TransformationFunctionProxy.getInstance().setIgnoreRecordList(ignoreRecords);
		if (!transformation.getRuleLanguageParser().isXslStylesheet()){
			// iterate over all rules which are functionCalls
			log.debug("functionCalls size: " + transformation.getRuleLanguageParser().getFunctionCalls().size());
			for (FunctionCall functionCall: transformation.getRuleLanguageParser().getFunctionCalls()){
				preprocess(objectRecords, functionCall);
			}			
		}
		for (String record: objectRecords){
			//log.debug(record);
			try {
				log.debug("now run transformation for record with index: " + index);
				String transformedRecord = transformation.transformRecord(record, index); 
				if (!ignoreRecords.contains("" + index))
					mdRecords.add(transformedRecord);
				else{
					// TODO analyse record-id and log it
				}
			} catch (TransformationException e) {
				throw new IllegalStateException(e);
			}
			index++;
		}
		totalTransformedRecords = totalTransformedRecords + mdRecords.size();
		totalIgnoredRecords = totalIgnoredRecords + ignoreRecords.size();
		log.debug("objRecordSize: " + objectRecords.size() + ", mdRecordSize: " + mdRecords.size() + ", ignoredRecordSize: " + ignoreRecords.size());
		return mdRecords.get(0);
	}
	
	private void initTransformationFunction(){
		if (this.vocabularyRegistry == null){
			throw new IllegalStateException("vocabularyReg is null");
		}
		Convert convertFunction = new Convert();
		convertFunction.setVocabularyRegistry(this.vocabularyRegistry);
		TransformationFunctionProxy.getInstance().setConvertFunction(convertFunction);
		
	}
	
	/**
	 * preprocesses function if function is configured resp.
	 * @param records list of object records
	 * @param aFunctionCall 
	 */
	private void preprocess(List<String> records, FunctionCall aFunctionCall){
		try{
			log.debug("preprocess");
			if (transformation.getRuleLanguageParser() == null){
				throw new IllegalStateException("rulelanguageparser not initialised");
			}
			if (transformation.getRuleLanguageParser().getNamespaceDeclarations() == null){
				throw new IllegalStateException("nsDecl is null");
			}
			PreProcessor preProc = new PreProcessor();
			preProc.setConvertFunction(TransformationFunctionProxy.getInstance().getConvertFunction());
			RetrieveValue retrieveValue = new RetrieveValue();
			retrieveValue.setResourceDao(resourceDao);
			preProc.setRetrieveFunction(retrieveValue);
			RegularExpression regExpr = new RegularExpression();
			preProc.setRegExprFunction(regExpr);
			TransformationFunctionProxy functionProxy = TransformationFunctionProxy.getInstance();
			preProc.setFunctionProxy(functionProxy);
			Extract extractFunction = new Extract();
			extractFunction.setFeatureExtraction(featureExtraction);
			preProc.setExtractFunction(extractFunction);
			if (aFunctionCall.doPreprocess() || aFunctionCall.isStatic()){
				//log.debug("now call preprocess with: " + aFunctionCall.getExternalFunctionName() + " " + aFunctionCall.getUuid());
				preProc.preprocess(
						aFunctionCall, 
						records, 
						transformation.getRuleLanguageParser().getNamespaceDeclarations(), 
						transformation.getStaticTransformationResults(), 
						transformation.getJobProperties(),
						transformation.getRuleLanguageParser().getVariableMappingRules());
				//log.debug("preprocess end");				
			}else{
				log.debug("skip preprocessing for function: " + aFunctionCall.getExternalFunctionName());
			}
			
		}catch(Exception e){
			throw new IllegalStateException(e);
		}
		
	}
	
	
	/**
	 * @param transformation the transformation to set
	 */
	public void setTransformation(ITransformation transformation) {
		this.transformation = transformation;
	}
	
	/**
	 * @return the transformation
	 */
	public ITransformation getTransformation() {
		return transformation;
	}

	/**
	 * @param vocabularyRegistry the vocabularyRegistry to set
	 */
	public void setVocabularyRegistry(VocabularyRegistry vocabularyRegistry) {
		this.vocabularyRegistry = vocabularyRegistry;
	}

	/**
	 * @return the vocabularyRegistry
	 */
	public VocabularyRegistry getVocabularyRegistry() {
		return vocabularyRegistry;
	}
	
	/**
	 * @return the resourceDao
	 */
	public ResourceDao getResourceDao() {
		return resourceDao;
	}

	/**
	 * @param resourceDao the resourceDao to set
	 */
	public void setResourceDao(ResourceDao resourceDao) {
		this.resourceDao = resourceDao;
	}

	/**
	 * @param featureExtraction the featureExtraction to set
	 */
	public void setFeatureExtraction(IFeatureExtraction featureExtraction) {
		this.featureExtraction = featureExtraction;
	}

	/**
	 * @return the featureExtraction
	 */
	public IFeatureExtraction getFeatureExtraction() {
		return featureExtraction;
	}

	/**
	 * @return the databaseConnector
	 */
	public IDatabaseConnector getDatabaseConnector() {
		return databaseConnector;
	}

	/**
	 * @param databaseConnector the databaseConnector to set
	 */
	public void setDatabaseConnector(IDatabaseConnector databaseConnector) {
		this.databaseConnector = databaseConnector;
	}
	
	public long getTotalTransformedRecords(){
		return this.totalTransformedRecords;
	}
	
	public long getTotalIgnoredRecords(){
		return this.totalIgnoredRecords;
	}

	/**
	 * @return the mappingFile
	 */
	public String getMappingFile() {
		return mappingFile;
	}

	/**
	 * @param mappingFile the mappingFile to set
	 */
	public void setMappingFile(String mappingFile) {
		this.mappingFile = mappingFile;
	}
}
