package eu.dnetlib.espas.gui.server;

import eu.dnetlib.espas.gui.shared.ActiveTopicQuestions;
import eu.dnetlib.espas.gui.shared.Question;
import eu.dnetlib.espas.gui.shared.Topic;
import org.apache.log4j.Logger;
import org.springframework.jdbc.datasource.DataSourceUtils;
import org.springframework.transaction.annotation.Transactional;

import javax.sql.DataSource;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * Created by stefania on 10/14/15.
 */
@Transactional(readOnly = false)
public class FAQServiceCore {

    private DataSource datasource = null;

    private static Logger logger = Logger.getLogger(FAQServiceCore.class);

    private final static String GET_TOPICS = "select * from topic order by weight desc;";

    private final static String INSERT_TOPIC = "insert into topic (name, description, weight, questionorder) values (?, ?, ?, ?);";

    private final static String UPDATE_TOPIC = "update topic set name = ?, description = ?, weight = ?, questionorder = ? where topicid = ? ;";

    private final static String DELETE_TOPIC = "delete from topic where topicid = ? ;";

    private final static String SET_QUESTION_ORDER = "update topic set questionorder = ? where topicid = ? ;";

    private final static String GET_QUESTIONS = "select f.faqid as faqid, f.question as question, f.answer as answer, " +
            "f.date as faqdate, f.active as active, f.weight as questionweight, f.hitcount as hitcount, t.topicid as topicid, " +
            "t.name as name, t.description as description, t.weight as topicweight, t.questionorder as questionorder, " +
            "t.date as topicdate " +
            "from faq as f " +
            "join topic t on f.topic=t.topicid " +
            "order by topicid, questionweight desc;";

    private final static String GET_QUESTIONS_BY_ID = "select f.faqid as faqid, f.question as question, f.answer as answer, " +
            "f.date as faqdate, f.active as active, f.weight as questionweight, f.hitcount as hitcount, t.topicid as topicid, " +
            "t.name as name, t.description as description, t.weight as topicweight, t.questionorder as questionorder, " +
            "t.date as topicdate " +
            "from faq as f " +
            "join topic t on f.topic=t.topicid " +
            "where f.topic = ? " +
            "order by topicid, questionweight desc;";

    private final static String INSERT_QUESTION = "insert into faq (question, answer, topic, weight) values (?, ?, ?, ?);";

    private final static String UPDATE_QUESTION = "update faq set question = ?, answer = ?, topic = ?, weight = ? where faqid = ? ;";

    private final static String DELETE_QUESTION = "delete from faq where faqid = ? ;";

    private final static String SET_QUESTION_ACTIVE = "update faq set active = true where faqid = ? ;";

    private final static String SET_QUESTION_INACTIVE = "update faq set active = false where faqid = ? ;";

    private final static String INCREMENT_QUESTION_HIT_COUNT = "update faq set hitcount = hitcount + 1 where faqid = ? ;";

    private final static String GET_ACTIVE_FAQS_BY_HITS = "select f.faqid as faqid, f.question as question, f.answer as answer, " +
            "f.date as faqdate, f.active as active, f.weight as questionweight, f.hitcount as hitcount " +
            "from faq as f " +
            "where f.active=true and f.topic=?" +
            "order by f.hitcount";

    private final static String GET_ACTIVE_FAQS_BY_WEIGHT = "select f.faqid as faqid, f.question as question, f.answer as answer, " +
            "f.date as faqdate, f.active as active, f.weight as questionweight, f.hitcount as hitcount " +
            "from faq as f " +
            "where f.active=true and f.topic=? " +
            "order by f.weight desc";

    public static final String GET_ACTIVE_TOPICS = "select t.topicid as topicid, " +
            "t.name as name, t.description as description, t.weight as topicweight, t.questionorder as questionorder, " +
            "t.date as topicdate " +
            "from topic t where exists (select 1 from faq f where f.active=true and f.topic=t.topicid) order by t.weight";

    public List<Topic> getTopics() {

        List<Topic> topics = new ArrayList<Topic>();

        try {

            Connection con = DataSourceUtils.getConnection(datasource);
            PreparedStatement stmt = con.prepareStatement(GET_TOPICS);

            ResultSet rs = stmt.executeQuery();

            while (rs.next()) {

                Topic topic = new Topic();
                topic.setId(rs.getInt("topicid"));
                topic.setTopicName(rs.getString("name"));
                topic.setTopicDescription(rs.getString("description"));
                topic.setDate(rs.getTimestamp("date"));
                topic.setWeight(rs.getFloat("weight"));
                topic.setQuestionOrder(rs.getString("questionorder"));

                topics.add(topic);
            }

            rs.close();
            stmt.close();
            DataSourceUtils.releaseConnection(con, datasource);

        } catch (SQLException e) {
            logger.error("Failed to get valid topics", e);
        }

        return topics;
    }

    public void insertTopic(Topic topic) {

        try {

            Connection con = DataSourceUtils.getConnection(datasource);
            PreparedStatement stmt = con.prepareStatement(INSERT_TOPIC);

            stmt.setString(1, topic.getTopicName());
            stmt.setString(2, topic.getTopicDescription());
            stmt.setFloat(3, topic.getWeight());
            stmt.setString(4, topic.getQuestionOrder());

            stmt.executeUpdate();

            stmt.close();
            DataSourceUtils.releaseConnection(con, datasource);

        } catch (SQLException e) {
            logger.error("Failed to insert topic", e);
        }
    }

    public void updateTopic(Topic topic) {

        try {

            Connection con = DataSourceUtils.getConnection(datasource);
            PreparedStatement stmt = con.prepareStatement(UPDATE_TOPIC);

            stmt.setString(1, topic.getTopicName());
            stmt.setString(2, topic.getTopicDescription());
            stmt.setFloat(3, topic.getWeight());
            stmt.setString(4, topic.getQuestionOrder());
            stmt.setInt(5, topic.getId());

            stmt.executeUpdate();

            stmt.close();
            DataSourceUtils.releaseConnection(con, datasource);

        } catch (SQLException e) {
            logger.error("Failed to update topic", e);
        }
    }

    public void deleteTopic(String topicId) {

        try {

            Connection con = DataSourceUtils.getConnection(datasource);
            PreparedStatement stmt = con.prepareStatement(DELETE_TOPIC);

            stmt.setInt(1, Integer.parseInt(topicId));

            stmt.executeUpdate();

            stmt.close();
            DataSourceUtils.releaseConnection(con, datasource);

        } catch (SQLException e) {
            logger.error("Failed to delete topic", e);
        }
    }

    public void setWeightQuestionOrder(String topicId) {

        try {

            Connection con = DataSourceUtils.getConnection(datasource);
            PreparedStatement stmt = con.prepareStatement(SET_QUESTION_ORDER);

            stmt.setString(1, "weight");
            stmt.setInt(2, Integer.parseInt(topicId));

            stmt.executeUpdate();

            stmt.close();
            DataSourceUtils.releaseConnection(con, datasource);

        } catch (SQLException e) {
            logger.error("Failed to update weight question order for topic", e);
        }
    }

    public void setHitCountQuestionOrder(String topicId) {

        try {

            Connection con = DataSourceUtils.getConnection(datasource);
            PreparedStatement stmt = con.prepareStatement(SET_QUESTION_ORDER);

            stmt.setString(1, "hitCount");
            stmt.setInt(2, Integer.parseInt(topicId));

            stmt.executeUpdate();

            stmt.close();
            DataSourceUtils.releaseConnection(con, datasource);

        } catch (SQLException e) {
            logger.error("Failed to update hit count question order for topic", e);
        }
    }

    public List<Question> getQuestions() {

        List<Question> questions = new ArrayList<Question>();

        try {

            Connection con = DataSourceUtils.getConnection(datasource);
            PreparedStatement stmt = con.prepareStatement(GET_QUESTIONS);

            ResultSet rs = stmt.executeQuery();

            while (rs.next()) {

                Question question = new Question();
                question.setId(rs.getInt("faqid"));
                question.setQuestion(rs.getString("question"));
                question.setAnswer(rs.getString("answer"));
                question.setDate(rs.getTimestamp("faqdate"));
                question.setIsActive(rs.getBoolean("active"));
                question.setWeight(rs.getFloat("questionweight"));
                question.setHitCount(rs.getInt("hitcount"));

                Topic topic = new Topic();
                topic.setId(rs.getInt("topicid"));
                topic.setTopicName(rs.getString("name"));
                topic.setTopicDescription(rs.getString("description"));
                topic.setDate(rs.getTimestamp("topicdate"));
                topic.setWeight(rs.getFloat("topicweight"));
                topic.setQuestionOrder("questionorder");
                question.setTopic(topic);

                questions.add(question);
            }

            rs.close();
            stmt.close();
            DataSourceUtils.releaseConnection(con, datasource);

        } catch (SQLException e) {
            logger.error("Failed to get valid questions", e);
        }

        return questions;
    }

    public List<Question> getQuestionsById(String topicId) {

        List<Question> questions = new ArrayList<Question>();

        try {

            Connection con = DataSourceUtils.getConnection(datasource);
            PreparedStatement stmt = con.prepareStatement(GET_QUESTIONS_BY_ID);

            stmt.setInt(1, Integer.parseInt(topicId));

            ResultSet rs = stmt.executeQuery();

            while (rs.next()) {

                Question question = new Question();
                question.setId(rs.getInt("faqid"));
                question.setQuestion(rs.getString("question"));
                question.setAnswer(rs.getString("answer"));
                question.setDate(rs.getTimestamp("faqdate"));
                question.setIsActive(rs.getBoolean("active"));
                question.setWeight(rs.getFloat("questionweight"));
                question.setHitCount(rs.getInt("hitcount"));

                Topic topic = new Topic();
                topic.setId(rs.getInt("topicid"));
                topic.setTopicName(rs.getString("name"));
                topic.setTopicDescription(rs.getString("description"));
                topic.setDate(rs.getTimestamp("topicdate"));
                topic.setWeight(rs.getFloat("topicweight"));
                topic.setQuestionOrder("questionorder");
                question.setTopic(topic);

                questions.add(question);
            }

            rs.close();
            stmt.close();
            DataSourceUtils.releaseConnection(con, datasource);

        } catch (SQLException e) {
            logger.error("Failed to get valid questions", e);
        }

        return questions;
    }

    public void insertQuestion(Question question) {

        try {

            Connection con = DataSourceUtils.getConnection(datasource);
            PreparedStatement stmt = con.prepareStatement(INSERT_QUESTION);

            stmt.setString(1, question.getQuestion());
            stmt.setString(2, question.getAnswer());
            stmt.setInt(3, question.getTopic().getId());
            stmt.setFloat(4, question.getWeight());

            stmt.executeUpdate();

            stmt.close();
            DataSourceUtils.releaseConnection(con, datasource);

        } catch (SQLException e) {
            logger.error("Failed to insert question", e);
        }
    }

    public void updateQuestion(Question question) {

        try {

            Connection con = DataSourceUtils.getConnection(datasource);
            PreparedStatement stmt = con.prepareStatement(UPDATE_QUESTION);

            stmt.setString(1, question.getQuestion());
            stmt.setString(2, question.getAnswer());
            stmt.setInt(3, question.getTopic().getId());
            stmt.setFloat(4, question.getWeight());
            stmt.setInt(5, question.getId());

            stmt.executeUpdate();

            stmt.close();
            DataSourceUtils.releaseConnection(con, datasource);

        } catch (SQLException e) {
            logger.error("Failed to update question", e);
        }
    }

    public void deleteQuestion(String questionId) {

        try {

            Connection con = DataSourceUtils.getConnection(datasource);
            PreparedStatement stmt = con.prepareStatement(DELETE_QUESTION);

            stmt.setInt(1, Integer.parseInt(questionId));

            stmt.executeUpdate();

            stmt.close();
            DataSourceUtils.releaseConnection(con, datasource);

        } catch (SQLException e) {
            logger.error("Failed to delete question", e);
        }
    }

    public void setQuestionActive(String questionId) {

        try {

            Connection con = DataSourceUtils.getConnection(datasource);
            PreparedStatement stmt = con.prepareStatement(SET_QUESTION_ACTIVE);

            stmt.setInt(1, Integer.parseInt(questionId));

            stmt.executeUpdate();

            stmt.close();
            DataSourceUtils.releaseConnection(con, datasource);

        } catch (SQLException e) {
            logger.error("Failed to set question active", e);
        }
    }

    public void setQuestionInactive(String questionId) {

        try {

            Connection con = DataSourceUtils.getConnection(datasource);
            PreparedStatement stmt = con.prepareStatement(SET_QUESTION_INACTIVE);

            stmt.setInt(1, Integer.parseInt(questionId));

            stmt.executeUpdate();

            stmt.close();
            DataSourceUtils.releaseConnection(con, datasource);

        } catch (SQLException e) {
            logger.error("Failed to set question inactive", e);
        }
    }

    public void incrementQuestionHitCount(String questionId) {

        try {

            Connection con = DataSourceUtils.getConnection(datasource);
            PreparedStatement stmt = con.prepareStatement(INCREMENT_QUESTION_HIT_COUNT);

            stmt.setInt(1, Integer.parseInt(questionId));

            stmt.executeUpdate();

            stmt.close();
            DataSourceUtils.releaseConnection(con, datasource);

        } catch (SQLException e) {
            logger.error("Failed to increment question hit count", e);
        }
    }

    public List<ActiveTopicQuestions> getActiveFAQs() {

        List<ActiveTopicQuestions> activeTopicQuestionsList = new ArrayList<ActiveTopicQuestions>();

        Map<Integer, Topic> topicsMap = new HashMap<Integer, Topic>();
        Map<Integer, List<Question>> questionsByTopicMap = new HashMap<Integer, List<Question>>();

        try {
            Connection con = DataSourceUtils.getConnection(datasource);
            PreparedStatement stmt = con.prepareStatement(GET_ACTIVE_TOPICS);
            ResultSet rs = stmt.executeQuery();

            while (rs.next()) {
                Topic topic = new Topic();

                topic.setId(rs.getInt("topicid"));
                topic.setTopicName(rs.getString("name"));
                topic.setTopicDescription(rs.getString("description"));
                topic.setDate(rs.getTimestamp("topicdate"));
                topic.setWeight(rs.getFloat("topicweight"));
                topic.setQuestionOrder("questionorder");

                topicsMap.put(topic.getId(), topic);
                questionsByTopicMap.put(topic.getId(), new ArrayList<Question>());
            }

            rs.close();
            stmt.close();

            for (Topic topic:topicsMap.values()) {
                if (topic.getQuestionOrder().equals("weight"))
                    stmt = con.prepareStatement(GET_ACTIVE_FAQS_BY_WEIGHT);
                else
                    stmt = con.prepareStatement(GET_ACTIVE_FAQS_BY_HITS);

                stmt.setInt(1, topic.getId());

                rs = stmt.executeQuery();

                while (rs.next()) {

                    Question question = new Question();
                    question.setId(rs.getInt("faqid"));
                    question.setQuestion(rs.getString("question"));
                    question.setAnswer(rs.getString("answer"));
                    question.setDate(rs.getTimestamp("faqdate"));
                    question.setIsActive(rs.getBoolean("active"));
                    question.setWeight(rs.getFloat("questionweight"));
                    question.setHitCount(rs.getInt("hitcount"));
                    question.setTopic(topic);

                    questionsByTopicMap.get(topic.getId()).add(question);
                }

                rs.close();
                stmt.close();
            }

            for(int topicId : topicsMap.keySet()) {

                ActiveTopicQuestions activeTopicQuestions = new ActiveTopicQuestions();
                activeTopicQuestions.setTopic(topicsMap.get(topicId));
                activeTopicQuestions.setQuestionList(questionsByTopicMap.get(topicId));

                activeTopicQuestionsList.add(activeTopicQuestions);
            }

            DataSourceUtils.releaseConnection(con, datasource);

        } catch (SQLException e) {
            logger.error("Failed to get active FAQs", e);
        }

        return activeTopicQuestionsList;
    }

    public DataSource getDatasource() {
        return datasource;
    }

    public void setDatasource(DataSource datasource) {
        this.datasource = datasource;
    }
}
