package eu.dnetlib.espas.gui.server.user;

import eu.dnetlib.espas.gui.shared.DataProvider;
import eu.dnetlib.espas.gui.shared.DataProviderStatus;
import eu.dnetlib.espas.gui.shared.UserAccessException;
import org.apache.log4j.Logger;
import org.springframework.jdbc.datasource.DataSourceUtils;
import org.springframework.transaction.annotation.Transactional;

import javax.sql.DataSource;
import java.sql.*;
import java.util.*;

/**
 * Created by stefania on 10/29/14.
 */
@Transactional(readOnly = false)
public class DataProviderAccessServiceCore {

    private DataSource datasource = null;

    private static Logger logger = Logger.getLogger(DataProviderAccessServiceCore.class);

    private final static String GET_DATAPROVIDER_BY_USER = "select * from " +
            "(select namespace, name, url, terms, sosserviceurl, latitude, longitude, shortname, array_agg(dpe.provideradmin) as admins " +
            "from dataprovider as dp " +
            "join dataprovider_espasuser as dpe on dp.namespace=dpe.dataprovider " +
            "group by namespace, name, url, terms, sosserviceurl, latitude, longitude ) as foo " +
            "where ARRAY[?]::text[] <@ foo.admins ;";

    private final static String SAVE_DATAPROVIDER = "insert into dataprovider (namespace, name, url, terms, sosserviceurl, " +
            "latitude, longitude, shortname) values (?, ?, ?, ?, ?, ?, ?, ?) ;";

    private final static String SAVE_DATAPROVIDER_ADMIN = "insert into dataprovider_espasuser (dataprovider, provideradmin) values (?, ?) ;";

    private final static String DELETE_DATAPROVIDER_ADMINS = "delete from dataprovider_espasuser as dpe where dpe.dataprovider = ? ;";

    private final static String UPDATE_DATAPROVIDER = "update dataprovider set url=?, name=?, terms=?, sosserviceurl=?, " +
            "latitude=?, longitude=?, shortname=? where namespace=? ;";

    private final static String DELETE_DATAPROVIDER = "delete from dataprovider as dp where dp.namespace = ? ;";

    private final static String GET_DATAPROVIDER = "select * from dataprovider as dp where dp.namespace = ? ;";

    private final static String GET_DATAPROVIDER_ADMINS = "select * from dataprovider_espasuser where dataprovider = ? ;";

    private final static String GET_DATAPROVIDERS = "select * from dataprovider order by name ;";

    private final static String GET_ALL_DATAPROVIDER_STATUSES = "select namespace, name, status, bar.date \n" +
            "from dataprovider dp \n" +
            "left join (\n" +
            "\tselect dataprovider, max(date) as date \n" +
            "\tfrom dataprovider_status \n" +
            "\tgroup by dataprovider) foo on dp.namespace= foo.dataprovider\n" +
            "left join (\n" +
            "\tselect status, date\n" +
            "\tfrom dataprovider_status dps) bar on bar.date=foo.date\n" +
            "group by bar.date, namespace, name, status \n" +
            "order by name";

    private final static String GET_DATAPROVIDER_STATUSES = "select namespace, name, status, bar.date \n" +
            "from dataprovider dp \n" +
            "left join (\n" +
            "\tselect dataprovider, max(date) as date \n" +
            "\tfrom dataprovider_status \n" +
            "\tgroup by dataprovider) foo on dp.namespace= foo.dataprovider \n" +
            "left join (\n" +
            "\tselect status, date\n" +
            "\tfrom dataprovider_status dps) bar on bar.date=foo.date\n" +
            "where ARRAY[dataprovider]::text[] <@ ?::text[] \n" +
            "group by bar.date, namespace, name, status \n" +
            "order by name";

    private final static String GET_ALL_COUNT = "select count(*) from identifier where namespace=? ;";

    private final static String GET_OBSERVATION_COUNT = "select count(*) from identifier where type='observation' and namespace=? ;";

    private final static String GET_INDIVIDUAL_COUNT = "select count(*) from identifier where type='individual' and namespace=? ;";

    private final static String GET_ORGANIZATION_COUNT = "select count(*) from identifier where type='organisation' and namespace=? ;";

    private final static String GET_PLATFORM_COUNT = "select count(*) from identifier where type='platform' and namespace=? ;";

    private final static String GET_PROJECT_COUNT = "select count(*) from identifier where type='project' and namespace=? ;";

    private final static String GET_INSTRUMENT_COUNT = "select count(*) from identifier where type='instrument' and namespace=? ;";

    private final static String GET_COMPUTATION_COUNT = "select count(*) from identifier where type='computation' and namespace=? ;";

    private final static String GET_OPERATION_COUNT = "select count(*) from identifier where type='operation' and namespace=? ;";

    private final static String GET_ACQUISITION_COUNT = "select count(*) from identifier where type='acquisition' and namespace=? ;";

    private final static String GET_COMPOSITE_PROCESS_COUNT = "select count(*) from identifier where type='compositeProcess' and namespace=? ;";

    private final static String GET_OBSERVATION_COLLECTION_COUNT = "select count(*) from identifier where type='observationCollection' and namespace=? ;";


    public List<DataProvider> getDataProvidersByUser(String userId)
            throws UserAccessException {

        List<DataProvider> dataProviders = new ArrayList<DataProvider>();

        try {

            Connection con = DataSourceUtils.getConnection(datasource);
            PreparedStatement stmt = con.prepareStatement(GET_DATAPROVIDER_BY_USER);
            stmt.setString(1, userId);

            ResultSet rs = stmt.executeQuery();


            while (rs.next()) {

                DataProvider dataProvider = new DataProvider();
                dataProvider.setName(rs.getString("name"));
                dataProvider.setShortName(rs.getString("shortname"));

                String fullNamespace = rs.getString("namespace");
                String[] namespaceParts = fullNamespace.split("/");
                dataProvider.setNamespace(namespaceParts[namespaceParts.length-1]);

                dataProvider.setWrapperURL(rs.getString("url"));

                dataProvider.setSosServiceURL(rs.getString("sosserviceurl"));
                dataProvider.setLatitude(rs.getDouble("latitude"));
                dataProvider.setLongitude(rs.getDouble("longitude"));

                dataProvider.setTerms(rs.getString("terms"));

                Array admins = rs.getArray("admins");
                String[] adminsArray = (String[]) admins.getArray();
                List<String> administrators = Arrays.asList(adminsArray);
                dataProvider.setAdministratorEmails(administrators);

                dataProviders.add(dataProvider);
            }

            rs.close();
            stmt.close();
            DataSourceUtils.releaseConnection(con, datasource);

        } catch (SQLException e) {

            logger.error("Failed to get data providers by user", e);
            throw new UserAccessException(e, UserAccessException.ErrorCode.SQL_ERROR);
        }

        return dataProviders;
    }

    public void saveDataProvider(DataProvider dataProvider)
            throws UserAccessException {

        try {
            String namespace = "http://resources.espas-fp7.eu/provider/" + dataProvider.getNamespace();

            Connection con = DataSourceUtils.getConnection(datasource);
            PreparedStatement stmt = con.prepareStatement(GET_DATAPROVIDER);
            stmt.setString(1, namespace);

            ResultSet rs = stmt.executeQuery();

            if(rs.next()) {

                throw new UserAccessException(UserAccessException.ErrorCode.USER_ALREADY_EXISTS);
            }

            stmt = con.prepareStatement(SAVE_DATAPROVIDER);

            stmt.setString(1, namespace);
            stmt.setString(2, dataProvider.getName());
            stmt.setString(3, dataProvider.getWrapperURL());
            stmt.setString(4, dataProvider.getTerms());
            stmt.setString(5, dataProvider.getSosServiceURL());
            stmt.setDouble(6, dataProvider.getLatitude());
            stmt.setDouble(7, dataProvider.getLongitude());
            stmt.setString(8, dataProvider.getShortName());

            stmt.executeUpdate();

            for(String admin : dataProvider.getAdministratorEmails()) {

                stmt = con.prepareStatement(SAVE_DATAPROVIDER_ADMIN);
                stmt.setString(1, namespace);
                stmt.setString(2, admin);

                stmt.executeUpdate();
            }

            stmt.close();
            DataSourceUtils.releaseConnection(con, datasource);

        } catch (SQLException e) {

            logger.error("Failed to save data provider", e);
            throw new UserAccessException(e, UserAccessException.ErrorCode.SQL_ERROR);
        }

    }

    public void updateDataProvider(DataProvider dataProvider)
            throws UserAccessException {

        try {

            Connection con = DataSourceUtils.getConnection(datasource);
            String namespace = "http://resources.espas-fp7.eu/provider/" + dataProvider.getNamespace();

            PreparedStatement stmt = con.prepareStatement(DELETE_DATAPROVIDER_ADMINS);
            stmt.setString(1, namespace);
            stmt.executeUpdate();

            stmt = con.prepareStatement(UPDATE_DATAPROVIDER);
            stmt.setString(1, dataProvider.getWrapperURL());
            stmt.setString(2, dataProvider.getName());
            stmt.setString(3, dataProvider.getTerms());
            stmt.setString(4, dataProvider.getSosServiceURL());
            stmt.setDouble(5, dataProvider.getLatitude());
            stmt.setDouble(6, dataProvider.getLongitude());
            stmt.setString(7, dataProvider.getShortName());
            stmt.setString(8, namespace);
            stmt.executeUpdate();

            for(String admin : dataProvider.getAdministratorEmails()) {

                stmt = con.prepareStatement(SAVE_DATAPROVIDER_ADMIN);
                stmt.setString(1, namespace);
                stmt.setString(2, admin);

                stmt.executeUpdate();
            }

            stmt.close();
            DataSourceUtils.releaseConnection(con, datasource);

        } catch (SQLException e) {

            logger.error("Failed to update data provider", e);
            throw new UserAccessException(e, UserAccessException.ErrorCode.SQL_ERROR);
        }
    }

    public void deleteDataProvider(DataProvider dataProvider)
            throws UserAccessException {

        try {

            Connection con = DataSourceUtils.getConnection(datasource);
            PreparedStatement stmt = con.prepareStatement(DELETE_DATAPROVIDER);

            String namespace = "http://resources.espas-fp7.eu/provider/" + dataProvider.getNamespace();
            stmt.setString(1, namespace);

            stmt.executeUpdate();

            stmt.close();
            DataSourceUtils.releaseConnection(con, datasource);

        } catch (SQLException e) {

            logger.error("Failed to delete data provider", e);
            throw new UserAccessException(e, UserAccessException.ErrorCode.SQL_ERROR);
        }
    }

    public DataProvider getDataProviderByNamespace(String namespace)
            throws UserAccessException {

        DataProvider dataProvider = new DataProvider();

        try {
            Connection con = DataSourceUtils.getConnection(datasource);
            PreparedStatement stmt = con.prepareStatement(GET_DATAPROVIDER);
            stmt.setString(1, namespace);

            ResultSet rs = stmt.executeQuery();

            if(rs.next()) {
                dataProvider.setNamespace(rs.getString("namespace"));
                dataProvider.setWrapperURL(rs.getString("url"));
                dataProvider.setName(rs.getString("name"));
                dataProvider.setShortName(rs.getString("shortname"));
                dataProvider.setTerms(rs.getString("terms"));

                dataProvider.setSosServiceURL(rs.getString("sosserviceurl"));
                dataProvider.setLatitude(rs.getDouble("latitude"));
                dataProvider.setLongitude(rs.getDouble("longitude"));

                List<String> administrators = new ArrayList<String>();

                stmt = con.prepareStatement(GET_DATAPROVIDER_ADMINS);
                stmt.setString(1, namespace);

                rs = stmt.executeQuery();
                while(rs.next()) {
                    administrators.add(rs.getString("provideradmin"));
                }
                dataProvider.setAdministratorEmails(administrators);
            }

			DataSourceUtils.releaseConnection(con, datasource);
        } catch (SQLException e) {

            logger.error("Failed to get data provider by namespace", e);
            throw new UserAccessException(e, UserAccessException.ErrorCode.SQL_ERROR);
        }
        return dataProvider;
    }

    public List<DataProvider> getDataProviders() throws UserAccessException {

        List<DataProvider> dataProviders = new ArrayList<DataProvider>();

        try {

            Connection con = DataSourceUtils.getConnection(datasource);
            PreparedStatement stmt = con.prepareStatement(GET_DATAPROVIDERS);

            ResultSet rs = stmt.executeQuery();

            while (rs.next()) {

                DataProvider dataProvider = new DataProvider();
                dataProvider.setName(rs.getString("name"));
                dataProvider.setShortName(rs.getString("shortname"));

                String fullNamespace = rs.getString("namespace");
                String[] namespaceParts = fullNamespace.split("/");
                dataProvider.setNamespace(namespaceParts[namespaceParts.length-1]);

                dataProvider.setWrapperURL(rs.getString("url"));

                dataProvider.setSosServiceURL(rs.getString("sosserviceurl"));
                dataProvider.setLatitude(rs.getDouble("latitude"));
                dataProvider.setLongitude(rs.getDouble("longitude"));

                dataProvider.setTerms(rs.getString("terms"));

                List<String> administrators = new ArrayList<String>();
                PreparedStatement statement = con.prepareStatement(GET_DATAPROVIDER_ADMINS);
                statement.setString(1, fullNamespace);

                ResultSet resultSet = statement.executeQuery();

                while(resultSet.next()) {
                    administrators.add(resultSet.getString("provideradmin"));
                }
                dataProvider.setAdministratorEmails(administrators);

                dataProviders.add(dataProvider);
            }

            rs.close();
            stmt.close();
            DataSourceUtils.releaseConnection(con, datasource);

        } catch (SQLException e) {

            logger.error("Failed to get the list of data providers", e);
            throw new UserAccessException(e, UserAccessException.ErrorCode.SQL_ERROR);
        }

        return dataProviders;
    }

    public List<DataProviderStatus> getAllDataProviderStatuses() throws UserAccessException {

        List<DataProviderStatus> dataProviderStatuses = new ArrayList<DataProviderStatus>();

        try {

            Connection con = DataSourceUtils.getConnection(datasource);
            PreparedStatement stmt = con.prepareStatement(GET_ALL_DATAPROVIDER_STATUSES);

            ResultSet rs = stmt.executeQuery();

            while (rs.next()) {

                DataProviderStatus dataProviderStatus = new DataProviderStatus();
                dataProviderStatus.setNamespace(rs.getString("namespace"));
                dataProviderStatus.setName(rs.getString("name"));
                dataProviderStatus.setStatus(rs.getBoolean("status"));
                dataProviderStatus.setLastUpdateDate(rs.getTimestamp("date"));

                dataProviderStatuses.add(dataProviderStatus);
            }

            rs.close();
            stmt.close();
            DataSourceUtils.releaseConnection(con, datasource);

        } catch (SQLException e) {

            logger.error("Failed to get the data providers' statuses", e);
            throw new UserAccessException(e, UserAccessException.ErrorCode.SQL_ERROR);
        }

        return dataProviderStatuses;
    }

    public List<DataProviderStatus> getDataProviderStatuses(List<String> namespaces) throws UserAccessException {

        List<DataProviderStatus> dataProviderStatuses = new ArrayList<DataProviderStatus>();

        try {

            Connection con = DataSourceUtils.getConnection(datasource);
            PreparedStatement stmt = con.prepareStatement(GET_DATAPROVIDER_STATUSES);

            String ids[] = new String[namespaces.size()];

            for (int i = 0; i < namespaces.size(); i++)
                ids[i] = namespaces.get(i);

            Array idArray = con.createArrayOf("varchar", ids);

            stmt.setArray(1, idArray);

            ResultSet rs = stmt.executeQuery();

            while (rs.next()) {

                DataProviderStatus dataProviderStatus = new DataProviderStatus();
                dataProviderStatus.setNamespace(rs.getString("namespace"));
                dataProviderStatus.setName(rs.getString("name"));
                dataProviderStatus.setStatus(rs.getBoolean("status"));
                dataProviderStatus.setLastUpdateDate(rs.getTimestamp("date"));

                dataProviderStatuses.add(dataProviderStatus);
            }

            rs.close();
            stmt.close();
            DataSourceUtils.releaseConnection(con, datasource);

        } catch (SQLException e) {

            logger.error("Failed to get the list of data providers' statuses", e);
            throw new UserAccessException(e, UserAccessException.ErrorCode.SQL_ERROR);
        }

        return dataProviderStatuses;
    }

    public Map<String, Integer> getEntriesCount(String dataProviderId) throws UserAccessException {

        Map<String, Integer> entriesCount = new HashMap<String, Integer>();

        try {

            Connection con = DataSourceUtils.getConnection(datasource);

            PreparedStatement stmt = con.prepareStatement(GET_ALL_COUNT);
            stmt.setString(1, dataProviderId);
            ResultSet rs = stmt.executeQuery();

            if (rs.next()) {
                entriesCount.put("all", rs.getInt(1));
            }

            rs.close();
            stmt.close();

            stmt = con.prepareStatement(GET_OBSERVATION_COUNT);
            stmt.setString(1, dataProviderId);
            rs = stmt.executeQuery();

            if (rs.next()) {
                entriesCount.put("observation", rs.getInt(1));
            }

            rs.close();
            stmt.close();

            stmt = con.prepareStatement(GET_INDIVIDUAL_COUNT);
            stmt.setString(1, dataProviderId);
            rs = stmt.executeQuery();

            if (rs.next()) {
                entriesCount.put("individual", rs.getInt(1));
            }

            rs.close();
            stmt.close();

            stmt = con.prepareStatement(GET_ORGANIZATION_COUNT);
            stmt.setString(1, dataProviderId);
            rs = stmt.executeQuery();

            if (rs.next()) {
                entriesCount.put("organization", rs.getInt(1));
            }

            rs.close();
            stmt.close();

            stmt = con.prepareStatement(GET_PLATFORM_COUNT);
            stmt.setString(1, dataProviderId);
            rs = stmt.executeQuery();

            if (rs.next()) {
                entriesCount.put("platform", rs.getInt(1));
            }

            rs.close();
            stmt.close();

            stmt = con.prepareStatement(GET_PROJECT_COUNT);
            stmt.setString(1, dataProviderId);
            rs = stmt.executeQuery();

            if (rs.next()) {
                entriesCount.put("project", rs.getInt(1));
            }

            rs.close();
            stmt.close();

            stmt = con.prepareStatement(GET_INSTRUMENT_COUNT);
            stmt.setString(1, dataProviderId);
            rs = stmt.executeQuery();

            if (rs.next()) {
                entriesCount.put("instrument", rs.getInt(1));
            }

            rs.close();
            stmt.close();

            stmt = con.prepareStatement(GET_OPERATION_COUNT);
            stmt.setString(1, dataProviderId);
            rs = stmt.executeQuery();

            if (rs.next()) {
                entriesCount.put("operation", rs.getInt(1));
            }

            rs.close();
            stmt.close();

            stmt = con.prepareStatement(GET_COMPUTATION_COUNT);
            stmt.setString(1, dataProviderId);
            rs = stmt.executeQuery();

            if (rs.next()) {
                entriesCount.put("computation", rs.getInt(1));
            }

            rs.close();
            stmt.close();

            stmt = con.prepareStatement(GET_ACQUISITION_COUNT);
            stmt.setString(1, dataProviderId);
            rs = stmt.executeQuery();

            if (rs.next()) {
                entriesCount.put("acquisition", rs.getInt(1));
            }

            rs.close();
            stmt.close();

            stmt = con.prepareStatement(GET_COMPOSITE_PROCESS_COUNT);
            stmt.setString(1, dataProviderId);
            rs = stmt.executeQuery();

            if (rs.next()) {
                entriesCount.put("compositeProcess", rs.getInt(1));
            }

            rs.close();
            stmt.close();

            stmt = con.prepareStatement(GET_OBSERVATION_COLLECTION_COUNT);
            stmt.setString(1, dataProviderId);
            rs = stmt.executeQuery();

            if (rs.next()) {
                entriesCount.put("observationCollection", rs.getInt(1));
            }

            rs.close();
            stmt.close();

            DataSourceUtils.releaseConnection(con, datasource);

        } catch (SQLException e) {

            logger.error("Failed to get the data providers' statuses", e);
            throw new UserAccessException(e, UserAccessException.ErrorCode.SQL_ERROR);
        }

        return entriesCount;
    }

    public void deleteDataProvider(String dataProviderId) throws UserAccessException {

        try {

            Connection con = DataSourceUtils.getConnection(datasource);
            PreparedStatement stmt = con.prepareStatement(DELETE_DATAPROVIDER);

            String namespace = "http://resources.espas-fp7.eu/provider/" + dataProviderId;
            stmt.setString(1, namespace);

            stmt.executeUpdate();

            stmt.close();
            DataSourceUtils.releaseConnection(con, datasource);

        } catch (SQLException e) {

            logger.error("Failed to delete data provider", e);
            throw new UserAccessException(e, UserAccessException.ErrorCode.SQL_ERROR);
        }
    }

    public DataSource getDatasource() {
        return datasource;
    }

    public void setDatasource(DataSource datasource) {
        this.datasource = datasource;
    }
}
