package eu.dnetlib.espas.gui.server.user;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.List;

import javax.sql.DataSource;

import org.apache.commons.codec.digest.DigestUtils;
import org.apache.log4j.Logger;

import eu.dnetlib.espas.gui.shared.User;
import eu.dnetlib.espas.gui.shared.UserAccessException;
import eu.dnetlib.espas.gui.shared.UserAccessException.ErrorCode;
import org.springframework.jdbc.datasource.DataSourceUtils;
import org.springframework.transaction.annotation.Transactional;

@Transactional(readOnly = false)
public class UserDAOImpl implements UserDAO {
	
	private static Logger logger = Logger.getLogger(UserDAOImpl.class);
	
	private DataSource datasource = null;
	
	private final static String INSERT_USER = "INSERT INTO espasuser (email, name, password, domain, country, intendeduse, comments, organizationname, " +
			"pendingdataprovider, plainpassword, activationid) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?);";
	
	private final static String INSERT_USER_ROLE = "INSERT INTO espasuser_role (\"user\", role) VALUES (?, 'user')";
	
	private final static String GET_USER_BY_ID = "SELECT * FROM espasuser WHERE lower(email) = lower(?) ;";
	
	private final static String GET_USERS_ROLES = "SELECT role from espasuser_role as eur where lower(eur.\"user\") = lower(?) ";
	
	private final static String GET_USERS_NAMESPACES = "SELECT dataprovider from dataprovider_espasuser as dpe where lower(dpe.provideradmin) = lower(?) ";
	
	private final static String ACTIVATE_USER = "UPDATE espasuser set activated = TRUE where lower(email) = lower(?) ;";
	
	private final static String UPDATE_USER_TOKEN = "UPDATE espasuser set resetToken = ? where lower(email) = lower(?) ;";
	
	private final static String UPDATE_USER_PASSWORD = "UPDATE espasuser set password = ?, plainpassword = ? where lower(email) = lower(?) ;";
	
	private final static String UPDATE_USER = "UPDATE espasuser set name = ?, domain = ?, country = ?, intendeduse = ?, comments = ?, " +
			"organizationname = ?, pendingdataprovider = ? where lower(email) = lower(?) ";
	
	private final static String GET_USERS = "select * from espasuser order by email ;";
	
	private final static String ADD_ROLE_TO_USER = "INSERT INTO espasuser_role (\"user\", role) VALUES (?, ?);";
	
	private final static String REMOVE_ROLE_FROM_USER = "DELETE FROM espasuser_role WHERE \"user\" = ? AND role = ? ";
	
	private final static String UPDATE_USER_PENDINGDATAPROVIDER = "UPDATE espasuser SET pendingdataprovider = ? where lower(email) = lower(?) ;";
	
	private final static String GET_PENDING_DATAPROVIDER_USERS = "select * from espasuser where pendingdataprovider = true order by email ;";
	
	private final static String GET_DATAPROVIDER_USERS_EMAILS = "select * from espasuser eu join espasuser_role eur " +
			"on eu.email=eur.\"user\" where eur.role = 'dataprovider' order by eu.email ;";

    private final static String INSERT_USER_LOGIN_HISTORY = "INSERT INTO loginhistory (\"user\") VALUES (?);";

    private static final String GET_DATAPROVIDER_USERS = "select * from espasuser eu join espasuser_role eur on eu.email=eur.\"user\" " +
            "where eur.role='dataprovider' order by eu.name ;";
	
	@Override
	public void insertUser(User user) throws UserAccessException {
		
		try {
			
			Connection con = DataSourceUtils.getConnection(datasource);
			PreparedStatement stmt = con.prepareStatement(GET_USER_BY_ID);
			stmt.setString(1, user.getEmail());
			ResultSet rs = stmt.executeQuery();
			if(rs.next()) {
				throw new UserAccessException(ErrorCode.USER_ALREADY_EXISTS);
			}
			rs.close();
			stmt.close();
			
			stmt = con.prepareStatement(INSERT_USER);
			
			stmt.setString(1, user.getEmail());
			stmt.setString(2, user.getName());
			stmt.setString(3, DigestUtils.md5Hex(user.getPlainPassword()));
			stmt.setString(4, user.getDomain());
			stmt.setString(5, user.getCountry());
			stmt.setString(6, user.getIntendedUseOfData());
			stmt.setString(7, user.getComments());
			stmt.setString(8, user.getOrganisationName());
			stmt.setBoolean(9, user.isPendingDataProvider());
			stmt.setString(10, user.getPlainPassword());
			stmt.setString(11, user.getActivationId());
			
			stmt.executeUpdate();
			stmt.close();
			
			stmt = con.prepareStatement(INSERT_USER_ROLE);
			stmt.setString(1, user.getEmail());
			
			logger.debug(INSERT_USER_ROLE);
			
			stmt.executeUpdate();
			
			stmt.close();
			DataSourceUtils.releaseConnection(con, datasource);
			
		} catch (SQLException e) {
			logger.error("Failed to insert user", e);
			throw new UserAccessException(e, ErrorCode.SQL_ERROR);
		}
	}

	@Override
	public User getUserById(String userEmail, String password) throws UserAccessException {
		
		User user = new User();
		Connection con = null;
		PreparedStatement stmt = null;
		ResultSet rs = null;
		
		try {
			
			con = DataSourceUtils.getConnection(datasource);
			stmt = con.prepareStatement(GET_USER_BY_ID);
			stmt.setString(1, userEmail);

            rs = stmt.executeQuery();
			
			if(rs.next()) {
				
				if(!rs.getString("password").equals(DigestUtils.md5Hex(password))) 
					throw new UserAccessException(ErrorCode.INVALID_PASSWORD);

				if(!rs.getBoolean("activated")) 
					throw new UserAccessException(ErrorCode.NOT_ACTIVATED);
				
				user.setComments(rs.getString("comments"));
				user.setCountry(rs.getString("country"));
				user.setDomain(rs.getString("domain"));
				user.setEmail(rs.getString("email"));
				user.setIntendedUseOfData(rs.getString("intendeduse"));
				user.setName(rs.getString("name"));
				user.setOrganisationName(rs.getString("organizationname"));
				user.setPassword(rs.getString("password"));
				user.setPlainPassword(rs.getString("plainpassword"));
				user.setPendingDataProvider(rs.getBoolean("pendingdataprovider"));
				user.setActivationId(rs.getString("activationid"));
				user.setActivated(rs.getBoolean("activated"));
				user.setResetToken(rs.getString("resetToken"));
				
				stmt.close();
				
				stmt = con.prepareStatement(GET_USERS_NAMESPACES);
				stmt.setString(1, userEmail);
				ResultSet resultSet = stmt.executeQuery();
				
				List<String> namespaces = new ArrayList<String>();
				while(resultSet.next()) {
					
					String fullNamespace = resultSet.getString("dataprovider");
					String[] namespaceParts = fullNamespace.split("/");
					namespaces.add(namespaceParts[namespaceParts.length-1]);
				}
				user.setNamespaces(namespaces);
				
				resultSet.close();
				stmt.close();
				
				stmt = con.prepareStatement(GET_USERS_ROLES);
				stmt.setString(1, userEmail);
				resultSet = stmt.executeQuery();
				
				List<String> roles = new ArrayList<String>();
				while(resultSet.next()) {
					
					roles.add(resultSet.getString("role"));
				}
				user.setRoles(roles);
				
				resultSet.close();
                stmt.close();

                stmt = con.prepareStatement(INSERT_USER_LOGIN_HISTORY);
                stmt.setString(1, userEmail);
                stmt.executeUpdate();

                stmt.close();
				
			} else {
				throw new UserAccessException(ErrorCode.INVALID_USERNAME);
			}
			
		} catch (SQLException e) {
			
			logger.error("Failed to get user by id", e);
			throw new UserAccessException(e, ErrorCode.SQL_ERROR);
			
		} finally {
			try {				
				rs.close();
				stmt.close();
				DataSourceUtils.releaseConnection(con, datasource);
			} catch(Exception e) {
			}
		}
	
		return user;
	}

	@Override
	public User getUserById(String userEmail) throws UserAccessException {
		
		User user = new User();
		
		try {
			
			Connection con = DataSourceUtils.getConnection(datasource);
			PreparedStatement stmt = con.prepareStatement(GET_USER_BY_ID);
			stmt.setString(1, userEmail);
			
			ResultSet rs = stmt.executeQuery();
			
			if(rs.next()) {
				
				user.setComments(rs.getString("comments"));
				user.setCountry(rs.getString("country"));
				user.setDomain(rs.getString("domain"));
				user.setEmail(rs.getString("email"));
				user.setIntendedUseOfData(rs.getString("intendeduse"));
				user.setName(rs.getString("name"));
				user.setOrganisationName(rs.getString("organizationname"));
				user.setPassword(rs.getString("password"));
				user.setPlainPassword(rs.getString("plainpassword"));
				user.setPendingDataProvider(rs.getBoolean("pendingdataprovider"));
				user.setActivationId(rs.getString("activationid"));
				user.setActivated(rs.getBoolean("activated"));
				user.setResetToken(rs.getString("resetToken"));
				
				stmt.close();
				
				stmt = con.prepareStatement(GET_USERS_NAMESPACES);
				stmt.setString(1, userEmail);
				ResultSet resultSet = stmt.executeQuery();
				
				List<String> namespaces = new ArrayList<String>();
				while(resultSet.next()) {
					
					String fullNamespace = resultSet.getString("dataprovider");
					String[] namespaceParts = fullNamespace.split("/");
					namespaces.add(namespaceParts[namespaceParts.length-1]);
				}
				user.setNamespaces(namespaces);
				
				resultSet.close();
				stmt.close();
				
				stmt = con.prepareStatement(GET_USERS_ROLES);
				stmt.setString(1, userEmail);
				resultSet = stmt.executeQuery();
				
				List<String> roles = new ArrayList<String>();
				while(resultSet.next()) {
					
					roles.add(resultSet.getString("role"));
				}
				user.setRoles(roles);
				
				resultSet.close();
				
			} else {
				
				rs.close();
				stmt.close();
				DataSourceUtils.releaseConnection(con, datasource);
				throw new UserAccessException(ErrorCode.INVALID_USERNAME);
			}
			
			rs.close();
			stmt.close();
			DataSourceUtils.releaseConnection(con, datasource);
			
		} catch (SQLException e) {
			
			logger.error("Failed to get user by id", e);
			throw new UserAccessException(e, ErrorCode.SQL_ERROR);
		}
		
		return user;
	}

	@Override
	public void updateUser(User user) throws UserAccessException {
		
		try {
			Connection con = DataSourceUtils.getConnection(datasource);
			PreparedStatement stmt = con.prepareStatement(UPDATE_USER);
			stmt.setString(1, user.getName());
			stmt.setString(2, user.getDomain());
			stmt.setString(3, user.getCountry());
			stmt.setString(4, user.getIntendedUseOfData());
			stmt.setString(5, user.getComments());
			stmt.setString(6, user.getOrganisationName());
			stmt.setBoolean(7, user.isPendingDataProvider());
			stmt.setString(8, user.getEmail());
			
			stmt.executeUpdate();
			stmt.close();
			
			if(user.getPlainPassword()!=null) {
				stmt = con.prepareStatement(UPDATE_USER_PASSWORD);
				stmt.setString(1, DigestUtils.md5Hex(user.getPlainPassword()));
				stmt.setString(2, user.getPlainPassword());
				stmt.setString(3, user.getEmail());
				
				stmt.executeUpdate();
				stmt.close();
			}
			
			DataSourceUtils.releaseConnection(con, datasource);
			
		} catch (SQLException e) {
			logger.error("Failed to update user", e);
			throw new UserAccessException(e, ErrorCode.SQL_ERROR);
		}
	}

	public DataSource getDatasource() {
		return datasource;
	}

	public void setDatasource(DataSource datasource) {
		this.datasource = datasource;
	}

	@Override
	public User activateUser(User user) throws UserAccessException {
		
		Connection con = null;
		PreparedStatement stmt = null;
		ResultSet rs = null;
		
		try {
			
			con = DataSourceUtils.getConnection(datasource);
			stmt = con.prepareStatement(GET_USER_BY_ID);
			stmt.setString(1, user.getEmail());
			
			rs = stmt.executeQuery();
			
			if(rs.next()) {
				
				if(rs.getString("activationid").equals(user.getActivationId())) {
					
					user.setComments(rs.getString("comments"));
					user.setCountry(rs.getString("country"));
					user.setDomain(rs.getString("domain"));
					user.setEmail(rs.getString("email"));
					user.setIntendedUseOfData(rs.getString("intendeduse"));
					user.setName(rs.getString("name"));
					user.setOrganisationName(rs.getString("organizationname"));
					user.setPassword(rs.getString("password"));
					user.setPlainPassword(rs.getString("plainpassword"));
					user.setPendingDataProvider(rs.getBoolean("pendingdataprovider"));
					user.setActivationId(rs.getString("activationid"));
					user.setActivated(true);
					
					stmt.close();
					
					stmt = con.prepareStatement(GET_USERS_NAMESPACES);
					stmt.setString(1, user.getEmail());
					ResultSet resultSet = stmt.executeQuery();
					
					List<String> namespaces = new ArrayList<String>();
					while(resultSet.next()) {
						
						String fullNamespace = resultSet.getString("dataprovider");
						String[] namespaceParts = fullNamespace.split("/");
						namespaces.add(namespaceParts[namespaceParts.length-1]);
					}
					user.setNamespaces(namespaces);
					resultSet.close();
					stmt.close();
					
					stmt = con.prepareStatement(GET_USERS_ROLES);
					stmt.setString(1, user.getEmail());
					resultSet = stmt.executeQuery();
					
					List<String> roles = new ArrayList<String>();
					while(resultSet.next()) {
						
						roles.add(resultSet.getString("role"));
					}
					user.setRoles(roles);
					
					resultSet.close();
					
					stmt = con.prepareStatement(ACTIVATE_USER);
					stmt.setString(1, user.getEmail());
					
					stmt.executeUpdate();
                    stmt.close();

                    stmt = con.prepareStatement(INSERT_USER_LOGIN_HISTORY);
                    stmt.setString(1, user.getEmail());
                    stmt.executeUpdate();

                    stmt.close();
					
				} else {
					throw new UserAccessException(ErrorCode.ACTIVATION_ERROR);
				}
				
			} else {
				throw new UserAccessException(ErrorCode.ACTIVATION_ERROR);
			}
			
			
		} catch (SQLException e) {
			
			logger.error("Failed to activate user", e);
			throw new UserAccessException(e, ErrorCode.SQL_ERROR);
			
		} finally {
			try {				
				rs.close();
				stmt.close();
				DataSourceUtils.releaseConnection(con, datasource);
			} catch(Exception e) {
			}
		}
		
		return user;
	}

	@Override
	public void updateResetToken(String userEmail, String resetToken)
			throws UserAccessException {
		
		try {
			Connection con = DataSourceUtils.getConnection(datasource);
			PreparedStatement stmt = con.prepareStatement(UPDATE_USER_TOKEN);
			stmt.setString(1, resetToken);
			stmt.setString(2, userEmail);
			
			stmt.executeUpdate();
			stmt.close();
			DataSourceUtils.releaseConnection(con, datasource);
			
		} catch (SQLException e) {
			logger.error("Failed to update reset token", e);
			throw new UserAccessException(e, ErrorCode.SQL_ERROR);
		}
	}

	@Override
	public void updateUserPassword(String userEmail, String password)
			throws UserAccessException {
		
		try {
			
			Connection con = DataSourceUtils.getConnection(datasource);
			PreparedStatement stmt = con.prepareStatement(UPDATE_USER_PASSWORD);
			stmt.setString(1, DigestUtils.md5Hex(password));
			stmt.setString(2, password);
			stmt.setString(3, userEmail);
			
			stmt.executeUpdate();
			
			stmt.close();
			DataSourceUtils.releaseConnection(con, datasource);
			
		} catch (SQLException e) {
			logger.error("Failed to update user password", e);
			throw new UserAccessException(e, ErrorCode.SQL_ERROR);
		}
	}

	@Override
	public List<User> getUsers() throws UserAccessException {
		
		List<User> users = new ArrayList<User>();

		try {
			
			Connection con = DataSourceUtils.getConnection(datasource);
			PreparedStatement stmt = con.prepareStatement(GET_USERS);
			ResultSet rs = stmt.executeQuery();

			while(rs.next()) {
				
				User user = new User();
				user.setName(rs.getString("name"));
				user.setEmail(rs.getString("email"));
                user.setCountry(rs.getString("country"));
                user.setDomain(rs.getString("domain"));
				user.setActivated(rs.getBoolean("activated"));
				user.setPendingDataProvider(rs.getBoolean("pendingdataprovider"));
				
				PreparedStatement statement = con.prepareStatement(GET_USERS_ROLES);
				statement.setString(1, user.getEmail());
				ResultSet resultSet = statement.executeQuery();
				
				List<String> roles = new ArrayList<String>();
				while(resultSet.next()) {
					
					roles.add(resultSet.getString("role"));
				}
				user.setRoles(roles);
				
				users.add(user);
				
				resultSet.close();
				statement.close();
			}
			
			rs.close();
			stmt.close();
			DataSourceUtils.releaseConnection(con, datasource);
			
		} catch (SQLException e) {
			logger.error("Failed to get the list of users", e);
			throw new UserAccessException(e, ErrorCode.SQL_ERROR);
		}
		return users;
	}

	@Override
	public void activateUsers(List<String> emails) throws UserAccessException {
	
		String statement = "UPDATE espasuser set activated = TRUE where";
		for(String email : emails) {
			statement += " email = '" + email + "' OR";
		}
		if(statement.endsWith("OR"))
			statement = statement.substring(0, statement.length()-2);
		statement += ";";
		
		try {
			
			Connection con = DataSourceUtils.getConnection(datasource);
			PreparedStatement stmt = con.prepareStatement(statement);
			
			stmt.executeUpdate();
			
			stmt.close();
			DataSourceUtils.releaseConnection(con, datasource);
			
		} catch (SQLException e) {
			logger.error("Failed to activate users", e);
			throw new UserAccessException(e, ErrorCode.SQL_ERROR);
		}
	}

	@Override
	public void deactivateUsers(List<String> emails) throws UserAccessException {
		
		String statement = "UPDATE espasuser set activated = FALSE where";
		for(String email : emails) {
			statement += " email = '" + email + "' OR";
		}
		if(statement.endsWith("OR"))
			statement = statement.substring(0, statement.length()-2);
		statement += ";";
		
		try {
			
			Connection con = DataSourceUtils.getConnection(datasource);
			PreparedStatement stmt = con.prepareStatement(statement);
			
			stmt.executeUpdate();
			
			stmt.close();
			DataSourceUtils.releaseConnection(con, datasource);
			
		} catch (SQLException e) {
			logger.error("Failed to deactivate users", e);
			throw new UserAccessException(e, ErrorCode.SQL_ERROR);
		}
	}

	@Override
	public void deleteUsers(List<String> emails) throws UserAccessException {
		
		String statement = "DELETE from espasuser where";
		for(String email : emails) {
			statement += " email = '" + email + "' OR";
		}
		if(statement.endsWith("OR"))
			statement = statement.substring(0, statement.length()-2);
		statement += ";";
		
		try {
			
			Connection con = DataSourceUtils.getConnection(datasource);
			PreparedStatement stmt = con.prepareStatement(statement);
			
			stmt.executeUpdate();
			
			stmt.close();
			DataSourceUtils.releaseConnection(con, datasource);
			
		} catch (SQLException e) {
			logger.error("Failed to delete users", e);
			throw new UserAccessException(e, ErrorCode.SQL_ERROR);
		}
	}

	@Override
	public void addRoleToUser(String email, String role)
			throws UserAccessException {
		
		try {
			
			Connection con = DataSourceUtils.getConnection(datasource);
			PreparedStatement stmt = con.prepareStatement(ADD_ROLE_TO_USER);
			stmt.setString(1, email);
			stmt.setString(2, role);
			stmt.executeUpdate();
			
			stmt.close();
			
			stmt = con.prepareStatement(UPDATE_USER_PENDINGDATAPROVIDER);
			stmt.setBoolean(1, false);
			stmt.setString(2, email);
			stmt.executeUpdate();
			
			stmt.close();
			DataSourceUtils.releaseConnection(con, datasource);
			
		} catch (SQLException e) {
			logger.error("Failed to add role to user", e);
			throw new UserAccessException(e, ErrorCode.SQL_ERROR);
		}
	}

	@Override
	public void removeRoleFromUser(String email, String role)
			throws UserAccessException {
		
		try {
			
			Connection con = DataSourceUtils.getConnection(datasource);
			PreparedStatement stmt = con.prepareStatement(REMOVE_ROLE_FROM_USER);
			stmt.setString(1, email);
			stmt.setString(2, role);
			
			stmt.executeUpdate();
			
			stmt.close();
			DataSourceUtils.releaseConnection(con, datasource);
			
		} catch (SQLException e) {
			logger.error("Failed to remove role from user", e);
			throw new UserAccessException(e, ErrorCode.SQL_ERROR);
		}
	}

	@Override
	public List<User> getPendingDataProviderUsers() throws UserAccessException {
		
		List<User> users = new ArrayList<User>();

		try {
			
			Connection con = DataSourceUtils.getConnection(datasource);
			PreparedStatement stmt = con.prepareStatement(GET_PENDING_DATAPROVIDER_USERS);
			ResultSet rs = stmt.executeQuery();

			while(rs.next()) {
				
				User user = new User();
				user.setName(rs.getString("name"));
				user.setEmail(rs.getString("email"));
				user.setActivated(rs.getBoolean("activated"));
				user.setPendingDataProvider(rs.getBoolean("pendingdataprovider"));
				
				PreparedStatement statement = con.prepareStatement(GET_USERS_ROLES);
				statement.setString(1, user.getEmail());
				ResultSet resultSet = statement.executeQuery();
				
				List<String> roles = new ArrayList<String>();
				while(resultSet.next()) {
					
					roles.add(resultSet.getString("role"));
				}
				user.setRoles(roles);
				
				users.add(user);
				
				resultSet.close();
				statement.close();
			}
			
			rs.close();
			stmt.close();
			DataSourceUtils.releaseConnection(con, datasource);
			
		} catch (SQLException e) {
			logger.error("Failed to get the list of pending data provider users", e);
			throw new UserAccessException(e, ErrorCode.SQL_ERROR);
		}
		return users;
	}

	@Override
	public void denyDataProviderRoleToUser(String email)
			throws UserAccessException {
		
		try {
			
			Connection con = DataSourceUtils.getConnection(datasource);
			PreparedStatement stmt = con.prepareStatement(UPDATE_USER_PENDINGDATAPROVIDER);
			stmt.setBoolean(1, false);
			stmt.setString(2, email);
			
			stmt.executeUpdate();
			
			stmt.close();
			DataSourceUtils.releaseConnection(con, datasource);
			
		} catch (SQLException e) {
			logger.error("Failed to deny data provider role to user", e);
			throw new UserAccessException(e, ErrorCode.SQL_ERROR);
		}
	}

	@Override
	public List<String> getDataProviderUsersEmails() throws UserAccessException {
		
		List<String> administrators = new ArrayList<String>();

		try {
			
			Connection con = DataSourceUtils.getConnection(datasource);
			PreparedStatement stmt = con.prepareStatement(GET_DATAPROVIDER_USERS_EMAILS);
			ResultSet rs = stmt.executeQuery();

			while(rs.next()) {
				
				administrators.add(rs.getString("email"));
			}
			
			rs.close();
			stmt.close();
			DataSourceUtils.releaseConnection(con, datasource);
			
		} catch (SQLException e) {
			logger.error("Failed to get data provider users' emails", e);
			throw new UserAccessException(e, ErrorCode.SQL_ERROR);
		}
		
		return administrators;
	}

    @Override
    public List<User> getDataProviderUsers() {

        List<User> users = new ArrayList<User>();

        try {

            Connection con = DataSourceUtils.getConnection(datasource);
            PreparedStatement stmt = con.prepareStatement(GET_DATAPROVIDER_USERS);

            ResultSet rs = stmt.executeQuery();

            while (rs.next()) {

                User user = new User();
                user.setEmail(rs.getString("email"));
                user.setName(rs.getString("name"));

                users.add(user);
            }

            rs.close();
            stmt.close();
            DataSourceUtils.releaseConnection(con, datasource);

        } catch (SQLException e) {
            logger.error("Error executing query getting data provider administrator users", e);
            e.printStackTrace();
        }

        return users;
    }
}
