package eu.dnetlib.msro.workflows.hadoop.hbase;

import java.io.IOException;
import java.util.Map;

import org.apache.commons.io.IOUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.beans.factory.annotation.Required;

import com.googlecode.sarasvati.Engine;
import com.googlecode.sarasvati.NodeToken;
import com.googlecode.sarasvati.env.Env;

import eu.dnetlib.data.hadoop.rmi.HadoopBlackboardActions;
import eu.dnetlib.data.hadoop.rmi.HadoopService;
import eu.dnetlib.enabling.resultset.rmi.ResultSetException;
import eu.dnetlib.enabling.tools.blackboard.BlackboardJob;
import eu.dnetlib.miscutils.functional.xml.DnetXsltFunctions;
import eu.dnetlib.msro.workflows.nodes.BlackboardJobNode;
import eu.dnetlib.msro.workflows.nodes.ProgressJobNode;
import eu.dnetlib.msro.workflows.nodes.blackboard.BlackboardWorkflowJobListener;
import eu.dnetlib.msro.workflows.resultset.ProcessCountingResultSetFactory;
import eu.dnetlib.msro.workflows.util.ProgressProvider;
import eu.dnetlib.msro.workflows.util.ResultsetProgressProvider;
import eu.dnetlib.msro.workflows.util.WorkflowsConstants;

public class StoreHBaseRecordsJobNode extends BlackboardJobNode implements ProgressJobNode {

	private static final Log log = LogFactory.getLog(StoreHBaseRecordsJobNode.class); // NOPMD by marko on 11/24/08 5:02 PM

	private String inputEprParam;
	private String hbaseTableProperty;
	private String cluster;
	private String xslt;

	private boolean simulation = false;

	private ProgressProvider progressProvider;

	private ProcessCountingResultSetFactory processCountingResultSetFactory;

	@Override
	protected String obtainServiceId(final NodeToken token) {
		return getServiceLocator().getServiceId(HadoopService.class);
	}

	@Override
	protected void prepareJob(final BlackboardJob job, final NodeToken token) throws Exception {
		log.info("Invoking blackboard method");

		job.setAction(HadoopBlackboardActions.IMPORT_EPR_HBASE.toString());
		job.getParameters().put("input_epr", DnetXsltFunctions.encodeBase64(prepareEpr(token)));
		job.getParameters().put("xslt", DnetXsltFunctions.encodeBase64(prepareXslt()));
		job.getParameters().put("table", getPropertyFetcher().getProperty(getHbaseTableProperty()));
		job.getParameters().put("cluster", cluster);
		job.getParameters().put("simulation", String.valueOf(isSimulation()));
	}

	@Override
	protected BlackboardWorkflowJobListener generateBlackboardListener(final Engine engine, final NodeToken token) {
		return new BlackboardWorkflowJobListener(engine, token) {

			@Override
			protected void populateEnv(final Env env, final Map<String, String> responseParams) {
				final String count = responseParams.get("count");
				log.info(String.format("Imported %s objects to HBase table %s, cluster %s", count, getPropertyFetcher().getProperty(getHbaseTableProperty()),
						getCluster()));
				env.setAttribute(WorkflowsConstants.MAIN_LOG_PREFIX + getName() + ":count", count);
			}
		};
	}

	private String prepareEpr(final NodeToken token) throws ResultSetException {
		final String epr = token.getEnv().getAttribute(inputEprParam);
		final ResultsetProgressProvider resultsetProgressProvider = processCountingResultSetFactory.createProgressProvider(token.getProcess(), epr);

		setProgressProvider(resultsetProgressProvider);

		return resultsetProgressProvider.getEpr().toString();
	}

	private String prepareXslt() throws IOException {
		return (xslt == null) || xslt.isEmpty() ? "" : IOUtils.toString(getClass().getResourceAsStream(xslt));
	}

	public String getInputEprParam() {
		return inputEprParam;
	}

	public void setInputEprParam(final String inputEprParam) {
		this.inputEprParam = inputEprParam;
	}

	public String getHbaseTableProperty() {
		return hbaseTableProperty;
	}

	public void setHbaseTableProperty(final String hbaseTableProperty) {
		this.hbaseTableProperty = hbaseTableProperty;
	}

	@Required
	public void setProcessCountingResultSetFactory(final ProcessCountingResultSetFactory processCountingResultSetFactory) {
		this.processCountingResultSetFactory = processCountingResultSetFactory;
	}

	@Override
	public ProgressProvider getProgressProvider() {
		return progressProvider;
	}

	public void setProgressProvider(final ProgressProvider progressProvider) {
		this.progressProvider = progressProvider;
	}

	public ProcessCountingResultSetFactory getProcessCountingResultSetFactory() {
		return processCountingResultSetFactory;
	}

	public String getXslt() {
		return xslt;
	}

	public void setXslt(final String xslt) {
		this.xslt = xslt;
	}

	public String getCluster() {
		return cluster;
	}

	public void setCluster(final String cluster) {
		this.cluster = cluster;
	}

	public boolean isSimulation() {
		return simulation;
	}

	public void setSimulation(final boolean simulation) {
		this.simulation = simulation;
	}

}
