package eu.dnetlib.msro.workflows.dedup;

import javax.annotation.Resource;

import org.apache.commons.lang.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.dom4j.DocumentException;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Required;

import com.googlecode.sarasvati.Arc;
import com.googlecode.sarasvati.NodeToken;

import eu.dnetlib.enabling.is.lookup.rmi.ISLookUpDocumentNotFoundException;
import eu.dnetlib.enabling.is.lookup.rmi.ISLookUpException;
import eu.dnetlib.enabling.is.lookup.rmi.ISLookUpService;
import eu.dnetlib.enabling.locators.UniqueServiceLocator;
import eu.dnetlib.miscutils.datetime.DateUtils;
import eu.dnetlib.msro.workflows.dedup.conf.DedupConfigurationOrchestration;
import eu.dnetlib.msro.workflows.dedup.conf.DedupConfigurationOrchestrationLoader;
import eu.dnetlib.msro.workflows.nodes.SimpleJobNode;

public class PrepareDedupIndexJobNode extends SimpleJobNode {

	/**
	 * logger.
	 */
	private static final Log log = LogFactory.getLog(PrepareDedupIndexJobNode.class); // NOPMD by marko on 11/24/08 5:02 PM

	@Resource
	private UniqueServiceLocator serviceLocator;

	@Autowired
	private DedupConfigurationOrchestrationLoader dedupOrchestrationLoader;

	private String rottenRecordsPathParam;

	private String hbaseTable;

	private String dedupConfig;

	@Override
	protected String execute(final NodeToken token) throws Exception {

		log.info("start preparing job");

		final String fields = getFields(env("format", token), env("layout", token));
		token.getEnv().setAttribute("index.fields", fields);

		if (!StringUtils.isBlank(getRottenRecordsPathParam())) {
			token.getEnv().setAttribute(getRottenRecordsPathParam(), "/tmp" + getFileName(token, "rottenrecords"));
		}

		token.getEnv().setAttribute("index.solr.url", getIndexSolrUrlZk());
		token.getEnv().setAttribute("index.solr.collection", getCollectionName(token));

		token.getEnv().setAttribute("index.shutdown.wait.time", getIndexSolrShutdownWait());
		token.getEnv().setAttribute("index.buffer.flush.threshold", getIndexBufferFlushTreshold());
		token.getEnv().setAttribute("index.solr.sim.mode", isFeedingSimulationMode());

		token.getEnv().setAttribute("index.feed.timestamp", DateUtils.now_ISO8601());

		final DedupConfigurationOrchestration conf = dedupConfigurations(token);

		token.getEnv().setAttribute("actionset", conf.getActionSetId());
		token.getEnv().setAttribute("entityType", conf.getEntity().getName());
		token.getEnv().setAttribute("entityTypeId", conf.getEntity().getCode());

		return Arc.DEFAULT_ARC;
	}

	private DedupConfigurationOrchestration dedupConfigurations(final NodeToken token) throws ISLookUpDocumentNotFoundException, ISLookUpException,
			DocumentException {
		if ((getDedupConfig() == null) || getDedupConfig().trim().isEmpty())
			throw new IllegalStateException("Cannot find dedup configurations in workflow env: '" + getDedupConfig() + "'");

		return dedupOrchestrationLoader.load(getDedupConfig());
	}

	private String getFields(final String format, final String layout) throws ISLookUpException {
		return isLookup(String
				.format("<FIELDS>{for $x in collection('')//RESOURCE_PROFILE[.//RESOURCE_TYPE/@value = 'MDFormatDSResourceType' and .//NAME='%s']//LAYOUT[@name='%s']/FIELDS/FIELD return $x[string(@path)]}</FIELDS>",
						format, layout));
	}

	public String getIndexSolrUrlZk() throws ISLookUpException {
		return isLookup("for $x in /RESOURCE_PROFILE[.//RESOURCE_TYPE/@value='IndexServiceResourceType'] return $x//PROTOCOL[./@name='solr']/@address/string()");
	}

	public String getIndexSolrShutdownWait() throws ISLookUpException {
		return queryForServiceProperty("solr:feedingShutdownTolerance");
	}

	public String getIndexBufferFlushTreshold() throws ISLookUpException {
		return queryForServiceProperty("solr:feedingBufferFlushThreshold");
	}

	public String isFeedingSimulationMode() throws ISLookUpException {
		return queryForServiceProperty("solr:feedingSimulationMode");
	}

	private String queryForServiceProperty(final String key) throws ISLookUpException {
		return isLookup("for $x in /RESOURCE_PROFILE[.//RESOURCE_TYPE/@value='IndexServiceResourceType'] return $x//SERVICE_PROPERTIES/PROPERTY[./@ key='"
				+ key + "']/@value/string()");
	}

	private String isLookup(final String xquery) throws ISLookUpException {
		log.debug("quering for service property: " + xquery);
		final String res = serviceLocator.getService(ISLookUpService.class).getResourceProfileByQuery(xquery);
		if (StringUtils.isBlank(res)) throw new IllegalStateException("unable to find unique service property, xquery: " + xquery);
		return res;
	}

	private String getFileName(final NodeToken token, final String fileNamePrefix) {
		return "/" + fileNamePrefix + "_" + getHbaseTable() + "_" + token.getEnv().getAttribute("format") + ".seq";
	}

	private String getCollectionName(final NodeToken token) {
		return env("format", token) + "-" + env("layout", token) + "-" + env("interpretation", token);
	}

	private String env(final String s, final NodeToken token) {
		return token.getEnv().getAttribute(s);
	}

	public String getHbaseTable() {
		return hbaseTable;
	}

	@Required
	public void setHbaseTable(final String hbaseTable) {
		this.hbaseTable = hbaseTable;
	}

	public String getRottenRecordsPathParam() {
		return rottenRecordsPathParam;
	}

	public void setRottenRecordsPathParam(final String rottenRecordsPathParam) {
		this.rottenRecordsPathParam = rottenRecordsPathParam;
	}

	public String getDedupConfig() {
		return dedupConfig;
	}

	public void setDedupConfig(final String dedupConfig) {
		this.dedupConfig = dedupConfig;
	}

}
