package eu.dnetlib.data.hadoop.utils;

import java.io.ByteArrayOutputStream;
import java.io.DataOutputStream;
import java.io.IOException;
import java.util.Map;

import org.apache.commons.lang.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.hbase.client.Scan;
import org.apache.hadoop.hbase.filter.PrefixFilter;
import org.apache.hadoop.hbase.util.Base64;
import org.dom4j.Document;
import org.dom4j.Node;

public class ScanFactory {

	private static final Log log = LogFactory.getLog(ScanFactory.class); // NOPMD by marko on 11/24/08 5:02 PM

	public static String getScan(final ScanProperties scanProperties) throws IOException {
		Scan scan = new Scan();

		scan.setCaching(scanProperties.getCaching());
		scan.setCacheBlocks(false); // don't set to true for MR jobs

		scan.setFilter(scanProperties.getFilterList());
		for (String family : scanProperties.getFamilies()) {
			scan.addFamily(family.getBytes());
		}

		log.debug("serializing scan");
		return convertScanToString(scan);
	}

	public static ScanProperties parseScanProperties(final Document doc, final Map<String, String> bbParams) {
		log.debug("setting job scanner");

		ScanProperties scanProperties = new ScanProperties(doc.valueOf("//FILTERS/@operator"));

		String caching = doc.valueOf("//SCAN/@caching");
		if (!StringUtils.isBlank(caching)) {
			log.info("overriding default scan caching with: " + caching);
			scanProperties.setCaching(Integer.valueOf(caching));
		}

		for (Object o : doc.selectNodes("//SCAN/FAMILIES/FAMILY")) {
			Node node = (Node) o;
			String value = node.valueOf("./@value");
			if ((value == null) || value.isEmpty()) {
				value = bbParams.get(node.valueOf("./@param"));
			}
			log.debug("scanner family value: " + value);
			scanProperties.getFamilies().add(value);
		}
		for (Object o : doc.selectNodes("//SCAN/FILTERS/FILTER")) {
			Node node = (Node) o;
			String filterType = node.valueOf("./@type");

			String value = node.valueOf("./@value");
			if ((value == null) || value.isEmpty()) {
				value = bbParams.get(node.valueOf("./@param"));
			}

			if (filterType.equals("prefix")) {
				log.debug("scanner prefix filter, value: " + value);
				scanProperties.getFilterList().addFilter(new PrefixFilter(value.getBytes()));
			} // TODO add more filterType cases here
		}
		return scanProperties;
	}

	/**
	 * Writes the given scan into a Base64 encoded string.
	 *
	 * @param scan
	 *            The scan to write out.
	 * @return The scan saved in a Base64 encoded string.
	 * @throws IOException
	 *             When writing the scan fails.
	 */
	static String convertScanToString(final Scan scan) throws IOException {
		ByteArrayOutputStream out = new ByteArrayOutputStream();
		DataOutputStream dos = new DataOutputStream(out);
		scan.write(dos);
		return Base64.encodeBytes(out.toByteArray());
	}

}
